#!/usr/bin/env python3
import sys
from sys import argv
from subprocess import Popen
from subprocess import PIPE
import argparse

argument_parser = argparse.ArgumentParser()

argument_parser.add_argument( '-H',
                     '--host',
                     required=True,
                     type=str,
                     help='The host you wish to run a plugin against.')

argument_parser.add_argument( '-u',
                     '--user',
                     required=True,
                     type=str,
                     help='Username for connecting to the remote system.')


argument_parser.add_argument( '-a',
                     '--args',
                     required=False,
                     type=str,
                     help='Arguments to be sent to the plugin. e.g. -warning 80 -critical 90'
)

arguments = argument_parser.parse_args(argv[1:])
plugin_code = """

<#
.DESCRIPTION
A PowerShell based plugin for Nagios and Nagios-like systems. This plugin checks the CPU utilization on Windows machines. This plugin gives you the average CPU usage across all CPUs and all cores.
Remember, thresholds must be breached before they are thrown.
E.g. numwarning 10 will need the number of files to be 11 or higher to throw a WARNING.
.SYNOPSIS
A PowerShell based plugin to check CPU utilization on Windows machines
.NOTES
This plugin does not have the option to show individual utilization per CPU or per core.
This plugin will return performance data.
.PARAMETER warning
The CPU utilization you will tolerate before throwing a WARNING
.PARAMETER critical
The CPU utilization you will tolerate before throwing a CRITICAL
.EXAMPLE
PS> .\check_cpu.ps1
.EXAMPLE
PS> .\check_cpu.ps1 -warning 80 -critical 90
#>
param(
    [Parameter(Mandatory=$false)][int]$warning = $null,
    [Parameter(Mandatory=$false)][int]$critical = $null
)

$message = "Nothing changed the status output!"
$exitcode = 3

function processCheck {
    param (
        [Parameter(Mandatory=$true)][int]$checkResult,
        [Parameter(Mandatory=$true)][int]$warningThresh,
        [Parameter(Mandatory=$true)][int]$criticalThresh,
        [Parameter(Mandatory=$false)][string]$returnMessage
    )

    [array]$returnArray
    if ((!$criticalThresh) -and (!$warningThresh) ) {
        $returnArray = @(0, "OK: $returnMessage")
    }
    elseif ($checkResult -gt $criticalThresh) {
        $returnArray = @(2, "CRITICAL: $returnMessage")
    }
    elseif ($checkResult -le $criticalThresh -and $checkResult -gt $warningThresh) {
        $returnArray = @(1, "WARNING: $returnMessage")
    }
    else {
        $returnArray = @(0, "OK: $returnMessage")
    }

    return $returnArray
}

$processors = (Get-CimInstance -ClassName Win32_Processor -ComputerName localhost).LoadPercentage
$cpuUsageSum = 0
$processorCount = $processors.Count

foreach ($processor in $processors) {
    $cpuUsageSum += $processor
}

$averageCpuUsage = $cpuUsageSum / $processorCount
$message = "Average CPU utilization is $averageCpuUsage"

$processArray = processCheck -checkResult $averageCpuUsage `
                             -warningThresh $warning `
                             -criticalThresh $critical `
                             -returnMessage "Average CPU usage is $averageCpuUsage% | 'CPU Usage'=$averageCpuUsage%;$warning;$critical"
$exitcode = $processArray[1]
$exitMessage = $processArray[2]

write-host "$exitMessage,$exitcode"

exit $exitcode

"""
echo_process = ""
arguments_length = 0
if arguments.args is not None:
    echo_process = Popen(["echo", "function checkplugin {\n", plugin_code, " }\n", "checkplugin ", arguments.args, "\n"], stdout=PIPE)
    arguments_length = len(arguments.args) + 1
else:
    echo_process = Popen(["echo", "function checkplugin {\n", plugin_code, " }\n", "checkplugin  \n"], stdout=PIPE)

ssh_process = Popen(["ssh", "-T", "-l", arguments.user, arguments.host, "powershell.exe"], stdin=echo_process.stdout, stdout=PIPE)
echo_process.stdout.close()
process_output = [ssh_process.communicate(), ssh_process.returncode]

decoded_stdout = process_output[0][0].decode()

if(process_output[1] == 255):
    print("CRITICAL: Connection to host failed. Check that the nagios user can passwordlessly connect to the host.")
    sys.exit(2)     

output_substring = decoded_stdout[(decoded_stdout.find("checkplugin  ") + 18 + arguments_length):(len(decoded_stdout) - 1)].rstrip()
split_output_substring = output_substring.split(',')

exit_status_code = int(split_output_substring[-1])
exit_message = ','.join(split_output_substring[:-1]) 
print(exit_message)
sys.exit(exit_status_code)