#!/usr/bin/php
<?php

#require_once 'Feed.php';

##########################################################################################################################################
#
# check_microsoft_365 - A /Nagios plugin to check Microsoft 365
#
# Copyright (c) 2021 Nagios Enterprises
# Version 1.0.0 Copyright (c) 2021 Nagios Enterprises, LLC (Laura Gute <lgute@nagios.com>)
#
# Notes:
#   
#   Version 1.1.1 - 2021/11/18
#       Added reportdatauserslist check for the wizard.  Used to determine if the reports users, etc. are obfuscated.
#       Fixed issue with Organization.
#
#   Version 1.1.0 - 2021/03/23
#       Major changes to handle big datasets and test with large datasets, from a file.
#
#   Version 1.0.0 - 2021/01/08
#       Initial release. 
#
#   This plugin will check the general health of Microsoft 365 services.
#   It allows you to set warning and critical thresholds.
#
# License Information:
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
##########################################################################################################################################

# Name of this file, minus extension.
preg_match('/(\w+).\w+$/', $argv[0], $name);
define("CHECK", $name[1]);
define("PROGRAM", $name[0]);
define("VERSION", "1.1.1");
define("STATUS_OK", 0);
define("STATUS_WARNING", 1);
define("STATUS_CRITICAL", 2);
define("STATUS_UNKNOWN", 3);
define("DEBUG", false);

$logger = new Logger(CHECK);

$overrideErrors = false;    # Used when we are running multiple tests, so all are executed.
$requestedTest;             # Keeps a record of the original test requested.  Used when we are running multiple tests.

$http_status_codes = array(100 => "Continue", 101 => "Switching Protocols", 102 => "Processing", 200 => "OK", 201 => "Created", 202 => "Accepted", 203 => "Non-Authoritative Information", 204 => "No Content", 205 => "Reset Content", 206 => "Partial Content", 207 => "Multi-Status", 300 => "Multiple Choices", 301 => "Moved Permanently", 302 => "Found", 303 => "See Other", 304 => "Not Modified", 305 => "Use Proxy", 306 => "(Unused)", 307 => "Temporary Redirect", 308 => "Permanent Redirect", 400 => "Bad Request", 401 => "Unauthorized", 402 => "Payment Required", 403 => "Forbidden", 404 => "Not Found", 405 => "Method Not Allowed", 406 => "Not Acceptable", 407 => "Proxy Authentication Required", 408 => "Request Timeout", 409 => "Conflict", 410 => "Gone", 411 => "Length Required", 412 => "Precondition Failed", 413 => "Request Entity Too Large", 414 => "Request-URI Too Long", 415 => "Unsupported Media Type", 416 => "Requested Range Not Satisfiable", 417 => "Expectation Failed", 418 => "I'm a teapot", 419 => "Authentication Timeout", 420 => "Enhance Your Calm", 422 => "Unprocessable Entity", 423 => "Locked", 424 => "Failed Dependency", 424 => "Method Failure", 425 => "Unordered Collection", 426 => "Upgrade Required", 428 => "Precondition Required", 429 => "Too Many Requests", 431 => "Request Header Fields Too Large", 444 => "No Response", 449 => "Retry With", 450 => "Blocked by Windows Parental Controls", 451 => "Unavailable For Legal Reasons", 494 => "Request Header Too Large", 495 => "Cert Error", 496 => "No Cert", 497 => "HTTP to HTTPS", 499 => "Client Closed Request", 500 => "Internal Server Error", 501 => "Not Implemented", 502 => "Bad Gateway", 503 => "Service Unavailable", 504 => "Gateway Timeout", 505 => "HTTP Version Not Supported", 506 => "Variant Also Negotiates", 507 => "Insufficient Storage", 508 => "Loop Detected", 509 => "Bandwidth Limit Exceeded", 510 => "Not Extended", 511 => "Network Authentication Required", 598 => "Network read timeout error", 599 => "Network connect timeout error");

/*****************************************************************************************************************************************
 * Performance Metrics
 *****************************************************************************************************************************************/

$MODES = array(
    #
    # Outlook, OneDrive, Word, Excel, PowerPoint, OneNote, SharePoint, Teams, Yammer.
    #
    ####################################################################################################################
    # Microsoft Graph REST API v1.0
    #
    # Endpoint:     https://graph.microsoft.com/v1.0/{resource}?[query_parameters]
    #
    # Namespace:    microsoft.graph
    #

    ####################################################################################################################
    #
    # Reports
    #
    #   Example Requests
    #   =================================================================
    #   GET /reports/getEmailAppUsageUserDetail(period='{period_value}')
    #   GET /reports/getEmailAppUsageUserDetail(date={date_value})
    #
    #   Parameter   Description
    #   =========   ==========================================================================================
    #   period      Length of time over which the report is aggregated.
    #               The supported values for {period_value} are: D7, D30, D90, and D180.
    #               Format Dn, where n represents the number of days over which the report is aggregated.
    #
    #   date        A date, within the last 30 days.  Specifies the date to view users who performed any activity.
    #               {date_value} must have a format of YYYY-MM-DD.
    #
    #   API Permissions - Reports.Read.All
    #
    # Teams, Outlook, Office 365, OneDrive, SharePoint      # Later - Skype, Yammer
    #
    # *** WHAT DO WE WANT TO MONITOR???? ***

    ### Outlook Activity Reports ###
    #
    #   getEmailActivityUserDetail - Details of users email activity.
    #       GET /reports/getEmailActivityUserDetail(period='{period_value}')
    #       GET /reports/getEmailActivityUserDetail(date={date_value})
    #
    #   getEmailActivityCounts - Email activity counts (sent, read, received).
    #       GET /reports/getEmailActivityCounts(period='{period_value}')
    #
    #   getEmailActivityUserCounts - Number of unique users performing send, read, and receive, email activities.
    #       GET /reports/getEmailActivityUserCounts(period='{period_value}')
    #

    ### Active ###
    'mailactivitybyuser' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'name' => 'Mailbox Activity',
        'help' => 'Monitor User\'s email activity: | Keep tabs on a user\'s send, receive, and read counts.',
        'stdout' => 'Mailbox Activity @result',
        'label' => 'mail_activity_by_user',
        'check' => '/reports/getEmailActivityUserDetail(period=\'@period\')',   # REST Query
        'period' => array('D7' => '7 days', 'D30' => '30 days', 'D90' => '90 days', 'D180' => '180 days'),
        'fields' => array('User Principal Name' => false, 'Display Name' => false, 'Send Count' => true, 'Receive Count' => true, 'Read Count' => true),
        'type' => 'standard',
        'warn' => '100',    # Default value, for the wizard, user may override.
        'crit' => '1000',   # Default value, for the wizard, user may override.
        'multiple' => true,
        'filterIdx' => 'User Principal Name',
        'filterLabel' => 'User',
        'filter' => 'user',
        'icon' => 'user.png',
        'modifier' => 1,
        'unit' => '',
        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'mailactivityuserdetail' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Mailbox Activity User Detail',
        'stdout' => 'Mailbox Activity User Detail is @result',
        'label' => 'mail_activity_user_detail',
        'check' => '/reports/getEmailActivityUserDetail(period=\'@period\')',   # REST Query
        'period' => array('D7' => '7 days', 'D30' => '30 days', 'D90' => '90 days', 'D180' => '180 days'),
        #'check' => '/reports/getEmailActivityUserDetail(date=2020-08-17)',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'mailactivityuserdetaildelta' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Mailbox Activity User Detail',
        'stdout' => 'Mailbox Activity User Detail is @result',
        'label' => 'mail_activity_user_detail',
        'check' => '/reports/getEmailActivityUserDetail(period=\'D7\')',   # REST Query
        #'check' => '/reports/getEmailActivityUserDetail(date=2020-06-10)',   # REST Query
        #'check' => '/reports/getEmailActivityUserDetail(date=2020-06-10)',   # REST Query
        'type' => 'delta',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'mailactivitycounts' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Mailbox Activity Counts',
        'stdout' => 'Mailbox Activity Counts is @result',
        'label' => 'mail_activity_counts',
        'check' => '/reports/getEmailActivityCounts(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'mailactivityusercounts' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Mailbox Activity User Counts',
        'stdout' => 'Mailbox Activity User Counts is @result',
        'label' => 'mail_activity_user_counts',
        'check' => '/reports/getEmailActivityUserCounts(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Outlook App Usage Reports ###
    #    
    #   Parameter   Description
    #   =========   ==========================================================================================
    #   period      Length of time over which the report is aggregated.
    #               The supported values for {period_value} are: D7, D30, D90, and D180.
    #               Format Dn, where n represents the number of days over which the report is aggregated.
    #
    #   getEmailAppUsageUserDetail - Details about which activities users performed on the various email apps.
    #       GET /reports/getEmailAppUsageUserDetail(period='{period_value}')
    #       GET /reports/getEmailAppUsageUserDetail(date='{date_value}')
    #
    #   getEmailAppUsageAppsUserCounts - The count of unique users per email app.
    #       GET /reports/getEmailAppUsageAppsUserCounts(period='{period_value}')
    #
    #   getEmailAppUsageUserCounts - Count of unique users that connected to Exchange Online using any email app.
    #       GET /reports/getEmailAppUsageUserCounts(period='{period_value}')
    #
    #   getEmailAppUsageVersionUserCounts - Count of unique users by Outlook desktop version.
    #       GET /reports/getEmailAppUsageVersionsUserCounts(period='{period_value}')
    #
    'emailappusageuserdetail' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Email App Usage User Detail',
        'stdout' => 'Email App Usage User Detail is @result',
        'label' => 'email_app_usage_user_detail',
        'check' => '/reports/getEmailAppUsageUserDetail(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'emailappusageappusercounts' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Email App Usage App User Counts',
        'stdout' => 'Email App Usage App User Counts is @result',
        'label' => 'email_app_usage_user_counts',
        'check' => '/reports/getEmailAppUsageAppsUserCounts(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'emailappusageusercounts' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Email App Usage User Counts',
        'stdout' => 'Email App Usage User Counts is @result',
        'label' => 'email_app_usage_user_counts',
        'check' => '/reports/getEmailAppUsageUserCounts(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'emailappusageversionusercounts' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Email App Usage Version User Counts',
        'stdout' => 'Email App Usage Version User Counts is @result',
        'label' => 'email_app_usage_version_user_counts',
        'check' => '/reports/getEmailAppUsageVersionsUserCounts(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Outlook Mailbox Usage Reports ###
    #
    #   Parameter   Description
    #   =========   ==========================================================================================
    #   period      Length of time over which the report is aggregated.
    #               The supported values for {period_value} are: D7, D30, D90, and D180.
    #               Format Dn, where n represents the number of days over which the report is aggregated.
    #
    #  getMailboxUsageDetail - Get details about mailbox usage.
    #      GET /reports/getMailboxUsageDetail(period='{period_value}')
    #
    #  getMailboxUsageMailboxCounts - Get the total number of user mailboxes in your organization and
    #                                 how many are active each day of the reporting period. A mailbox
    #                                 is considered active if the user sent or read any email.
    #      GET /reports/getMailboxUsageMailboxCounts(period='{period_value}')
    #           
    #  getMailboxUsageQuotaStatusMailboxCounts - Get the count of user mailboxes in each quota category.
    #      GET /reports/getMailboxUsageQuotaStatusMailboxCounts(period='{period_value}')
    #           
    #  getMailboxUsageStorage - Get the count of user mailboxes in each quota category.
    #      GET /reports/getMailboxUsageStorage(period='{period_value}')
    #

    ### Active ###
    'mailusagebyuser' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'name' => 'Mailbox Usage',
        'help' => 'Details of user\'s mailbox usage: | Item count, Storage bytes and Deleted bytes.',
        'stdout' => 'Mailbox Usage @result',
        'label' => 'mail_usage_by_user',
        'check' => '/reports/getMailboxUsageDetail(period=\'@period\')',   # REST Query
        'period' => array('D7' => '7 days', 'D30' => '30 days', 'D90' => '90 days', 'D180' => '180 days'),
        'fields' => array('User Principal Name' => false, 'Display Name' => false, 'Item Count' => true, 'Storage Used (Byte)' => true, 'Deleted Item Count' => true, 'Deleted Item Size (Byte)' => true),
        'type' => 'standard',
        'warn' => '2000',   # Default value, for the wizard, user may override.
        'crit' => '10000', # Default value, for the wizard, user may override.
        'multiple' => true,
        'filterIdx' => 'User Principal Name',
        'filterLabel' => 'User',
        'filter' => 'user',
        'icon' => 'user.png',
        'modifier' => 1,
        'unit' => '',
        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'mailusagedetail' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Mailbox Usage Detail',
        'stdout' => 'Mailbox Usage Detail is @result',
        'label' => 'mail_usage_detail',
        'check' => '/reports/getMailboxUsageDetail(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Active ###
    'mailboxusage' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'name' => 'Mailbox Counts',
        'help' => 'Organization\'s users mailboxes: | Total number of mailboxes how many were active.',
        'stdout' => 'Active User Mailboxes @result',
        'label' => 'mailbox_usage',
        'check' => '/reports/getMailboxUsageMailboxCounts(period=\'@period\')',   # REST Query
        'period' => array('D7' => '7 days', 'D30' => '30 days', 'D90' => '90 days', 'D180' => '180 days'),
        'fields' => array('Total' => true, 'Active' => true),
        'type' => 'standard',
        'warn' => '',    # Default value, for the wizard, user may override.
        'crit' => '',    # Default value, for the wizard, user may override.
        'multiple' => false,
        'modifier' => 1,
        'unit' => '',
        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'mailusagecounts' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Mailbox Usage Mailbox Counts',
        'stdout' => 'Mailbox Usage Mailbox Counts is @result',
        'label' => 'mail_usage_counts',
        'check' => '/reports/getMailboxUsageMailboxCounts(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'mailusagequota' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Mailbox Usage Quota Status',
        'stdout' => 'Mailbox Usage Quota Status is @result',
        'label' => 'mail_usage_quota',
        'check' => '/reports/getMailboxUsageQuotaStatusMailboxCounts(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'mailusagestorage' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Mailbox Usage Storage',
        'stdout' => 'Mailbox Usage Storage is @result',
        'label' => 'mail_usage_storage',
        'check' => '/reports/getMailboxUsageStorage(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ####################################################################################################################
    #
    # Office 365 Reports
    #
    #   API Permissions - Reports.Read.All
    #
    ### Office 365 Activations Reports ###
    #
    #   The Microsoft 365 activation reports can give you a view of which users have activated their
    #   Microsoft 365 subscriptions on at least one device. These reports provide a breakdown of the
    #   Microsoft 365 ProPlus, Project, and Visio Pro for Office 365 subscription activations, as well
    #   the breakdown of activations across desktop and devices. These reports could help you identify
    #   users who might need additional support to activate their Office subscription.
    #
    #   Parameter   Description
    #   =========   ==========================================================================================
    #   period      Length of time over which the report is aggregated.
    #               The supported values for {period_value} are: D7, D30, D90, and D180.
    #               Format Dn, where n represents the number of days over which the report is aggregated.
    #
    #   getOffice365ActivationsUserDetail - Get user detail.
    #       GET /reports/getOffice365ActivationsUserDetail
    #
    #   getOffice365ActivationCounts - Count of Office 365 activations on desktops and devices.
    #       GET /reports/getOffice365ActivationCounts
    #
    #   getOffice365ActivationsUserCounts - Count of users who enabled and who have activated Office subscriptions on desktop or devices.
    #       GET /reports/getOffice365ActivationsUserCounts
    #

    ### Active ###
    ### No record means no activations??
    ### What do we actually want to show, for this????
    'o365activationsbyuser' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'name' => 'Office 365 User Activations',
        'help' => 'Office 365 Products and Subscriptions: | User\'s activations on desktops and devices',
        'stdout' => 'Office 365 Activations @result',
        'label' => 'o365_activations_by_user',
        'check' => '/reports/getOffice365ActivationsUserDetail',   # REST Query
#        'check' => '/reports/getOffice365ActivationsUserDetail?$format=application/json',   # REST Query
# Product Type should go somewhere...
        'fields' => array('User Principal Name' => false, 'Display Name' => false, 'Product Type' => false, 'Last Activated Date' => false, 'Windows' => true, 'Mac' => true, 'Windows 10 Mobile' => true, 'iOS' => true, 'Android' => true, 'Activated On Shared Computer' => true),   # Maybe a field type 'ALL'?
        'type' => 'standard',
        'warn' => '1:2',     # Default value, for the wizard, user may override. outside of the range {1 .. 2}, i.e., 0, 3, ...
        'crit' => '3',    # Default value, for the wizard, user may override.
        'multiple' => true,
        'filterIdx' => 'User Principal Name',
        'filterLabel' => 'User',
        'filter' => 'user',
        'icon' => 'user.png',
        'modifier' => 1,
        'unit' => '',
        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'o365activationsuserdetail' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Office 365 Activations User Detail',
        'stdout' => 'Office 365 Activations User Detail is @result',
        'label' => 'o365_activations_user_detail',
        'check' => '/reports/getOffice365ActivationsUserDetail',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Active ###
    'o365activationsbyproduct' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'name' => 'Office 365 Activations',
        'help' => 'Office 365 Products and Subscriptions: | Organization\'s total activations on desktops and devices.',
        'stdout' => 'Office 365 Activations @result',
        'label' => 'o365_activations',
        'check' => '/reports/getOffice365ActivationCounts',   # REST Query
        'fields' => array('Product Type' => false, 'Windows' => true, 'Mac' => true, 'Android' => true, 'iOS' => true, 'Windows 10 Mobile' => true),   # Maybe a field type 'ALL'?
        'type' => 'standard',
        'warn' => '40:50',  # Default value, for the wizard, user may override. < 40 && > 50 - i.e., 0
        'crit' => '1:100',  # Default value, for the wizard, user may override. outside of the range of {1 .. 100}
        'multiple' => true,
        'filterIdx' => 'Product Type',
        'filterLabel' => 'Product Type',
        'filter' => 'product',
        'icon' => 'cog.png',
        'modifier' => 1,
        'unit' => '',
        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'o365activationcounts' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Office 365 Activation Counts',
        'stdout' => 'Office 365 Activation Counts is @result',
        'label' => 'o365_activation_counts',
        'check' => '/reports/getOffice365ActivationCounts',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'o365activationusers' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Office 365 Activations User Counts',
        'stdout' => 'Office 365 Activations User Counts is @result',
        'label' => 'o365_activation_users',
        'check' => '/reports/getOffice365ActivationsUserCounts',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Office 365 Active Users Reports ###
    #    
    #   You can use the Microsoft 365 active users reports to find out how many product licenses
    #   are being used by individuals in your organization, and drill down for information about
    #   which users are using what products. These reports can help administrators identify
    #   underutilized products or users that might need additional training or information.
    #
    #   Parameter   Description
    #   =========   ==========================================================================================
    #   period      Length of time over which the report is aggregated.
    #               The supported values for {period_value} are: D7, D30, D90, and D180.
    #               Format Dn, where n represents the number of days over which the report is aggregated.
    #
    #   getOffice365ActiveUserDetail - Details about Office 365 active users.
    #       GET /reports/getOffice365ActiveUserDetail(period='{period_value}')
    #
    #   getOffice365ActivativeUserCounts - Count of daily active users, by product.
    #       GET /reports/getOffice365ActiveUserCounts(period='{period_value}')
    #
    #   getOffice365ServicesUserCounts - Count of users by activity type and service.
    #       GET /reports/getOffice365ServicesUserCounts(period='{period_value}')
    #
    'o365activeuserdetail' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Office 365 Active User Detail',
        'stdout' => 'Office 365 Activite User Detail is @result',
        'label' => 'o365_active_user_detail',
        'check' => '/reports/getOffice365ActiveUserDetail(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Active ###
    'o365productusage' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'name' => 'Office 365 User Product Usage',
        'help' => 'Daily user activity, by product',
        'stdout' => 'Office 365 Active Users @result',
        'label' => 'o365_active_users',
        'check' => '/reports/getOffice365ActiveUserCounts(period=\'D7\')',   # REST Query
        'period' => array('D7' => '7 days', 'D30' => '30 days', 'D90' => '90 days', 'D180' => '180 days'),
        'fields' => array('Office 365' => true, 'Exchange' => true, 'OneDrive' => true, 'SharePoint' => true, 'Skype For Business' => true, 'Yammer' => true, 'Teams' => true, 'Report Date' => false),   # Maybe a field type 'ALL'?
        'type' => 'standard',
        'warn' => '',    # Default value, for the wizard, user may override.
        'crit' => '',    # Default value, for the wizard, user may override.
        'multiple' => false,
        'modifier' => 1,
        'unit' => '',
        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'o365activeuser' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Office 365 Activity User Counts',
        'stdout' => 'Office 365 Activity User Counts is @result',
        'label' => 'o365_active_user_counts',
        'check' => '/reports/getOffice365ActiveUserCounts(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Active ###
    'o365serviceusage' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'name' => 'Office 365 User Service Usage',
        'help' => 'User Service Activity/Inactivity',
        'stdout' => 'Office 365 Services Users @result',
        'label' => 'o365_service_users',
        'check' => '/reports/getOffice365ServicesUserCounts(period=\'@period\')',   # REST Query
        'period' => array('D7' => '7 days', 'D30' => '30 days', 'D90' => '90 days', 'D180' => '180 days'),
        'fields' => array('Report Refresh Date' => false, 'Exchange Active' => true, 'Exchange Inactive' => true, 'OneDrive Active' => true, 'OneDrive Inactive' => true, 'SharePoint Active' => true, 'SharePoint Inactive' => true, 'Skype For Business Active' => true, 'Skype For Business Inactive' => true, 'Yammer Active' => true, 'Yammer Inactive' => true, 'Teams Active' => true, 'Teams Inactive' => true, 'Office 365 Active' => true, 'Office 365 Inactive' => true, 'Report Period' => false),
        'type' => 'standard',
        'warn' => '1:50',   # Default value, for the wizard, user may override. 1: => outside of the range of {1 .. ∞}
        'crit' => '100',    # Default value, for the wizard, user may override.
        'multiple' => false,
        'modifier' => 1,
        'unit' => '',
        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'o365servicesuser' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Office 365 Services User Counts',
        'stdout' => 'Office 365 Services User Counts is @result',
        'label' => 'o365_services_user',
        'check' => '/reports/getOffice365ServicesUserCounts(period=\'D7\')',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Office 365 Groups Activity Reports ###
    #
    #   You can use the Groups activity reports to gain insights into the activity of Microsoft 365 groups
    #   in your organization and see how many Microsoft 365 groups are being created and used.
    #
    #   Parameter   Description
    #   =========   ==========================================================================================
    #   period      Length of time over which the report is aggregated.
    #               The supported values for {period_value} are: D7, D30, D90, and D180.
    #               Format Dn, where n represents the number of days over which the report is aggregated.
    #
    #   date        A date, within the last 30 days.  Specifies the date to view users who performed any activity.
    #               {date_value} must have a format of YYYY-MM-DD.
    #
    #   getOffice365GroupsActivityDetail - Details about Office 365 Groups activity by group.
    #       GET /reports/getOffice365GroupsActivityDetail(period='{period_value}')
    #       GET /reports/getOffice365GroupsActivityDetail(date='{date_value}')
    #
    #   getOffice365GroupsActivityCounts - Number of group activities across group workloads.
    #       GET /reports/getOffice365GroupsActivityCounts(period='{period_value}')
    #
    #   getOffice365GroupsActivityGroupCounts - Daily total number of groups and, email, Yammer and SharePoint activity.
    #       GET /reports/getOffice365GroupsActivityGroupCounts(period='{period_value}')
    #
    #   getOffice365AGroupsActivityGroupStorage - Total storage used across all group mailboxes and group sites.
    #       GET /reports/getOffice365GroupsActivityStorage(period='{period_value}')
    #
    #   getOffice365GroupsActivityGroupFileCounts - Total number of files and how many were active across all group sites.
    #       GET /reports/getOffice365GroupsActivityFileCounts(period='{period_value}')
    #
    #

    ### Active ###
    'o365groupsactivitybygroup' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'name' => 'Office 365 Group Activity',
        'help' => 'Group\'s Office 365 Service Activity',
        'stdout' => 'Office 365 Activity @result',
        'label' => 'o365_groups_activity_by_user',
        'check' => '/reports/getOffice365GroupsActivityDetail(period=\'@period\')', # REST Query
        #'check' => '/reports/getOffice365GroupsActivityDetail(date=\'@date\')',     # Date in the past 30 days YYYY-MM-DD
        'period' => array('D7' => '7 days', 'D30' => '30 days', 'D90' => '90 days', 'D180' => '180 days'),
        'fields' => array('Owner Principal Name' => false, 'Group Display Name' => false, 'SharePoint Active File Count' => true, 'Exchange Mailbox Total Item Count' => true, 'Exchange Mailbox Storage Used (Byte)' => true, 'SharePoint Total File Count' => true, 'SharePoint Site Storage Used (Byte)' => true),   # Maybe a field type 'ALL'?
        'type' => 'group',
        'warn' => '',    # Default value, for the wizard, user may override.  May be 0 or ''.
        'crit' => '',    # Default value, for the wizard, user may override.
        'multiple' => true,
        'filterIdx' => 'Group Display Name',
        'filterLabel' => 'Group',
        'filter' => 'group',
        'icon' => 'group.png',
        'modifier' => 1,
        'unit' => '',
        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'o365groupsdetail' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Office 365 Groups Activity Detail',
        'stdout' => 'Office 365 Groups Activity Detail is @result',
        'label' => 'o365_group_detail',
        'check' => '/reports/getOffice365GroupsActivityDetail(period=\'D7\')',   # REST Query
        'type' => 'group',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'o365groupsactivity' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Office 365 Group Activity',
        'stdout' => 'Office 365 Group Activity is @result',
        'label' => 'o365_groups_activity_counts',
        'check' => '/reports/getOffice365GroupsActivityCounts(period=\'D7\')',   # REST Query
        'type' => 'group',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'o365groupsactivitygroup' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Office 365 Groups Activity Group Counts',
        'stdout' => 'Office 365 Groups Activity Group Counts is @result',
        'label' => 'o365_groups_activity_group_counts',
        'check' => '/reports/getOffice365GroupsActivityGroupCounts(period=\'D7\')',   # REST Query
        'type' => 'group',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'o365groupsactivitystorage' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Office 365 Group Activity Storage',
        'stdout' => 'Office 365 Group Activity Storage is @result',
        'label' => 'o365_groups_activity_storage',
        'check' => '/reports/getOffice365GroupsActivityStorage(period=\'D7\')',   # REST Query
        'type' => 'group',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Active ###
    'o365groupsfileactivity' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'name' => 'Office 365 Group Files',
        'help' => 'Office 365 Group Total and Active Files',
        'stdout' => 'Office 365 Group File Activity @result',
        'label' => 'o365_groups_file_activity',
        'check' => '/reports/getOffice365GroupsActivityFileCounts(period=\'@period\')',   # REST Query
        'period' => array('D7' => '7 days', 'D30' => '30 days', 'D90' => '90 days', 'D180' => '180 days'),
        'fields' => array('Total' => true, 'Active' => true, 'Report Date' => false),   # Maybe a field type 'ALL'?
        'type' => 'group',
        'warn' => '',    # Default value, for the wizard, user may override.
        'crit' => '',    # Default value, for the wizard, user may override.
        'multiple' => false,
        'modifier' => 1,
        'unit' => '',
        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'o365groupsactivityfile' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'help' => 'Office 365 Group Activity File Counts',
        'stdout' => 'Office 365 Group Activity File Counts is @result',
        'label' => 'o365_groups_activity_file_counts',
        'check' => '/reports/getOffice365GroupsActivityFileCounts(period=\'D7\')',   # REST Query
        'type' => 'group',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ####################################################################################################################
    #
    #   getM365AppUserDetail - provides the details about which apps and platforms users have used.
    #       GET https://graph.microsoft.com/beta/reports/getM365AppUserDetail(period='D7')/content?$format=application/json
    #
    #   NOTE: For testing report file paging.
    #

    'm365appuserdetail' => array(
        'version' => 'beta',
        'api' => EndpointConstants::GRAPH_REPORTS_API,
#        'api' => EndpointConstants::GRAPH_API,  # Works with application/json and beta
        'name' => 'M365 User App and Platform Usage',
        'help' => 'Microsoft 365 Users App and Platform Usage',
        'stdout' => 'Microsoft 365 Usage @result',
        'label' => 'm365_app_user_detail',
        'check' => '/reports/getM365AppUserDetail(period=\'@period\')/content?$top=10&$format=text/csv', # Ignores top=10
#        'check' => '/reports/getM365AppUserDetail(period=\'@period\')/content?$format=text/csv', # REST Query
#        'check' => '/reports/getM365AppUserDetail(period=\'@period\')/content?$format=application/json&$top=10', # Works with beta & GRAPH_API (almost)
        'period' => array('D7' => '7 days', 'D30' => '30 days', 'D90' => '90 days', 'D180' => '180 days'),
        'fields' => array('Report Refresh Date' => false, 'User Principal Name' => true, 'Last Activation Date' => true, 'Last Activity Date' => true, 'Report Period' => true, 'Windows' => true, 'Mac' => true, 'Mobile' => true, 'Web' => true, 'Outlook' => true, 'Word' => true, 'Excel' => true, 'PowerPoint' => true, 'OneNote' => true, 'Teams' => true, 'Outlook (Windows)' => true, 'Word (Windows)' => true, 'Excel (Windows)' => true, 'PowerPoint (Windows)' => true, 'OneNote (Windows)' => true, 'Teams (Windows)' => true, 'Outlook (Mac)' => true, 'Word (Mac)' => true, 'Excel (Mac)' => true, 'PowerPoint (Mac)' => true, 'OneNote (Mac)' => true, 'Teams (Mac)' => true, 'Outlook (Mobile)' => true, 'Word (Mobile)' => true, 'Excel (Mobile)' => true, 'PowerPoint (Mobile)' => true, 'OneNote (Mobile)' => true, 'Teams (Mobile)' => true, 'Outlook (Web)' => true, 'Word (Web)' => true, 'Excel (Web)' => true, 'PowerPoint (Web)' => true, 'OneNote (Web)' => true, 'Teams (Web)' => true,),   # Maybe a field type 'ALL'?
        'type' => 'standard',
        'warn' => '',    # Default value, for the wizard, user may override.  May be 0 or ''.
        'crit' => '',    # Default value, for the wizard, user may override.
        'multiple' => true,
        'filterIdx' => 'User Principal Name',
        'filterLabel' => 'User',
        'filter' => 'user',
        'icon' => 'user.png',
        'modifier' => 1,
        'unit' => '',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ####################################################################################################################
    #
    # Security
    #
    #   Example Requests
    #   =================================================================
    #   GET /security/alerts
    #   GET /security/secureScores
    #
    #   API Permissions - SecurityEvents.Read.All, SecurityEvents.Read.All
    #
    # Teams, Outlook, Office 365, OneDrive, SharePoint      # Later - Skype, Yammer
    #
    # WHAT DO WE WANT TO MONITOR????
    #
    ### Security Errors ###
    #
    #

    ### Security Alert Queries ###
    #
    #   alerts - Retrieve a list of alert objects.
    #       GET /security/alerts
    #       GET /security/alerts?$top=1
    #       GET /security/alerts?$filter={property} eq '{property-value}'
    #       GET /security/alerts?$filter={property} eq '{property-value}'&$top=5
    #       GET /security/alerts?$filter={property} eq '{property-value}' and {property} eq '{property-value}'
    #
    #   alert - Retrieve the properties and relationships of an alert object.
    #       GET /security/alerts/{alert_id}
    #
    'alerts' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Security Alerts',
        'stdout' => 'Security Alerts is @result',
        'label' => 'security_alerts',
        'check' => '/security/alerts?$top=5',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'alert' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Security Alert',
        'stdout' => 'Security Alert is @result',
        'label' => 'security_alert',
        'check' => '/security/alerts/@id',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Security Score Queries ###
    #
    #   Microsoft Secure Score is a security analytics solution that gives you visibility into your security portfolio
    #   and how to improve it. With a single score, you can better understand what you have done to reduce your risk
    #   in Microsoft solutions. You can also compare your score with other organizations and see how your score has been
    #   trending over time. The Microsoft Graph Security secureScore and secureScoreControlProfile entities help you
    #   balance your organization's security and productivity needs while enabling the appropriate mix of security
    #   features. You can also project what your score would be after you adopt security features.
    #
    #   secureScore resource type
    #
    #   Represents a tenant's secure score per day of scoring data, at the tenant and control level. By default, 90 days
    #   of data is held. This data is sorted by createdDateTime, from latest to earliest. This will allow you to page
    #   responses by using $top=n, where n = the number of days of data that you want to retrieve.
    #
    #   Microsoft Secure Score is a measurement of an organization's security posture, with a higher number indicating
    #   more improvement actions taken.
    #
    #   secureScores - Retrieve a list of secureScore objects.
    #       GET /security/secureScores
    #       GET /security/secureScores?$top=1
    #       GET /security/secureScores?$top=1&$skip=7
    #       GET /security/secureScores?$filter={property} eq '{property-value}'
    #
    #   secureScore - Retrieve the properties and relationships of a secureScore object.
    #               - {id} = azureTenantId_yyyy-mm-dd, where dd is 24-48 h:w
    #               ours prior to current date.
    #       GET /security/secureScores/{id}
    #
    'securescorebyid' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Secure Score',
        'stdout' => 'Secure Score is @result',
        'label' => 'secure_score',
        'check' => '/security/secureScores/@id',   # REST Query
        #'check' => '/security/secureScores/aeb3589c-ecf7-477d-b308-d9b16ad9dd45_2020-06-16',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Active ###
    'securescore' => array(
        'api' => EndpointConstants::GRAPH_API,
        'name' => 'Secure Score',
        'help' => 'Current measurement of organization\'s/tenant\'s daily security posture: Improvement actions raise scores.',
        'stdout' => 'Secure Scores @result',
        'label' => 'secure_scores',
        'check' => '/security/secureScores?$top=1',   # REST Query - $top=1 limits the page size to 1 record.
        'limit' => 1,       # Only retrieve one record (don't follow nextLink).
        'fields' => array('createdDateTime' => false, 'currentScore' => true),
        'type' => 'standard',
        'warn' => '30:',    # Default value, for the wizard, user may override.
        'crit' => '20:',    # Default value, for the wizard, user may override.
        'multiple' => false,
        'modifier' => 1,
        'unit' => '',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
#        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    'securescores' => array(
        'api' => EndpointConstants::GRAPH_API,
        'name' => 'Secure Scores',
        'help' => 'Tenant\'s daily Secure Scores',
        'stdout' => 'Secure Scores is @result',
        'label' => 'secure_scores',
        'check' => '/security/secureScores',   # REST Query
        'fields' => array('createdDateTime', 'currentScore'),
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Security Scores Control Profile Queries ###
    #
    #   secureScoreControlProfile - 
    #       GET /security/secureScoreControlProfiles/{id}
    #
    #   secureScoreControlProfiles - 
    #       GET /security/secureScoreControlProfiles
    #       GET /security/secureScoreControlProfiles?$top=1
    #       GET /security/secureScoreControlProfiles?$filter={property} eq '{property-value}'
    #
    'securescorecontrolprofile' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Secure Score Control Profile',
        'stdout' => 'Secure Score Control Profile is @result',
        'label' => 'secure_score_control_profile',
        'check' => '/security/secureScoreControlProfiles/@id',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'securescorecontrolprofiles' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Secure Score Control Profile',
        'stdout' => 'Secure Score Control Profiles is @result',
        'label' => 'secure_score_control_profiles',
        'check' => '/security/secureScoreControlProfiles',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Users
    # https://docs.microsoft.com/en-us/graph/api/resources/users?view=graph-rest-1.0
    #
    ### Users/User ###
    # https://docs.microsoft.com/en-us/graph/api/resources/user?view=graph-rest-1.0
    #
    #   users - Users in the organization
    #       GET /users
    #       GET /users/{id}
    #       GET /users/{id}/photo/$value
    #       GET /users/{id}/manager
    #       GET /users/{id}/messages
    #       GET /users/{id}/events
    #       GET /users/{id}/drive
    #       GET /users/{id}/memberOf
    #
    #   usersDelta - Get newly created, updated, or deleted users without having to perform a full read of the entire user collection.
    #       GET /users/delta
    #
    #   usersDeleted - Retrieve a list of recently deleted users.
    #       GET /directory/deleteditems/microsoft.graph.user
    #
    #   API Permissions - User.Read.All (Directory.Read.All) - least to most privileged
    #
    'usermailactivityalldata' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Users',
        'stdout' => 'User\'s mail activity is @result',
        'label' => 'secure_score_control_profile',
        #'check' => "/users/lxxxx@nagios.com/messages?\$count=true&\$filter=receivedDateTime+ge+2020-08-01&",   # REST Query
        'check' => "/users/lxxxx@nagiosdev.onmicrosoft.com/messages?\$count=true&\$filter=receivedDateTime+ge+2020-08-06+and+from/emailAddress/address+ne+'lxxxx@nagiosdev.onmicrosoft.com'",   # REST Query
        #'check' => "/users/lxxxx@nagios.com/messages?\$count=true&\$filter=isRead+eq+true+and+receivedDateTime+ge+2020-01-01+and+from/emailAddress/address+eq+'lxxxx@nagios.com'&\$select=id",
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    # Requested: Sent (What date/ date range???). Only subject,receivedDateTime,isRead
    'usermailsent' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Users',
        'stdout' => 'User\'s mail activity is @result',
        'label' => 'secure_score_control_profile',
        'check' => "/users/lxxxx@nagios.com/messages?\$count=true&\$filter=receivedDateTime+ge+2020-08-06+and+from/emailAddress/address+eq+'lxxxx@nagios.com'&\$select=subject,receivedDateTime,isRead",   # REST Query
        #'check' => "/users/lxxxx@nagiosdev.onmicrosoft.com/messages?\$count=true&\$filter=receivedDateTime+ge+2020-08-06+and+from/emailAddress/address+eq+'lxxxx@nagiosdev.onmicrosoft.com'&\$select=subject,receivedDateTime,isRead",   # REST Query
        #'check' => "/users/lxxxx@nagiosdev.onmicrosoft.com/messages?\$count=true&\$filter=receivedDateTime+ge+2020-02-06+and+from/emailAddress/address+eq+'lxxxx@nagiosdev.onmicrosoft.com'&\$select=id",   # REST Query: specify "id" for minimum data returned
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    # Requested: Received (What date/ date range???). Only subject,receivedDateTime,isRead
    'usermailreceived' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Users',
        'stdout' => 'User\'s mail activity is @result',
        'label' => 'secure_score_control_profile',
        'check' => "/users/lxxxx@nagios.com/messages?\$count=true&\$filter=receivedDateTime+ge+2020-02-06+and+from/emailAddress/address+ne+'lxxxx@nagios.com'&\$select=subject,receivedDateTime,isRead",   # REST Query
        #'check' => "/users/lxxxx@nagiosdev.onmicrosoft.com/messages?\$count=true&\$filter=receivedDateTime+ge+2020-02-06+and+from/emailAddress/address+ne+'lxxxx@nagiosdev.onmicrosoft.com'&\$select=receivedDateTime,isRead&\$top=500",   # REST Query
        # Report Refresh Date: 2020-08-12 (D7) Last Activity: 2020-08-06 - 2020-08-05
        #'check' => "/users/lxxxx@nagiosdev.onmicrosoft.com/messages?\$count=true&\$filter=receivedDateTime+ge+2020-08-05+and+receivedDateTime+lt+2020-08-12&\$select=receivedDateTime,isRead&\$top=500",   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    # Requested: Received unread (What date/ date range???). Only subject,receivedDateTime,isRead
    'usermailreceivedunread' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Users',
        'stdout' => 'User\'s mail activity is @result',
        'label' => 'secure_score_control_profile',
        'check' => "/users/lxxxx@nagios.com/messages?\$count=true&\$filter=isRead+eq+false+and+receivedDateTime+ge+2020-08-06+and+from/emailAddress/address+ne+'lxxxx@nagios.com'&\$select=subject,receivedDateTime,isRead",   # REST Query
        #'check' => "/users/lxxxx@nagiosdev.onmicrosoft.com/messages?\$count=true&\$filter=isRead+eq+false+and+receivedDateTime+ge+2020-08-06+and+from/emailAddress/address+ne+'lxxxx@nagiosdev.onmicrosoft.com'&\$select=subject,receivedDateTime,isRead",   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    # Requested: Received read (What date/ date range???). Only subject,receivedDateTime,isRead
    'usermailreceivedread' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Users',
        'stdout' => 'User\'s mail activity is @result',
        'label' => 'secure_score_control_profile',
        'check' => "/users/lxxxx@nagios.com/messages?\$count=true&\$filter=isRead+eq+true+and+receivedDateTime+ge+2020-08-06+and+from/emailAddress/address+ne+'lxxxx@nagios.com'&\$select=subject,receivedDateTime,isRead",   # REST Query
        #'check' => "/users/lxxxx@nagiosdev.onmicrosoft.com/messages?\$count=true&\$filter=isRead+eq+true+and+receivedDateTime+ge+2020-08-06+and+from/emailAddress/address+ne+'lxxxx@nagiosdev.onmicrosoft.com'&\$select=subject,receivedDateTime,isRead",   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'users' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Users',
        'stdout' => 'Users is @result',
        'label' => 'Users in the organization',
        'check' => '/users',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'usersdelta' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'User Delta',
        'stdout' => 'User Delta is @result',
        'label' => 'users_delta',
        'check' => '/users/delta',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'usersdeleted' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Deleted Users',
        'stdout' => 'Deleted Users is @result',
        'label' => 'users_deleted',
        'check' => '/directory/deletedItems/microsoft.graph.user',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Groups
    # https://docs.microsoft.com/en-us/graph/api/resources/groups-overview?view=graph-rest-1.0
    #
    ### Groups/Group ###
    # https://docs.microsoft.com/en-us/graph/api/resources/group?view=graph-rest-1.0
    #
    #   groups - Groups in the organization, includes, but not limited to, Office 365 Groups.
    #       GET /groups
    #
    #   groupsDelta - Get newly created, updated, or deleted groups without having to perform a full read of the entire group collection.
    #       GET /groups/delta
    #
    #   groupsDeleted - Get recently deleted groups.
    #       GET /directory/deleteditems/microsoft.graph.group
    #
    #   API Permissions - Group.Read.All
    #
    'groups' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Users',
        'stdout' => 'Secure Score Control Profile is @result',
        'label' => 'secure_score_control_profile',
        'check' => '/groups',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'groupsdelta' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Group Deltas',
        'stdout' => 'Group Deltas is @result',
        'label' => 'groups_delta',
        'check' => '/groups/delta',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'groupsdeleted' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Deleted Groups',
        'stdout' => 'Deleted Groups is @result',
        'label' => 'groups_deleted',
        'check' => '/directory/deletedItems/microsoft.graph.group',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ####################################################################################################################
    #
    # Files
    #
    #   Example Requests
    #   =================================================================
    #   GET /groups/{groupId}/drives
    #   GET /sites/{siteId}/drives
    #

    ### Drives ###
    #
    #   drives - A user, group or sites drives.
    #       GET /groups/{groupId}/drives
    #       GET /sites/{siteId}/drives
    #       GET /users/{userId}/drives
    #
    #   API Permissions - Files.Read.All, Sites.Read.All
    #
    'drives' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Drives',
        'stdout' => 'Drives is @result',
        'label' => 'drives',
        'check' => '/users/40f42634-0ed2-40c4-9cf0-626d0f055828/drives',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ####################################################################################################################
    #
    # Identity and Access
    #
    #   Example Requests
    #   =================================================================
    #   GET /applications
    #   GET /subscribedSkus
    #

    ### Applications ###
    #
    #   applications - List of applications in the organization.
    #       GET /applications
    #
    #   application - Get a specific application belonging to the organization.
    #       GET /applications/{id}
    #
    #   application: delta
    #       GET /applications/delta - Get newly created, updated, or deleted applications.
    #
    #   applications: deleted - List of applications in the organization, deleted.
    #       GET /directory/deleteditems/microsoft.graph.application
    #
    #   API Permissions - Application.Read.All
    #
    'applications' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Applications',
        'stdout' => 'Applications is @result',
        'label' => 'applications',
        'check' => '/applications',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'application' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Application',
        'stdout' => 'Application is @result',
        'label' => 'application',
        'check' => '/applications/@id',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'applicationsdelta' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Applications Delta',
        'stdout' => 'Applications Delta is @result',
        'label' => 'applications_delta',
        'check' => '/applications/delta',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'applicationsdeleted' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Applications Deleted',
        'stdout' => 'Applications Deleted is @result',
        'label' => 'applications_deleted',
        'check' => '/directory/deleteditems/microsoft.graph.application',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Directory Audits ###
    #
    #   directoryAudits - List of audit logs generated by Azure Active Directory,
    #                     including audit logs generated by various services within Azure AD,
    #                     including user, app, device and group Management, privileged identity management (PIM),
    #                     access reviews, terms of use, identity protection,
    #                     password management (self-service and admin password resets), and self- service group management, etc.
    #       GET /auditLogs/directoryaudits
    #
    #   directoryAudit - A specific Azure Active Directory audit log item.
    #       GET /auditLogs/directoryaudits/{id}
    #
    #   API Permissions - AuditLog.Read.All
    #
    'directoryaudits' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Directory Audits',
        'stdout' => 'Directory Audits is @result',
        'label' => 'directory_audits',
        'check' => '/auditLogs/directoryAudits',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'directoryaudit' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Directory Audit',
        'stdout' => 'Directory Audit is @result',
        'label' => 'directory_audit',
        'check' => '/auditLogs/directoryAudits/@id',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Sign-in Audits ###
    #
    #   signIns - User and application sign-in activity for a tenant (directory).
    #       GET /auditLogs/signIns
    #
    #   signIn  - A specific sign-in.
    #       GET /auditLogs/signIns/{id}
    #
    #   API Permissions - AuditLog.Read.All
    #
    'signins' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Sign-in Audits',
        'stdout' => 'Sign-in Audits is @result',
        'label' => 'sign_in_audits',
        'check' => '/auditLogs/signIns',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'signin' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Sign-in Audit',
        'stdout' => 'Sign-in Audit is @result',
        'label' => 'sign_in_audit',
        'check' => '/auditLogs/signIns/@id',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Subscribed SKU ###
    #
    #   subscribedskus - List of commercial subscriptions for the organization.
    #       GET /subscribedSkus
    #
    #   subscribedsku - Get a specific subscription belonging to the organization.
    #       GET /subscribedSkus/{id}
    #
    #   API Permissions - Directory.Read.All
    #
    'subscribedskus' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Emerging Issues',
        'stdout' => 'Emerging Issues is @result',
        'label' => 'emerging_issues',
        'check' => '/subscribedSkus',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'subscribedsku' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Emerging Issues',
        'stdout' => 'Emerging Issues is @result',
        'label' => 'emerging_issues',
        'check' => '/subscribedSkus/@id',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

### Others to research...
    #https://docs.microsoft.com/en-us/graph/api/overview?view=graph-rest-1.0
    #https://docs.microsoft.com/en-us/graph/api/resources/directoryobject?view=graph-rest-1.0
    ### Users, cntd ###
    # https://docs.microsoft.com/en-us/graph/api/resources/users?view=graph-rest-1.0
    #
    ### Users/Directory objects ###
    # https://docs.microsoft.com/en-us/graph/api/resources/directoryobject?view=graph-rest-1.0
    #
    #  ONLY INCLUDED DATA, NOT MANAGEMENT FUNCTIONS
    #
    #  userPrincipalName => lxxxx@nagios.com
    #  id => 40f42634-0ed2-40c4-9cf0-626d0f055828
    #
    #   directoryObject - Represents an Azure Active Directory object.
    #       POST /users/{id | userPrincipalName}/getMemberObjects
    #       GET  /users/{id}/manager
    #       GET  /users/{id}/messages
    #       GET  /users/{id}/events
    #       GET  /users/{id}/drive
    #       GET  /users/{id}/memberOf
    #
    #   usersDelta - Get newly created, updated, or deleted users without having to perform a full read of the entire user collection.
    #       GET /users/delta
    #
    #   usersDeleted - Retrieve a list of recently deleted users.
    #       GET /directory/deleteditems/microsoft.graph.user
    #
    #   API Permissions - User.Read.All (Directory.Read.All) - least to most privileged
    #   (User.Read.All and GroupMember.Read.All), (User.Read.All and Group.Read.All), (User.Read.All?? and GroupMember.Read.All), Group.Read.All, Directory.Read.All
    #
    ####################################################################################################################
    #
    # Users/Directory Objects
    #
    #   Example Requests
    #   =================================================================
    #   GET /applications
    #   GET /subscribedSkus
    #

    ### Directory Objects ###
    #
    #   licensedetails - Retrieve a list of licenseDetails objects for enterprise users.
    #       GET /users/{id}/licenseDetails
    #
    #   API Permissions - User.Read.All, Directory.Read.All
    #
    'userlicensedetails' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Enterprise User\'s License Details',
        'stdout' => 'Enterprise User\'s License Details is @result',
        'label' => 'userlicensedetails',
        'check' => '/users/@id/licenseDetails',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Directory Objects ###
    #
    #   licensedetails - Retrieve a list of licenseDetails objects for enterprise users.
    #       GET /users/{id}/licenseDetails
    #
    #   API Permissions - User.Read.All, Directory.Read.All
    #
    'userlicensedetails' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Enterprise User\'s License Details',
        'stdout' => 'Enterprise User\'s License Details is @result',
        'label' => 'userlicensedetails',
        'check' => '/users/@id/licenseDetails',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Directory Objects ###
    #
    #   licensedetails - Retrieve a list of licenseDetails objects for enterprise users.
    #       GET /users/{id}/licenseDetails
    #
    #   API Permissions - User.Read.All, Directory.Read.All
    #
    'userlicensedetails' => array(
        'api' => EndpointConstants::GRAPH_API,
        'help' => 'Enterprise User\'s License Details',
        'stdout' => 'Enterprise User\'s License Details is @result',
        'label' => 'userlicensedetails',
        'check' => '/users/@id/licenseDetails',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ####################################################################################################################
    #
    # Azure Managment API
    #
    ### Resource Health Emerging Issues ###
    #
    #   emergingIssues - 
    #       GET /providers/Microsoft.ResourceHealth/emergingIssues?api-version=2018-07-01
    #
    'emergingissues' => array(
        'api' => EndpointConstants::MANAGEMENT_AZURE_API,
        'help' => 'Emerging Issues',
        'stdout' => 'Emerging Issues is @result',
        'label' => 'emerging_issues',
        'check' => 'providers/Microsoft.ResourceHealth/emergingIssues?api-version=2018-07-01',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ### Operations ###
    #
    #   operations - 
    #       GET /providers/Microsoft.Advisor/operations?api-version=2017-04-01
    #
    'operations' => array(
        'api' => EndpointConstants::MANAGEMENT_AZURE_API,
        'help' => 'Advisor REST API operations',
        'stdout' => 'Advisor REST API operations is @result',
        'label' => 'advisor_rest_api_operations',
        'check' => 'providers/Microsoft.Advisor/operations?api-version=2017-04-19',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ####################################################################################################################
    #
    # Azure AD Graph API
    #
    # https://graph.windows.net/{tenant_id}/{resource_path}?{api_version}
    #
    ###  ###
    #
    #   subscribedskus - 
    #       GET /subscribedSkus?api-version=1.6
    #
    'ADsubscribedskus' => array(
        'api' => EndpointConstants::AZURE_AD_API,
        'help' => 'Emerging Issues',
        'stdout' => 'Emerging Issues is @result',
        'label' => 'emerging_issues',
        'check' => '/subscribedSkus?api-version=1.6',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

    ####################################################################################################################
    #
    # Office 365 Service Communications API
    #
    # https://manage.office.com/api/v1.0/{tenant_id}/ServiceComms/{operation}
    #
    # Requires the Office 365 Audit Log and Audit Log Search, are turned on.
    #
    ###  ###
    #
    #   services - 
    #       GET /ServiceComms/Services
    #
    'services' => array(
        #'api' => EndpointConstants::AZURE_AD_API,
        'api' => EndpointConstants::MANAGE_OFFICE_API,
        'help' => 'Services',
        'stdout' => 'Services is @result',
        'label' => 'services',
        'check' => '/ServiceComms/Services',   # REST Query
        'type' => 'standard',
        'modifier' => 1,
        'unit' => '%',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),

### Others to research...
###

    ###
    ### Wizard data queries.  Figure out a better way to do this...
    ### Store this data???
    ###
    'nextLinkTest' => array(
        'api' => EndpointConstants::GRAPH_API,
        'check' => '/users?$select=userPrincipalName,displayName&$orderby=userPrincipalName&$top=10',   # REST Query
        'type' => 'list',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'userslist' => array(
        'api' => EndpointConstants::GRAPH_API,
        'check' => '/users?$select=userPrincipalName,displayName&$orderby=userPrincipalName',   # REST Query
        'type' => 'list',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'groupslist' => array(
        'api' => EndpointConstants::GRAPH_API,
        'check' => '/groups?$select=id,displayName&$orderby=displayName',   # REST Query
        'type' => 'list',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'productslist' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'check' => '/reports/getOffice365ActivationCounts',   # REST Query
        'type' => 'list',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'reportdatauserslist' => array(
        'api' => EndpointConstants::GRAPH_REPORTS_API,
        'check' => '/reports/getEmailActivityUserDetail(period=\'D7\')',   # REST Query
        'fields' => array('User Principal Name' => true),
        'type' => 'list',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'organization' => array(
        'api' => EndpointConstants::GRAPH_API,
        'check' => '/organization?$select=displayName,tenantType&$top=1',   # REST Query
        'type' => 'list',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    ###

    # General tests and debugging tests.
    'time2token' => array(
        'api' => EndpointConstants::GRAPH_API,  # This can be overridden on the command line.
        'help' => 'Time to get MS Identity Platform token.',
        'stdout' => 'Time to get identity token @results',
        'label' => 'time',
        'unit' => 's',
        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'time2connect' => array(
        'api' => EndpointConstants::GRAPH_API,  # This can be overridden on the command line.
        'help' => 'Time to query the Graph endpoint.',
        'stdout' => 'Time to query @api endpoint @results',
        'label' => 'time',
        'endpoint' => '',   # Most basic Query, to test basic access.
        'unit' => 's',
        'enabled' => true,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'test' => array(
        'help' => 'Run tests of all queries against the endpoint.',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    ),
    'runall' => array(
        'help' => 'Run tests of all queries against the endpoint, with status and performance data.',
        'enabled' => false,  # enabled => true, then list in USAGE.  enabled => false, available on the QT, for development.
    )
);

function parse_args() {
    global $logger;
    $logger->debug("", __METHOD__, __LINE__);

    $argSpecs = array(
        // Required arguments
        // Individual specifies a flag, rather than a key/value pair.
        array('short' => 'A',
            'long' => 'appid',
            'default' => '',
            'required' => true,
            'individual' => false,
            'help' => 'Specify Microsoft Azure Application (client) ID'),
        array('short' => 'T',
            'long' => 'tenant',
            'default' => '',
            'required' => true,
            'individual' => false,
            'help' => 'Specify Microsoft Azure Directory (tenant) ID'),
        array('short' => 'S',
            'long' => 'secret',
            'default' => '',
            'required' => true,
            'individual' => false,
            'help' => 'Specify Application\'s Client Secret'),
        array('short' => 'P',
            'long' => 'period',
            'default' => '',
            'required' => false,
            'individual' => false,
            'help' => 'Specify the period for the report, typically D7, D30, D90, D180 (D7 => 7 days)'),
        array('short' => 'F',
            'long' => 'filter',
            'default' => '',
            'required' => false,
            'individual' => false,
            'help' => 'Reduce the dataset from the Graph database, to the user/drive/etc.'),

        array('short' => 'w',
            'long' => 'warning',
            'default' => '',
            'required' => true,
            'individual' => false,
            'help' => 'Specify warning range'),
        array('short' => 'c',
            'long' => 'critical',
            'default' => '',
            'required' => true,
            'individual' => false,
            'help' => 'Specify critical range'),
        // Test to run
        array('short' => '',
            'long' => 'mode',
            'default' => 'False',
            'required' => true,
            'individual' => false,
            'help' => 'Must choose one and only one Test'),

        // Optional Nagios Plugin Information
        // Default checktype = 'Exchange'
        array('short' => '',
            'long' => 'checktype',
            'default' => '',
            'required' => false,
            'individual' => false,
            'help' => 'Specify endpoint checks'),
        array('short' => 'h',
            'long' => 'help',
            'default' => '',
            'required' => false,
            'individual' => true,
            'help' => 'This usage message'),
        array('short' => 'V',
            'long' => 'version',
            'default' => '',
            'required' => false,
            'individual' => true,
            'help' => 'The version of this plugin'),
        array('short' => 'v',
            'long' => 'verbose',
            'default' => '',
            'required' => false,
            'individual' => true,
            'help' => 'Increase verbosity up to 3 times, e.g., -vvv'),
        array('short' => '',
            'long' => 'complex',
            'default' => '',
            'required' => false,
            'individual' => true,
            'help' => 'Show traditional/complex log file entries, for use with verbose.'),
        array('short' => '',
            'long' => 'refresh',
            'default' => true,
            'required' => false,
            'individual' => true,
            'help' => 'Refresh the report data from the Graph database.'),
        array('short' => '',
            'long' => 'testurl',
            'default' => '',
            'required' => false,
            'individual' => false,
            'help' => 'Load data using the testurl, instead of MS Graph.'),

        # Data queries for the wizard.
        array('short' => '',
            'long' => 'modelist',
            'default' => '',
            'required' => false,
            'individual' => true,
            'help' => 'List of modes/tests available for wizard'),
    );
    
    $options = parse_specs($argSpecs);

    return $options;
}

function parse_specs($argSpecs) {
    global $MODES;
    global $VERSION;
    global $USAGE_STRING1;
    global $perfType;
    global $logger;
    global $requestedTest;
    $logger->debug("", __METHOD__, __LINE__);
    
    $shortOptions = '';
    $longOptions = array();
    $options = array();
    
    /**************************************************************************************
     * Create the array that will be passed to getopt, which accepts an array of arrays.
     * Each internal array has three entries, short option, long option and required.
     **************************************************************************************/
    foreach($argSpecs as $argSpec) {

        if (!empty($argSpec['short'])) {
            # getopt() - Optional ::, isn't working, so just make everything, that isn't a flag, required, for getopt().
            #$shortOptions .= "{$argSpec['short']}".(!empty($argSpec['required']) ? ':' : (empty($argSpec['individual']) ? '::' : ''));
            $shortOptions .= "{$argSpec['short']}".(!empty($argSpec['required']) ? ':' : (empty($argSpec['individual']) ? ':' : ''));
        }

        if (!empty($argSpec['long'])) {
            # getopt() - Optional ::, isn't working, so just make everything, that isn't a flag, required, for getopt().
            #$longOptions[] = "{$argSpec['long']}".(!empty($argSpec['required']) ? ':' : (empty($argSpec['individual']) ? '::' : ''));
            $longOptions[] = "{$argSpec['long']}".(!empty($argSpec['required']) ? ':' : (empty($argSpec['individual']) ? ':' : ''));
        }
    }

    /**************************************************************************************
     * Parse the command line args, with the builtin getopt function
     **************************************************************************************/
    $parsedArgs = getopt($shortOptions, $longOptions);

    /**************************************************************************************
     * This version of getopt() stops when it hits arguments that are not in the short
     * and/or long options.  Which means, if there is a "bad/unrecognized" argument, but
     * there were enough to run the script, anything after the "bad" argument is ignored.
     * So, if you add -vvv, or anything else to the end of the line, it will be ignored!
     * This is undesireable behavior, so catch it and get out with an error message.
     **************************************************************************************/
    $argx = 0;
    $prefix = "-";
    $allArgs = array();
    $argv = $GLOBALS["argv"];
    $argc = count($GLOBALS["argv"]);
    
    # Keep track of all the arguments.
    while (++$argx < $argc) {
        $matches = array();

        if (preg_match('/^-/', $argv[$argx])) {
            preg_match("/-{1,2}(\w*)/", $argv[$argx], $matches);    # Strip the -
            $arg = $matches[1];
            $arg = (strpos($arg, "vv") === 0) ? "v" : $arg; # Simplify -vv and -vvv to -v (so it will match).

            # Keep track of all the arguments and the prefixes.
            # If the next argument is data, rather than another prefix, grab it and save it along with the current prefix (-d master) vs (-v).
            # Increment the counter, so the data is skipped, on the next pass.
            $allArgs[$arg] = array($matches[0] => ((($argx + 1 < $argc) && !preg_match("/^-/", $argv[$argx + 1])) ? $argv[++$argx] : ""));
        }
    }

    foreach ($allArgs as $prefix => $arg) {
        $key = key($arg);

        if (!array_key_exists($prefix, $parsedArgs)) {
            nagios_exit("Illegal argument \"".$key." ".$arg[$key]."\", please check your command line.", STATUS_UNKNOWN);
        }
    }
    
    /**************************************************************************************
     * Handle version request
     **************************************************************************************/
    if (array_key_exists('version', $parsedArgs) || array_key_exists('V', $parsedArgs)) {
        nagios_exit($VERSION.PHP_EOL, STATUS_OK);
    }
    
    /**************************************************************************************
     * NOTE:  Backslashes in the data MUST be escaped, in order to make it here.
     **************************************************************************************/
    $logger->debug("argv [".var_export($GLOBALS["argv"], true)."]", __METHOD__, __LINE__);
    $logger->debug("argc [".$GLOBALS["argc"]."]", __METHOD__, __LINE__);
    $logger->debug("parsedArgs ".var_export($parsedArgs, true), __METHOD__, __LINE__);

    // Handle help request
    if (array_key_exists("help", $parsedArgs) || array_key_exists("h", $parsedArgs)) {
        print_usage();
        nagios_exit('', STATUS_OK);
    }
    
    /**************************************************************************************
     * Mode list for the Wizard.
     **************************************************************************************/
    if (array_key_exists('mode', $parsedArgs) && $parsedArgs['mode'] == 'modelist') {
        wizard_mode_list(); # This function does nagios_exit.
    }

    /**************************************************************************************
     * Make sure the input variables are sane.
     * Also check to make sure that all flags marked as required are present.
     **************************************************************************************/
    foreach($argSpecs as $argSpec) {
        $lOptions = $argSpec["long"];
        $sOptions = $argSpec["short"];
        
        if (array_key_exists($lOptions, $parsedArgs) && array_key_exists($sOptions, $parsedArgs)) {
            plugin_error("Command line parsing error: Inconsistent use of flag: ".$argSpec["long"]);
        }

        if (array_key_exists($lOptions, $parsedArgs)) {
            $options[$lOptions] = $parsedArgs[$lOptions];

        } elseif (array_key_exists($sOptions, $parsedArgs)) {
            $options[$lOptions] = $parsedArgs[$sOptions];

        } elseif ($argSpec["required"] == true) {
            plugin_error("Command line parsing error: Required variable \"".$argSpec["long"]."\" not present.");
        }

        // Find and verify the mode test.  --custom overrides mode.
        if ($argSpec['long'] == 'mode') {
            if (array_key_exists($options['mode'], $MODES)) {
                $logger->debug("TEST: ".$options['mode'], __METHOD__, __LINE__);
                $requestedTest = $options['mode'];  # The initial test/mode.  Used when running multiple tests.
            } else {
                plugin_error("Command line parsing error: The specified test \"".$options['mode']."\" is invalid, for the required argument \"mode\".  Please provide a valid test from the list.");
            }
        }
    }
    
    /**************************************************************************************
     * Handle verbosity request
     **************************************************************************************/
    if (array_key_exists('verbose', $parsedArgs) || array_key_exists('v', $parsedArgs)) {
        $level = $logger->getLogLevel();
        $verbosity = $logger->getLevelName($level);

        // We support up to 3 verbosity decreases, e.g., -vvv.
        if (array_key_exists('v', $parsedArgs) && is_array($parsedArgs['v'])) {
            $verboseCntr = 0;

            foreach ($parsedArgs['v'] as $v) {
                $verboseCntr++;

                if ($verboseCntr >= 4) {
                    $logger->notice("Lowest verbosity has already been reached! [".$verboseCntr."]", __METHOD__, __LINE__);
                    break;
                }

                $logger->verbose();
            }

        // Just lower verbosity once.
        } else {
            $logger->verbose();
        }

        $logger->debug("Adding verbosity... Original Log Level [".$level."], New Log Level [".$logger->getLogLevel()."]", __METHOD__, __LINE__);
        $logger->notice("Adding verbosity... Original Log Level [".$verbosity."], New Log Level [".$logger->getLevelName($logger->getLogLevel())."]", __METHOD__, __LINE__);
    }
    
    /**************************************************************************************
     * Handle complex logging request
     **************************************************************************************/
    if (array_key_exists("complex", $parsedArgs)) {
        $logger->complex();
    }
    
    /**************************************************************************************
     * Handle refresh report data request
     **************************************************************************************/
    if (array_key_exists("refresh", $parsedArgs)) {
        $options["refresh"] = true;
    } else {
        $options["refresh"] = false;
    }
    
    /**************************************************************************************
     * Handle testurl data override.
     * Example:
     * http://localhost/nagiosxi/includes/configwizards/microsoft_365/plugins/testdata/test001.html
     * ...or report csv data
     * http://localhost/nagiosxi/includes/configwizards/microsoft_365/plugins/testdata/reportTests.php?redirectUrl=mailusagebyuser_report.csv
     **************************************************************************************/
    $options["testurl"] = null;

    if (array_key_exists("testurl", $parsedArgs)) {
        $options["testurl"] = $parsedArgs['testurl'];
    }
    
    /**************************************************************************************
     * Handle filter, required by GRAPH_REPORTS_API.
     **************************************************************************************/
# If this is going to be required, need to use a different argument, than filter.
#
#    if (array_key_exists('type', $MODES[$options['mode']]) && $MODES[$options['mode']]['type'] != 'list' &&
#        $MODES[$options['mode']]['api'] == EndpointConstants::GRAPH_REPORTS_API && !array_key_exists("filter", $parsedArgs)) {
#
#        print_usage("Required argument \"--filter {name:value}\", is missing.  Please check your command line.");
#    }

    $logger->debug("Options\n".var_export($options, true), __METHOD__, __LINE__);

    return $options;
}

/******************************************************************************************
 * 
 ******************************************************************************************/
function wizard_mode_list() {
    global $MODES;
    global $logger;
    $logger->debug("", __METHOD__, __LINE__);

    $activeModes = array();

    foreach ($MODES as $mode => $test) {
        if ($test['enabled'] == true && $mode != 'time2token' && $mode != 'time2connect' && $mode != 'test' && $mode != 'runall') {
            $activeModes = array_merge($activeModes, array($mode => $test));
        }
    }

    $logger->debug("Active modes ".var_export($activeModes, true), __METHOD__, __LINE__);

    nagios_exit(json_encode($activeModes), STATUS_OK);
}

# Returns a JSON list of the organization or domains users (email and name)
function wizard_list($bodyJSON) {
    global $MODES;
    global $logger;
    $logger->debug("", __METHOD__, __LINE__);

    nagios_exit($bodyJSON, STATUS_OK);
}

function plugin_error($error_message) {
    global $logger;
    $logger->debug("", __METHOD__, __LINE__);

    print_usage("***ERROR***:\n\n{$error_message}\n\n");
}

function main() {
    global $logger;
    $logger->debug("", __METHOD__, __LINE__);

    // Handle the user's arguments.
    $options = parse_args();
    
    $check = new Check($options);
    $check->run_check();
}

########################################################################
# Argument handling functions
########################################################################

$VERSION = VERSION;  // Make the constant into a variable, so it can be embedded.  Yes this is annoying.
$PROGRAM = PROGRAM;  // Make the constant into a variable, so it can be embedded.  Yes this is annoying.
$CHECK = CHECK;  // Make the constant into a variable, so it can be embedded.  Yes this is annoying.
$mode = new Mode();  // Now it can be embedded in the variable definition, like VERSION.

$USAGE_STRING1 = <<<USAGE

{$CHECK} {$VERSION} - Copyright 2021 Nagios Enterprises, LLC.
                             Portions Copyright others (see source code).
                 
USAGE;

$USAGE_STRING2 = <<<USAGE

Usage: {$PROGRAM} <options>

usage = "usage: {$PROGRAM} -A appid -T tenant -S secret --mode testname -w warning -c critical [-P period] [-F filter] [-v | -vv | -vvv | --verbose --complex --refresh --testurl]"

Options:
    -h, --help          Print detailed help screen.
    -V, --version       Print version information.
    -v, --verbose       Optional: Increases verbosity.  Verbosity may be increased up to 3 times,
                        using -v, e.g., -vvv
        --complex       Optional: Show Traditional/Complex log entries, for use with verbose.
        --refresh       Optional: Refresh the report data from Graph database.
        --testurl       Optional: Pull test data from the specified url.

    -A, --appid         Required: Microsoft Azure Application (client) ID
    -T, --tenant        Required: Microsoft Azure Directory (tenant) ID
    -S, --secret        Required: Application Client Secret

        --mode          Required: Must specify one and only one test, from the list, below...
                                  (-h or --help for more information)

    -F, --filter        Required: Limit result to 1 entry from the Report.  CSV Report filtering
                                  occurs after the report is downloaded. (GRAPH_REPORTS_API) (Not currently enforced) 
                        Optional: Filter/Reduce the dataset from the Graph database. (GRAPH_API)

                                Example...                                 
                                '{"User Principal Name":"lxxxx@nagios.com"}'
                                '("User Principal Name":"lxxxx@nagios.com")'

{$mode->print_list_of_tests()}

    -P, --period        Optional: Specify the number of days, over which to aggregate the report.
                                  Default is D7 (7 days).
                                  D7 => 7 days, D30 => 30 days, D90 => 90 days, D180 => 180 days

    -w, --warning=<WARNING>     Required: The warning values, see:
    -c, --critical=<CRITICAL>   Required: The critical values, see
                          
Note: Warning and critical threshold values should be formatted via the
Nagios Plugin guidelines. See guidelines here:
https://nagios-plugins.org/doc/guidelines.html#THRESHOLDFORMAT
    
Examples:   10          Alerts if value is > 10
            30:         Alerts if value < 30
            ~:30        Alerts if value > 30
            30:100      Alerts if 30 > value > 100
            @10:200     Alerts if 30 >= value <= 100
            @10         Alerts if value = 10
                          
This plugin checks the status of Microsoft 365 services

USAGE;


########################################################################
# Argument handling functions
########################################################################

function print_usage($message=null) {
    global $USAGE_STRING1;
    global $USAGE_STRING2;
    global $logger;
    $logger->debug("", __METHOD__, __LINE__);

    if ($message) echo(PHP_EOL.$message.PHP_EOL);

    nagios_exit($USAGE_STRING1.$USAGE_STRING2, STATUS_UNKNOWN); // Exit status 3 for Nagios plugins is 'unknown'.
}

class Mode {
    function print_list_of_tests() {
        global $MODES;
        global $options;
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $outputString = "                        Available Tests".PHP_EOL.PHP_EOL;

        foreach ($MODES as $name => $test) {
            if ($test['enabled'] == true) {
                $logger->debug("print_list_of_tests key [".($name)."] ".var_export($test, true), __METHOD__, __LINE__);
                $outputString .= "                        ".$name.PHP_EOL;
            }
        }

        return $outputString;
    }
}

########################################################################
# Database Connection and test setup class and functions
########################################################################

class Check {
    private $options = null;
    private $connection = null;
    private $testCode = null;
    private $testStatusMsg = null;
    private $testObj = null;

    public function __construct($options) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $this->options = $options;
    }

    public function run_check() {
        global $MODES;
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $mode = $this->options['mode'];
        $logger->debug("MODES[$mode]".var_export($MODES[$mode], true), __METHOD__, __LINE__);

        $connectionTime = 0;
        $authVersion = 'v2'; #= $MODES[$mode]['version'];
        $result = null;

        # Create the test object now, so we can skip Authentication, unless needed.
        # If we hit the endpoints too often, we can get temporarily locked out.
        $requestFactory = new RequestFactory();
        $this->testObj = $requestFactory->createRequest($MODES[$mode], $this->options);

        ################
        #
        # SECURITY TOKEN
        #
        # If we need to update the data, get a new security token
        try {
            if (empty(self::$testurl) && (!$this->testObj->getIsCurrent() || $this->options['refresh'])) {
                $logger->info("Connecting to Microsoft Identity Platform endpoint", __METHOD__, __LINE__);
        
                // OAuth 2.0 Client Credentials Grant/Flow (two-legged OAuth).
                // Attempt to connect to the MS Identity Platform endpoint and get an access token.
                if ($authVersion == 'v2') {
                    $logger->debug("Microsoft Identity Platform v2 endpoint", __METHOD__, __LINE__);
                    $authToken = new AuthRequest($MODES[$mode], $this->options);
                } else if ($authVersion == 'v1') {
                    $logger->debug("Microsoft Identity Platform v1 endpoint", __METHOD__, __LINE__);
                    $authToken = new AuthRequest($MODES[$mode], $this->options);
                } else {
                    nagios_exit("Invalid version [".$authVersion."] for Microsoft Identity Platform endpoint. ".__METHOD__." ".__LINE__, STATUS_CRITICAL);
                }

                $authResults = $authToken->getAuthResults();

                if (!$authResults['status_cd']) {
                    nagios_exit("ERROR: ".$authResults['error_code'].": ".$authResults['status_msg'], STATUS_CRITICAL);
                }
        
                $connectionTime = $authResults['total_time'];

                # Set the token, in the testObj, now that we have it.
                $logger->info("authToken [".$authToken->getToken()."]", __METHOD__, __LINE__);
                $this->testObj->setRequestHeader($authToken->getToken());
                $logger->debug("testObj->getToken() [".$this->testObj->getToken()."]", __METHOD__, __LINE__);
            }
        } catch (PDOException $e) {
            $outputMsg = "CRITICAL: Could not connect to $endpoint as ".$this->options['appid']." (Exception: " . $e->getMessage() . ").\n";
            nagios_exit($outputMsg, STATUS_CRITICAL);
        }

        ################
        #
        # QUERY GRAPH ENDPOINT
        #
        if ($mode != 'time2token' && $mode != 'time2connect') {
            $logger->info("Connecting to Microsoft Graph endpoint", __METHOD__, __LINE__);

            if ($this->testObj == null) {
                nagios_exit("Error creating request object.", STATUS_CRITICAL);
            }
       
            $check = $MODES[$mode]['check'];

            # If there is a @period in the check, add the value from $this->options['period'], otherwise the first value in the 'period' array.
            if (preg_match("/@period/i", $check)) {
                $logger->debug("MODES[mode]['period'] ".var_export($MODES[$mode]['period'], true), __METHOD__, __LINE__);

                $periodArray = $MODES[$mode]['period'];
                reset($periodArray);    # Set to first element.  FYI: reset also returns the first value (not key) of the array.
                $defaultPeriod = key($periodArray);

                $logger->debug("defaultPeriod [".$defaultPeriod."]", __METHOD__, __LINE__);

                # Verify the period from the command line is valid?  This should really be done with the other command line validation.
                $check = strtr($check, array("@period" => (array_key_exists('period', $this->options)) ? $this->options['period'] : $defaultPeriod));
            }

            $logger->debug("check [".$check."]", __METHOD__, __LINE__);

            try {
                $logger->debug("check [".$check."]", __METHOD__, __LINE__);

                # Query the appropriate endpoint, with the security token, or get saved, "unexpired" data.
                # NOTE: Reports require a second call to getNewReport, to get the data from the url provided in queryEndpoint().
                if (!empty($this->options['testurl']) || !$this->testObj->getIsCurrent() || $this->options['refresh']) {
                    $logger->debug("### queryEndpoint()", __METHOD__, __LINE__);
                    $result = $this->testObj->queryEndpoint((($mode != 'time2connect') ? $check : ""), null);

                    $httpCode = (array_key_exists('http_code', (array) $result)) ? $result['http_code'] : 200;
                    $httpMsg = (array_key_exists('http_msg', (array) $result)) ? $result['http_msg'] : "OK";
                    $errorCode = (array_key_exists('error_code', (array) $result)) ? $result['error_code'] : 0;
                    $statusMsg = (array_key_exists('status_msg', (array) $result)) ? $result['status_msg'] : "OK";

                    # cUrl error.
                    if ($errorCode != 0 && !array_key_exists('http_code', $result)) {
                        $outputMsg = "CRITICAL: Error querying endpoint. Error code [".$errorCode."] message [".$statusMsg."]\n";
                        nagios_exit($outputMsg, STATUS_CRITICAL);
                    }

                    # Graph error.
                    if ($httpCode != 200) {
                        $outputMsg = "CRITICAL: Error querying endpoint. HTTP error [".$httpCode."] message [".$httpMsg."]. Graph error [".$errorCode."] message [".$statusMsg."]\n";
                        nagios_exit($outputMsg, STATUS_CRITICAL);
                    }

                    $logger->info("Successful connecting to MS Graph [".$connectionTime."]", __METHOD__, __LINE__);
                }
            
                # If a simple list is requested...
                if ($MODES[$mode]['type'] == 'list') {
                    if ($MODES[$mode]['api'] == EndpointConstants::GRAPH_REPORTS_API) {
                        $result = $this->testObj->processData('json');

                        wizard_list($result[0]); # Calls nagios_exit;
                    }

                    wizard_list($this->testObj->getBody()); # Calls nagios_exit;
                }

                $result = $this->testObj->processData($check);

                $logger->debug("Microsoft Graph request results ".var_export($result, true), __METHOD__, __LINE__);

                $connectionTime = (empty($result['total_time'])) ? $connectionTime : $result['total_time'];

            } catch (PDOException $e) {
                $outputMsg = "CRITICAL: Could not connect to $endpoint as ".$this->options['appid']." (Exception: " . $e->getMessage() . ").\n";
                nagios_exit($outputMsg, STATUS_CRITICAL);
            }
        }

        $connectionTime = $this->testObj->sigFig($connectionTime, 6);

        // Run all the tests.
        if (!$mode || $mode == 'time2connect') {

            $logger->debug("*** No Mode specified or time2Connect was specified: connectionTime [$connectionTime]", __METHOD__, __LINE__);

            $exitCode = $this->testObj->processResults($connectionTime, null, null);
                            
            nagios_exit($this->testObj->getOutputMsg(), $exitCode);

        } elseif ($mode == 'time2token') {

            $logger->debug("*** Error connecting to endpoint or time2token was specified: connectionTime [$connectionTime]", __METHOD__, __LINE__);

            $exitCode = $this->testObj->processResults($connectionTime, null, null);

            nagios_exit($this->testObj->getOutputMsg(), $exitCode);

        } elseif ($mode == 'test') {
# NOT TESTED
            $logger->debug("*** Run All the plugins", __METHOD__, __LINE__);
            $this->run_plugin_tests();
            
        } elseif ($mode == 'runall') {
# NOT TESTED
            $logger->debug("*** Run All the Tests", __METHOD__, __LINE__);
            $this->run_all_plugins($connectionTime);

        } else {
            $logger->debug("*** Run Only One Test", __METHOD__, __LINE__);

            $exitCode = $this->testObj->processMultiResults($result, $MODES[$mode]);

            nagios_exit($this->testObj->getOutputMsg(), $exitCode);
        }
    }

    /**********************************************************************************
     * This is just a basic "do all the plugins function" test.
     * Use run_plugin_tests() to get output from each plugin.
     *
     * Note:  Not as useful as the Python version, since try/catch is a bit
     *        lacking in the current 5.4 version of PHP.
     */
    function run_plugin_tests() {
        global $MODES;
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $failed = 0;
        $total  = 0;

        foreach ($MODES as $mode => $test) {
            $logger->debug("mode [$mode]", __METHOD__, __LINE__);

            if ($mode == 'time2token' || $mode == 'time2connect' || $mode == 'test' || $mode == 'runall') {
                $logger->debug("SKIPPING $mode", __METHOD__, __LINE__);
                continue;
            }

            $total += 1;

            // Set the next test to run.
            $this->options['mode'] = $mode;
#            $statusCode = $this->execute_query();

            if ($statusCode == STATUS_OK ||
                $statusCode == STATUS_WARNING ||
                $statusCode == STATUS_CRITICAL) {

                print(strtr("@testName passed!".PHP_EOL, array("@testName" => $mode)));

            } else {
                $failed += 1;

                print(strtr("@testName failed with: @testStatusMsg".PHP_EOL, array("@testName" => $mode, "@testStatusMsg" => $this->testStatusMsg)));
            }
        }

        print(strtr("@failed/@total tests failed.", array("@failed" => $failed, "@total" => $total)));
    }
        
    /**********************************************************************************
     * Runs all the plugins and displays the output from each plugin/test.
     */
    function run_all_plugins($connectionTime) {
        global $MODES;
        global $logger;
        global $overrideErrors;
        $logger->debug("", __METHOD__, __LINE__);

        $failed = 0;
        $total  = 0;
        $overrideErrors = true;

        foreach ($MODES as $mode => $test) {
            $logger->debug("mode [$mode]", __METHOD__, __LINE__);

            $total += 1;

            // Set the next test to run.
            $this->options['mode'] = $mode;

            if ($mode == 'test' || $mode == 'runall') {
                $logger->debug("SKIPPING $mode", __METHOD__, __LINE__);
                $total--;

                continue;
            } else if ($mode == 'time2token') {

                $unit = array_key_exists("unit", $MODES[$mode]) ? $MODES[$mode]['unit'] : "";
                $label = $MODES[$mode]['label'];
                $stdout = $MODES[$mode]['stdout'];
                $logger->debug("*** time2token was specified: connectionTime [$connectionTime]", __METHOD__, __LINE__);

                $exitCode = $this->testObj->processResults($connectionTime, null, null);

                $this->outputMsg = $this->testObj->getOutputMsg();

            } else if ($mode == 'time2connect') {

                $unit = array_key_exists("unit", $MODES[$mode]) ? $MODES[$mode]['unit'] : "";
                $label = $MODES[$mode]['label'];
                $stdout = $MODES[$mode]['stdout'];
                $logger->debug("*** No Mode specified or time2Connect was specified: connectionTime [$connectionTime]", __METHOD__, __LINE__);

                $this->testObj = new GraphRequest($MODES, $this->options);
                $exitCode = $this->testObj->processResults($connectionTime, null, null);

                $this->outputMsg = $this->testObj->getOutputMsg();

            } else {
                $logger->debug("*** ".$MODES[$mode].": connectionTime [$connectionTime]", __METHOD__, __LINE__);
                $this->testObj = new GraphRequest($MODES, $this->options);
#                $statusCode = $this->execute_query();
            }

            print($this->outputMsg);
        }

        print(strtr("@total tests.", array("@total" => $total)).PHP_EOL);
    }
}

####################################################################################################
# Microsoft Endpoint API classes
####
#
# OAuth 2.0 Client Credentials Grant/Flow (two-legged OAuth).
# Attempt to connect to the MS Identity Platform endpoint and get an access token.
####################################################################################################
class AuthRequest {
    protected static $testMode = null;
    protected static $token = null;
    protected static $authResults = null;
    protected static $url = null;
    protected static $scopes = null;


    public function __construct($testMode, $options) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        #$logger->debug("testMode ".var_export($testMode, true), __METHOD__, __LINE__);
        $logger->debug("options ".var_export($options, true), __METHOD__, __LINE__);

        // Log in to Microsoft using client-credential flow
#        $url = "https://login.microsoftonline.com/".$tenant."/oauth2/v2.0/token";
#        $url = "https://login.microsoftonline.com/".$tenant."/oauth2/token";
        if ($testMode['api'] == EndpointConstants::MANAGEMENT_AZURE_API) {
            self::$url = strtr(EndpointConstants::AUTH_TOKEN_V1_ENDPOINT, array("@tenant" => $options['tenant']));
            self::$scopes = EndpointConstants::MANAGEMENT_AZURE_RESOURCE;
        } else if ($testMode['api'] == EndpointConstants::GRAPH_API) {
            self::$url = strtr(EndpointConstants::AUTH_TOKEN_V2_ENDPOINT, array("@tenant" => $options['tenant']));
            self::$scopes = EndpointConstants::GRAPH_SCOPE;
        } else if ($testMode['api'] == EndpointConstants::GRAPH_REPORTS_API) {
            self::$url = strtr(EndpointConstants::AUTH_TOKEN_V2_ENDPOINT, array("@tenant" => $options['tenant']));
            self::$scopes = EndpointConstants::GRAPH_SCOPE;
        } else if ($testMode['api'] == EndpointConstants::AZURE_AD_API) {
            self::$url = strtr(EndpointConstants::AUTH_TOKEN_V1_ENDPOINT, array("@tenant" => $options['tenant']));
            self::$scopes = EndpointConstants::AZURE_AD_RESOURCE;
        } else if ($testMode['api'] == EndpointConstants::MANAGE_OFFICE_API) {
            #self::$url = strtr(EndpointConstants::AUTH_TOKEN_V1_ENDPOINT, array("@tenant" => $options['tenant']));
            self::$url = strtr(EndpointConstants::AUTH_TOKEN_V2_ENDPOINT, array("@tenant" => $options['tenant']));
            #self::$scopes = EndpointConstants::MANAGE_OFFICE_RESOURCE;
            self::$scopes = EndpointConstants::MANAGE_OFFICE_SCOPE;
        }

        self::$authResults = $this->getAccessToken($options['tenant'], $options['appid'], $options['secret']);

        $logger->debug("tenant [".$options['tenant']."] applicationId [".$options['appid']."] applicationSecret [".$options['secret']."]", __METHOD__, __LINE__);
        $logger->debug("api [".$testMode['api']."]", __METHOD__, __LINE__);
    }

    public function getToken() {
        return self::$token;
    }

    public function getAuthResults() {
        return self::$authResults;
    }

    /**
     * Get an access token for a Microsoft API Endpoint.
     *
     * @return string $accessToken
     */
    function getAccessToken($tenant, $applicationId, $applicationSecret) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $logger->debug("tenant [".$tenant."] applicationId [".$applicationId."] applicationSecret [".$applicationSecret."]", __METHOD__, __LINE__);

        $body = "grant_type=client_credentials&".
                "client_id=".$applicationId."&".
                "client_secret=".$applicationSecret."&".
                self::$scopes;
        $logger->debug("self::url [".self::$url."] body [".$body."]", __METHOD__, __LINE__);

        $curlHandle = curl_init();
        $curlError = curl_errno($curlHandle);

        if (!$curlHandle) {
            die("Couldn't initialize a cURL handle");

            if (version_compare(PHP_VERSION, '5.5.0') >= 0) {
                $curlError = curl_strerror($curlError);
            }

            $processedResults = array_merge($response, array('error_code' => $curlError, "status_msg" => "Failed to retrieve access token! [".$curlError."]"));
            return $processedResults;
        }

        $options = array (
            CURLOPT_URL => self::$url,
            CURLOPT_POST => true,           # Regular HTTP POST (instead of default GET).
            CURLOPT_RETURNTRANSFER => true, # Return response as a string.
            CURLOPT_HEADER => false,        # Do NOT include the header in the response.
            CURLOPT_POSTFIELDS => $body,    # Data to post in an HTTP POST operation.
            CURLINFO_HEADER_OUT => true,    # Track the handle's request string.
            CURLOPT_CONNECTTIMEOUT => 10,   # Wait up to 10 seconds, to connect.
            CURLOPT_TIMEOUT => 60,          # Execute cURL request for up to 60 seconds.
        );

        curl_setopt_array($curlHandle, $options);

        // Request to Microsoft Identity Platform Endpoint.
        $curlResults = curl_exec($curlHandle);
        $curlError = curl_errno($curlHandle);

        if (curl_errno($curlHandle)) {
            $curlError = $curlError;

            if (version_compare(PHP_VERSION, '5.5.0') >= 0) {
                $curlError = curl_strerror($curlError);
            }

            $logger->info("Error: ".curl_errno($curlHandle), __METHOD__, __LINE__);
            $processedResults = (!is_array($curlResults)) ? array('status_cd' => false, 'error_code' => $curlError, "status_msg" => "Failed to retrieve access token! [".$curlError."]") : array_merge($curlResults, array('error_code' => $curlError, "status_msg" => "Failed to retrieve access token! [".$curlError."]"));

            return $processedResults;
        } 

        $info = curl_getinfo($curlHandle);
        curl_close($curlHandle);

        $connectTime = round($info['total_time'], 6);

        $response = json_decode($curlResults, true);    # ['access_token'];

        if (array_key_exists("access_token", $response)) {
            self::$token = $response["access_token"];
            $processedResults = array_merge($response, array("status_cd" => true, "status_msg" => "Success acquiring access token", "total_time" => $connectTime));
            self::$token = $response['access_token'];
        } else if (array_key_exists("error", $response)) {
            $processedResults = array_merge($response, array("status_cd" => false, "status_msg" => "Failed to retrieve access token", "total_time" => $connectTime), $response);
        } else {
            $processedResults = array_merge($response, array("status_cd" => false, "status_msg" => "Failed to retrieve access token", "total_time" => $connectTime));
        }

        $logger->debug("Microsoft Identity Platform token results ".var_export($processedResults, true), __METHOD__, __LINE__);

        return $processedResults;
    }
}

#
# Factory to create the proper request object.
#
class RequestFactory {
    public function createRequest($mode, $options) {
        global $logger;

        switch ($mode['api']) {
            case EndpointConstants::GRAPH_API:
                $logger->debug("GRAPH_API", __METHOD__, __LINE__);
                return new GraphRequest($mode, $options);
            case EndpointConstants::GRAPH_REPORTS_API:
                $logger->debug("GRAPH_REPORTS_API", __METHOD__, __LINE__);
                return new GraphReportRequest($mode, $options);
            case EndpointConstants::AZURE_AD_API:
                $logger->debug("AZURE_AD_API", __METHOD__, __LINE__);
                return new AzureADRequest($mode, $options);
            case EndpointConstants::MANAGEMENT_AZURE_API:
                $logger->debug("MANAGEMENT_AZURE_API", __METHOD__, __LINE__);
                return new AzureManagementRequest($mode, $options);
            case EndpointConstants::MANAGE_OFFICE_API:
                $logger->debug("MANAGE_OFFICE_API", __METHOD__, __LINE__);
                return new ManageOfficeRequest($mode, $options);
            default:
                # Error, this should never happen.
                $logger->error("ERROR! Invalid API", __METHOD__, __LINE__);
                return null;
        }
    }
}

abstract class BaseRequest {
    # Set in constructor.
    protected static $api = null;
    protected static $options = null;
    protected static $stdout = null;
    protected static $label = null;
    protected static $unit = null;
    protected static $type = null;
    protected static $multiple = false;
    protected static $modifier = null;
    protected static $fields = array();
    protected static $limit = null;
    protected static $filter = null;
    protected static $filterIdx = null;

    protected static $result = array();
    protected static $calculatedResult = null;
    protected static $outputMsg = null;

    protected static $testStatusMsg = null;
    protected static $connectTime = null;

    # Used by function handleHeaderLine($curl, $line) - curl callback function setup with CURLOPT_HEADERFUNCTION option.
    protected static $headers = array();
    protected static $code = null;
    protected static $headerLength = 0;

    # Used by function queryEndpoint($check)
    protected static $urlBase = null;       # Set in the child constructor
    protected static $token = null;         # Set in the child constructor - security token for curl request.
    protected static $header = array();     # Set in the child constructor - http headers for curl request.
    protected static $graphError = null;
    protected static $httpCode = null;
    protected static $curlError = null;
    protected static $curlInfo = null;
    protected static $body = null;
    protected static $testurl = null;

    protected static $isCurrent = false;
    
    protected static $response = array();   # Set in function getReport($reportUrl).

    protected static $jsonResults = array();   # Used to recursively retrieve the complete data set.

    public function __construct($mode, $options) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        self::$options = $options;
        self::$stdout = array_key_exists("stdout", $mode) ? $mode['stdout'] : "";
        self::$label = array_key_exists("label", $mode) ? $mode['label'] : "";
        self::$api = array_key_exists("api", $mode) ? $mode['api'] : "";
        self::$unit = array_key_exists("unit", $mode) ? $mode['unit'] : "";
        self::$multiple = array_key_exists("multiple", $mode) ? $mode['multiple'] : false;
        self::$modifier = (array_key_exists("modifier", $mode) && !empty($modifier)) ? $mode['modifier'] : 1;   // if no modifier, default is 1, no change.
        self::$fields = array_key_exists("fields", $mode) ? $mode['fields'] : "";
        self::$limit = array_key_exists("limit", $mode) ? $mode['limit'] : "";
        self::$type = array_key_exists("type", $mode) ? $mode['type'] : "";

        self::$filter = array_key_exists("filter", $options) ? $options['filter'] : "";
        self::$filterIdx = array_key_exists("filterIdx", $mode) ? $mode['filterIdx'] : "";


        # This is currently only for reports.  May be necessary for everything, at some point, so in the Base class.
        # Check if we need to pull new data (primarily for reports).
        $filename = self::getFileName();
        $logger->debug("filename [".$filename."]", __METHOD__, __LINE__);

        # If the report is current and refresh has NOT been requested, then the report is current.
        self::$isCurrent = self::isCurrentReport($filename) && !self::$options['refresh'];

        # Used to pull test data, instead of from the Graph database.
        self::$testurl = self::$options['testurl'];

        $logger->debug("isCurrent [".self::$isCurrent."]", __METHOD__, __LINE__);
        $logger->debug("options (Array) stdout [".self::$stdout."] label [".self::$label."] unit [".self::$unit."] modifier [".self::$modifier."] filter [".self::$filter."] filterIdx [".self::$filterIdx."]", __METHOD__, __LINE__);
    }

    public function getToken() {
        return self::$token;
    }

    public function setToken($token) {
        self::$token = $token;
    }

    public function getIsCurrent() {
        return self::$isCurrent;
    }

    public function getBody() {
        return self::$body;
    }

    public function getOutputMsg() {
        return self::$outputMsg;
    }

    public function getTestStatusMsg() {
        return self::$testStatusMsg;
    }

    // Create a temporary file in the temporary files directory.
    protected function getFileName() {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $tmpDir = sys_get_temp_dir();
        $report = self::$options['tenant'].self::$options['mode'];

        if (!is_dir($tmpDir) || !is_writable($tmpDir)) {
            $outputMsg = ((!is_dir($tmpir)) ? "The system tmp directory [$tmpDir] does NOT exist!" : "The system tmp directory [$tmpDir] MUST be writeable!");
            $logger->error($outputMsg, __METHOD__, __LINE__);
            nagios_exit($outputMsg, STATUS_UNKNOWN); 
        }

        $uniqueId = md5($report);
        $logger->debug("Generate filename (md5): uniqueId [$uniqueId] report [$report] mode [".self::$options['mode']."] tenant [".self::$options['tenant']."]", __METHOD__, __LINE__);

        // The hash always returns the same result, so had to add the name of the test, so multiple runs don't overwrite each other.
        // e.g., individual files for each test.
        $fullFilePath = strtr("@tmpDirectory/ms365-@uniqueId.tmp", array("@tmpDirectory" => $tmpDir, "@uniqueId" => $uniqueId));
        $logger->debug("fullFilePath [$fullFilePath]", __METHOD__, __LINE__);

        return $fullFilePath;
    }

    protected function isCurrentReport($filename) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        if (file_exists($filename)) {

            $reportAge = filemtime($filename);
            $logger->debug($filename." was last modified: " . date ("F d Y H:i:s.", $reportAge), __METHOD__, __LINE__);

            # File older than 4 hours
            if (time() - $reportAge > 4 * 3600) {
                $logger->debug($filename." is old!  Pull latest report " . date ("F d Y H:i:s.", $reportAge), __METHOD__, __LINE__);
                return false;
            }

            # File is age is good.
            return true;
        }

        # No file created, yet.
        $logger->debug($filename." does NOT exist, yet", __METHOD__, __LINE__);
        return false;
    }

    // Function to process the results
    // process_results makes $outputMsg - the Status & Performance data available and returns the exit code.
    public static function processResults($queryDuration, $finalResult, $outputMsg) {
        global $logger;
        global $overrideErrors;
        $logger->debug("", __METHOD__, __LINE__);
        $logger->debug("input queryDuration [$queryDuration] options (Array) outputMsg [$outputMsg]", __METHOD__, __LINE__);

        // Save the outputMsg
        self::$outputMsg = $outputMsg;

        $state = "OK";
        $exitCode = STATUS_OK;
        $warning = self::$options['warning'];
        $critical = self::$options['critical'];
        
        if (!empty($warning)) {
            switch (self::check_nagios_threshold($warning, $finalResult)) {
                case STATUS_UNKNOWN:
                    $exitCode = STATUS_UNKNOWN;
                    $state = "UNKNOWN";
                    self::$outputMsg = "ERROR: In range threshold START:END, START must be less than or equal to END";
                case 1:
                    $state = "WARNING";
                    $exitCode = STATUS_WARNING;
            }

            $logger->debug("Check Warning Threshold: exitCode [$exitCode ] state [$state] outputMsg [self::$outputMsg]", __METHOD__, __LINE__);
        }
        
        if (!empty($critical)) {
            switch (self::check_nagios_threshold($critical, $finalResult)) {
                case STATUS_UNKNOWN:
                    $exitCode = STATUS_UNKNOWN;
                    $state = "UNKNOWN";
                    self::$outputMsg = "ERROR: In range threshold START:END, START must be less than or equal to END";
                case 1:
                    $state = "CRITICAL";
                    $exitCode = STATUS_CRITICAL;
            }

            $logger->debug("Check Critical Threshold: exitCode [$exitCode ] state [$state] outputMsg [self::$outputMsg]", __METHOD__, __LINE__);
        }
        
        if (self::$options['mode'] == 'time2token' || self::$options['mode'] == 'time2connect') {
            $queryResult = $queryDuration;
        } else {
            $queryResult = $finalResult;
        }

        $mappedResult = strtr(self::$stdout, array("@api" => self::$api, "@result" => $queryResult));
        $logger->debug("mappedResult: [$mappedResult] stdout [".self::$stdout."] queryResult [".$queryResult."]", __METHOD__, __LINE__);

        // The output consists of Status Data|Performance Data
        // Status & Performance Data Format from the Nagios Plugins Development Guidelines @ https://nagios-plugins.org/doc/guidelines.html#AEN33
        //
        // Status Data Format
        // SERVICE STATUS: Information text
        //
        // Performance Data Format
        // 'label'=value[UOM];[warning];[critical];[min];[max]
        if (self::$multiple) {
            $logger->info("MULTIPLE", __METHOD__, __LINE__);
            self::$outputMsg = strtr("@state: [@id] @mappedResult|@label=@result@unit;@warning;@critical;;".PHP_EOL,
                                    array("@state" => $state, "@id" => self::$options['filter'], "@mappedResult" => $mappedResult,
                                          "@label" => self::$label, "@result" => $queryResult, "@unit" => self::$unit,
                                          "@warning" => $warning, "@critical" => $critical));
        } else {
            $logger->info("SINGLE", __METHOD__, __LINE__);
            self::$outputMsg = strtr("@state: @mappedResult|@label=@result@unit;@warning;@critical;;".PHP_EOL,
                                    array("@state" => $state, "@mappedResult" => $mappedResult,
                                          "@label" => self::$label, "@result" => $queryResult, "@unit" => self::$unit,
                                          "@warning" => $warning, "@critical" => $critical));
        }

        $logger->info("outputMsg: [".self::$outputMsg."]", __METHOD__, __LINE__);
        $logger->debug("@state [$state] @stdout [".self::$stdout."] @label [".self::$label."] @result [$mappedResult] @unit [".self::$unit."] @warning [$warning] @critical [$critical]", __METHOD__, __LINE__);

        return $exitCode;
    }

    private static function checkThreshold($warning, $critical, $result) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);
        $logger->debug("checkThreshold() warning [".$warning."] critical [".$critical."] result [".$result."]", __METHOD__, __LINE__);

        $state = "OK";
        $exitCode = STATUS_OK;
        $outputMsg = "";

        if (!empty($warning)) {
            switch (self::check_nagios_threshold($warning, $result)) {
                case STATUS_UNKNOWN:
                    $exitCode = STATUS_UNKNOWN;
                    $state = "UNKNOWN";
                    $outputMsg = "ERROR: In range threshold START:END, START must be less than or equal to END";

                case 1:
                    $exitCode = STATUS_WARNING;
                    $state = "WARNING";
                    $outputMsg = "";
            }
        }

        if (!empty($critical)) {
            switch (self::check_nagios_threshold($critical, $result)) {
                case STATUS_UNKNOWN:
                    $exitCode = STATUS_UNKNOWN;
                    $state = "UNKNOWN";
                    $outputMsg = "ERROR: In range threshold START:END, START must be less than or equal to END";

                case 1:
                    $exitCode = STATUS_CRITICAL;
                    $state = "CRITICAL";
                    $outputMsg = "";
            }
        }

        $thresholdResults = array('exitCode' => $exitCode, 'state' => $state, 'outputMsg' => $outputMsg);
        $logger->debug("Check Warning/Critical Thresholds: ".var_export($thresholdResults, true), __METHOD__, __LINE__);

        return $thresholdResults;
    }

    // Function to process the results
    // process_results makes $outputMsg - the Status & Performance data available and returns the exit code.
    # Is this really only Multiple Results???
    public static function processMultiResults($result, $mode) {
        global $logger;
        global $overrideErrors;
        $logger->debug("", __METHOD__, __LINE__);

        # No result - ERROR
        if (!array_key_exists('matchCount', $result) || $result['matchCount'] == 0) {
            nagios_exit("WARNING: No data for this ".$mode["filterLabel"]."!", STATUS_WARNING);
        }

        $state = "OK";
        $exitCode = STATUS_OK;
        $warning = self::$options['warning'];
        $critical = self::$options['critical'];

        # Number of warnings and criticals, needs to be the same and match fields that need to be monitored.
        $warningValues  = explode(',', $warning);
        $criticalValues = explode(',', $critical);

        $thresholdChecks = array();
        $results = "";
        $idx = 0;

        foreach ($mode['fields'] as $field => $isMonitored) {
            if (!$isMonitored) {
                $logger->debug("NOT Monitored! SKIPPING", __METHOD__, __LINE__);
                continue;
            }

            $warning = (array_key_exists($idx, $warningValues) ? $warningValues[$idx] : $warningValues[0]);
            $critical = (array_key_exists($idx, $criticalValues) ? $criticalValues[$idx] : $criticalValues[0]);
            $thresholdChecks[$field] = self::checkThreshold($warning, $critical, $result[0][$field]);

            if ($thresholdChecks[$field]['exitCode'] > $exitCode) {
                $state = $thresholdChecks[$field]['state'];
                $exitCode = $thresholdChecks[$field]['exitCode'];
            }

            $fieldName = preg_replace('/\s+/', '_', $field);
            $results .= $fieldName."=".$result[0][$field]." ";

            $idx++;
        }

        $mappedResults = trim($results);
        
        # Multiple results
        if (self::$multiple) {
            $logger->info("MULTIPLE", __METHOD__, __LINE__);
            self::$outputMsg = strtr("@state: [@id] @mappedResults | @mappedResults;@warning;@critical;;".PHP_EOL, 
                                    array("@state" => $state, "@id" => self::$options['filter'], "@mappedResults" => $mappedResults, 
                                   "@warning" => self::$options['warning'], "@critical" => self::$options['critical']));
        # Single result
        # Is this necessary???
        } else {
            $logger->info("SINGLE", __METHOD__, __LINE__);
            self::$outputMsg = strtr("@state: @mappedResults | @mappedResults;@warning;@critical;;".PHP_EOL, 
                                    array("@state" => $state, "@mappedResults" => $mappedResults, 
                                   "@warning" => self::$options['warning'], "@critical" => self::$options['critical']));
        }

        $logger->info("outputMsg: [".self::$outputMsg."]", __METHOD__, __LINE__);
        $logger->debug("@state [$state] @stdout [".self::$stdout."] @label [".self::$label."] @result [$mappedResults] @warning [$warning] @critical [$critical]", __METHOD__, __LINE__);

        return $exitCode;
    }

    // Seems to return 0 for OK, 1 for warning/critical (depending on threshold), 3 for UNKNOWN.
    private static function check_nagios_threshold($threshold, $value) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $logger->debug("threshold [".$threshold."], value [".$value."]", __METHOD__, __LINE__);

        $inside = ((substr($threshold, 0, 1) == '@') ? true : false);
        $range = str_replace('@','', $threshold);
        $parts = explode(':', $range);
        
        if (count($parts) > 1) {
            $start = $parts[0];
            $end = $parts[1];

        } else {
            $start = 0;
            $end = $range;
        }
        
        if (substr($start, 0, 1) == "~") {
            $start = -999999999;
        }

        if ($end == "") {
            $end = 999999999;
        }

        $logger->debug("start [".$start."] > end [".$end."]", __METHOD__, __LINE__);

        if ($start > $end) {
            $logger->debug("STATUS_UNKNOWN [".STATUS_UNKNOWN."] start [".$start."] > end [".$end."]", __METHOD__, __LINE__);

            return STATUS_UNKNOWN;
        }
        
        if ($start <= $value && $value <= $end) {
            $logger->debug("inside", __METHOD__, __LINE__);

            return $inside;
        }

        $logger->debug("!inside", __METHOD__, __LINE__);

        return !$inside;
    }

    #
    # Point in Time should never need a modifier, so this is a bit overkill/confusion.
    #
    private function calculate_result() {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $logger->info("calculatedResult [".self::$calculatedResult."] modifier [".self::$modifier."]", __METHOD__, __LINE__);
        $calculate = self::$calculatedResult;

        if (!empty(self::$modifier)) {
            $calculate = floatval($calculate) * floatval(self::$modifier);
            $logger->debug("calculate [".$calculate."]", __METHOD__, __LINE__);
        }

        self::$calculatedResult = number_format($calculate, 1, '.', '');
        $logger->debug("calculatedResult [".self::$calculatedResult."]", __METHOD__, __LINE__);

        return STATUS_OK;
    }

    #
    # Round anything with decimal places > 3, to 4 significant figures, otherwise, ignore.
    # Prevent scientific notation in large numbers.
    # 
    function sigFig($value, $significantDigits) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

#        $value = ".000000000000000000000000000025677";
#        $value = "1234567349812384891.49872345";
        $decimalPart = ".".substr(strrchr($value, "."), 1);
        $decimalCount = (strpos($decimalPart, ".") == 0) ? strlen($decimalPart) - 1 : 0;
        $integerPart = strstr($value, ".", true);
        $integerCount = strlen($integerPart);
        $logger->debug("decimalPart [$decimalPart] decimalCount [$decimalCount] integerPart [$integerPart] integerCount [$integerCount]", __METHOD__, __LINE__);

        if ($decimalCount > 3) {
            if ($value == 0) {
                $decimalPlaces = $significantDigits - 1;
            } else if ($value < 1) {
                $decimalPlaces = $significantDigits - floor(log10($value)) - 1;
                #$decimalPlaces = $significantDigits;
            } elseif ($value < 0) {
                #$decimalPlaces = $significantDigits - floor(log10($value * -1)) - 1;
                $decimalPlaces = $significantDigits - floor(log10($value * -1)) - 2;
            } elseif ($value < 1000) {
                #$decimalPlaces = $significantDigits - floor(log10($value * -1)) - 1;
                $decimalPlaces = 3;
            } else {
                #$decimalPlaces = $significantDigits - floor(log10($value)) - 1;
                $decimalPlaces = 1;
            }

            if ($integerCount > 16) {
                # Avoid scientific notation...
                $roundedDecimals = floatval(round($decimalPart, $decimalCount));

                if ($roundedDecimals >= 1) {
                    $integerPart += 1;

                    if ($roundedDecimals == 1) {
                        $roundedDecimals = str_repeat("0", $decimalPlaces);
                    }
                } else {
                    $roundedDecimals = substr(strrchr($roundedDecimals, "."), 1, $decimalPlaces);
                }

                $answer = "$integerPart.$roundedDecimals";
            } else if ($decimalPlaces > 0) {
                $answer = number_format($value, $decimalPlaces, ".", "");
            } else {
                $answer = round($value, $decimalPlaces);
            }
        } else {
            $answer = $value;
        }

        return $answer;
    }

    #
    # Used by queryEndpoint() to collect the headers, and total header length, from the cURL HTTP Response.
    # 
    function handleHeaderLine($curl, $line) {
        global $logger;
#        $logger->debug("", __METHOD__, __LINE__);
#        $logger->debug("### handleHeaderLine ### [".$line."]", __METHOD__, __LINE__);

        self::$headerLength += strlen($line);    # Keep track so it can be stripped, later.

        if (is_null(self::$code) && preg_match('@^HTTP/\d\.\d (\d+) @', $line, $matches)) {
            self::$code = $matches[1];
        } else {
            # Remove the trailing newline
            $trimmed = rtrim($line);

            if (strlen($trimmed)) {
                # If this line begins with a space or tab, it is a continuation of the previous header.
                if (($trimmed[0] == ' ') || ($trimmed[0] == "\t")) {
                    # Collapse the leading whitespace into one space.
                    $trimmed = preg_replace('@^[ \t]+@', ' ', $trimmed);
                    self::$headers[count(self::$headers) - 1] .= $trimmed;

                # Otherwise, it is a new header.
                } else {
                    self::$headers[] = $trimmed;
                }
            }
        }

        return strlen($line);
    }

    /**
     * Request to one of the Microsoft API endpoints (Microsoft Graph, Azure, Outlook, etc.)
     *
     * {HTTP method} https://graph.microsoft.com/{version}/{resource}?{query-parameters}                      # Microsoft Graph
     * {HTTP method} https://management.azure.com/{resource_path}?{api_version}{query_parameters}             # Azure Management API
     * {HTTP method} https://graph.windows.net/{tenant_id}/{resource_path}?{api_version}{query_parameters}    # Azure AD Graph
     *
     * @param string $check     The resource for the test/check.
     * @param string $nextLink  The link to the next set of data.
     * @param string $depth     Recursion level, when nextLink in use.
     *
     */
    function queryEndpoint($check, $nextLink=null, $depth=0) {
        global $logger;
        global $http_status_codes;
        $logger->debug("", __METHOD__, __LINE__);

        $logger->debug("check [".$check."] depth [".$depth."] self::limit [".self::$limit."]", __METHOD__, __LINE__);
        $logger->debug("full url [".self::$urlBase.$check."]", __METHOD__, __LINE__);
        $logger->debug("nextLink [".$nextLink."] depth [".$depth."]", __METHOD__, __LINE__);

        $curlHandle = curl_init();

        $url = (!isset($nextLink) ? self::$urlBase.$check : $nextLink); # Link to the next set of data, for the query.

        # If we are bypassing, with test data, use the testurl.
        if (!empty(self::$testurl) && empty($nextLink)) {
            $url = self::$testurl;
            $logger->info("### BYPASSING - using testurl [".var_export($url, true)."]", __METHOD__, __LINE__);
        }

        $curlOptions = array (
            CURLOPT_URL => $url,                    # Link to the next set of data, for the query.
            CURLOPT_HTTPGET => 1,                   # For clarity, not required.
            CURLOPT_RETURNTRANSFER => true,         # Return response data, as a string.
            CURLOPT_HEADER => true,                 # Do NOT include the header in the response.
            CURLOPT_HTTPHEADER => self::$header,    # Array of HTTP header fields.
            #CURLINFO_HEADER_OUT => true,           # Track the handle's request string.
            CURLOPT_CONNECTTIMEOUT => 10,           # Wait up to 10 seconds, to connect.
            CURLOPT_TIMEOUT => 60,                  # Execute cURL request for up to 60 seconds.
        );

        $logger->debug("### curl options [".var_export($curlOptions, true)."]", __METHOD__, __LINE__);

        curl_setopt_array($curlHandle, $curlOptions);

        #curl_setopt($curlHandle, CURLOPT_HEADERFUNCTION, 'self::handleHeaderLine');   # Separates header from body. 
        #curl_setopt($curlHandle, CURLOPT_HEADERFUNCTION, 'handleHeaderLine');   # Separates header from body. 
        #curl_setopt($curlHandle, CURLOPT_HEADERFUNCTION, 'BaseRequest::handleHeaderLine');   # Separates header from body. 
        #curl_setopt($curlHandle, CURLOPT_HEADERFUNCTION, array(&$this, 'handleHeaderLine'));   # Separates header from body. 
        self::$headerLength = 0;    # Set/reset for each page.
        curl_setopt($curlHandle, CURLOPT_HEADERFUNCTION, array($this, 'handleHeaderLine'));   # Separates header from body. 

        // Request to Microsoft Graph
        $response = curl_exec($curlHandle);
        $logger->debug("Complete response: ".var_export($response, true), __METHOD__, __LINE__);
        self::$curlError = curl_errno($curlHandle);

        if (self::$curlError) {
            $logger->error("Error: ".self::$curlError, __METHOD__, __LINE__);
            $curlError = self::$curlError;

            if (version_compare(PHP_VERSION, '5.5.0') >= 0) {
                $curlError = curl_strerror(self::$curlError);
            }

            $processedResults = array_merge($response, array('error_code' => self::$curlError, "status_msg" => $curlError));

            return $processedResults;
        } 

        self::$curlError = curl_errno($curlHandle);     # Set again, after curl_exec()
        self::$curlInfo = curl_getinfo($curlHandle);
        self::$connectTime = round(self::$curlInfo['total_time'], 6);
        $logger->debug("curl: ".var_export(self::$curlInfo, true), __METHOD__, __LINE__);

        curl_close($curlHandle);

        # Error code returned by Graph or Server.
        self::$httpCode = self::$curlInfo['http_code'];

        $body = substr($response, self::$headerLength);
        $logger->debug("body ".var_export($body, true)."###\n", __METHOD__, __LINE__);
        $logger->debug("self::limit [".self::$limit."] ###\n", __METHOD__, __LINE__);

        # Check for \\\\" and reduce to \".
        $body = preg_replace('/\\\\\\"/', '"', $body);
        # Check for \\\' and reduce to \'.
        $body = preg_replace("/\\\\'/", "'", $body);
        # Check for '"{', which breaks json format - remove the ".
        $body = preg_replace("/\"\{/", "{", $body);
        # Check for '}"', which breaks json format - remove the ".
        $body = preg_replace("/\}\"/", "}", $body);

        $jsonResponse = json_decode($body, true);
        $logger->info("jsonReponse ".var_export($jsonResponse, true)."###\n", __METHOD__, __LINE__);

        if (self::$httpCode != 200 && self::$httpCode != 302) {
            # {"error":{"code":"S2SUnauthorized","message":"Invalid permission."}}
            if (isset($jsonResponse)) {
                self::$graphError = $jsonResponse;

                # If there is further error details, grab that, instead.
                if (array_key_exists('error', self::$graphError) && array_key_exists('message', self::$graphError['error'])) {
                    self::$graphError = self::$graphError['error']['message'];
                }

                $logger->debug("graphError ".var_export(self::$graphError, true)."###\n", __METHOD__, __LINE__);
            }

            $processedResults = array('http_code' => self::$httpCode, 'http_msg' => $http_status_codes[self::$httpCode], 'error_code' => self::$graphError['error']['code'], 'status_msg' => self::$graphError['error']['message'], 'total_time' => self::$connectTime);
            $logger->info("FATAL ERROR: ".var_export($processedResults, true), __METHOD__, __LINE__);

            return $processedResults;
        }

        # Check if there are more data pages to pull.
        if ((empty(self::$limit) || self::$limit > $depth + 1) && is_array($jsonResponse) && array_key_exists('@odata.nextLink', $jsonResponse)) {
            $logger->debug("nextLink FOUND", __METHOD__, __LINE__);
            # Used to recursively retrieve the complete data set.
            $nextLink = str_replace(" ", "+", urldecode($jsonResponse['@odata.nextLink']));
            $logger->debug("nextLink ".var_export($nextLink, true)."###\n", __METHOD__, __LINE__);

            $tmpResults = $this->queryEndpoint($check, $nextLink, ++$depth);
            self::$body['value'] = (self::$body == null) ? $tmpResults['value'] : array_merge($tmpResults['value'], self::$body['value']);

        # Last data page.
        } else if ($depth > 0 && !array_key_exists('@odata.nextLink', $jsonResponse)) {
            $logger->debug("nextLink LAST", __METHOD__, __LINE__);

            # The last set of data.
            return $jsonResponse;
        }
        
        # Used in processData()
        # Simple data request (no nextLink)
        # JSON String for Configwizard.
        if (self::$body == null) {
            $logger->debug("Simple Data Request", __METHOD__, __LINE__);
            self::$body['value'] = $jsonResponse['value'];
            self::$body = json_encode(self::$body);
        } else if ($depth == 1) {
            $logger->debug("Done with Recursion, depth == 1", __METHOD__, __LINE__);
            # Add the first set of data.
            self::$body['value'] = array_merge($jsonResponse['value'], self::$body['value']);
            self::$body = json_encode(self::$body);
        } else {
            $logger->debug("Recursion, depth [".$depth."]", __METHOD__, __LINE__);
        }

        $logger->debug("self::headers ".var_export(self::$headers, true)." ###\n", __METHOD__, __LINE__);
        $logger->debug("self::code ".var_export(self::$code, true)." ###\n", __METHOD__, __LINE__);
        $logger->debug("self::headerLength ".var_export(self::$headerLength, true)." ###\n", __METHOD__, __LINE__);
        $logger->debug("self::body ".var_export(self::$body, true)." ###\n", __METHOD__, __LINE__);

        return $jsonResponse;
    }

    /**
     * Request to one of the Microsoft API endpoints (Microsoft Graph, Azure, Outlook, etc.)
     *
     * {HTTP method} https://graph.microsoft.com/{version}/{resource}?{query-parameters}                      # Microsoft Graph
     * {HTTP method} https://management.azure.com/{resource_path}?{api_version}{query_parameters}             # Azure Management API
     * {HTTP method} https://graph.windows.net/{tenant_id}/{resource_path}?{api_version}{query_parameters}    # Azure AD Graph
     *
     * @return array $processedResults
     */
    function processData() {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        if (self::$body === false && self::$code != '302') {
# FIX
            $processedResults = array_merge(self::$body, array("error" => self::$code, "status_msg" => "Failed to retrieve access token", "total_time" => -1));

            return $processedResults;
        }

        self::$connectTime = round(self::$curlInfo['total_time'], 6);

        $jsonResponse = json_decode(self::$body, true);    # ['access_token'];

        $logger->debug("jsonResponse: ".var_export($jsonResponse, true), __METHOD__, __LINE__);

        # Simplest way to get rid of issues with single quotes.
        self::$body = preg_replace('/\\\'/', "&#039;", self::$body);

        # Handle the results...
        if (!array_key_exists('error', $jsonResponse)) {
            if (!empty($jsonResults)) {
                $processedResults = array_merge($jsonResponse, array("http_code" => self::$curlInfo['http_code'], "status_cd" => true, "status_msg" => "Success connecting to Microsoft Graph API", "total_time" => self::$connectTime));
            } else {
                nagios_exit("CRITICAL: Empty jsonResults!", STATUS_CRITICAL);
            }

        } else if (array_key_exists('error', $jsonResponse)) {
            $processedResults = array_merge($jsonResponse, array("http_code" => self::$curlInfo['http_code'], "status_cd" => true, "status_msg" => "Error connecting to Microsoft Graph API, code {".$jsonResponse['error']['code']."}, message {".$jsonResponse['error']['message']."}", "total_time" => self::$connectTime));
        } else {
            $processedResults = array("http_code" => self::$curlInfo['http_code'], "status_cd" => false, "status_msg" => "Failed to connect to Microsoft Graph API", "total_time" => self::$connectTime);
        }

        return $processedResults;
    }
}

class GraphRequest extends BaseRequest {
    const ENDPOINT = "https://graph.microsoft.com/";
    const VERSION = "v1.0";

    public function __construct($mode, $options) {
        global $logger;
        parent::__construct($mode, $options);

        $logger->debug("", __METHOD__, __LINE__);

        // Query MS Graph database/api.
        self::$urlBase = self::ENDPOINT.(array_key_exists('version', $mode) ? $mode['version'] : self::VERSION);

        $logger->debug("options (Array) stdout [".self::$stdout."] label [".self::$label."] unit [".self::$unit."] modifier [".self::$modifier."] url [".self::$urlBase."]", __METHOD__, __LINE__);
    }

    public function setRequestHeader($token) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        self::$token = $token;

        self::$header = array(
            'Host: graph.microsoft.com',
            'Accept: application/json',
            'Authorization: Bearer '.self::$token);
    }
}

class GraphReportRequest extends GraphRequest {
    protected static $reportAge = null;

    function processData($format='array') {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $filter = (array_key_exists('filter', self::$options) && !empty(self::$options['filter'])) ? self::$options['filter'] : "";

        # With reports, follow the url and download the report file, unless we already have a saved "unexpired" file.
        $redirectUrl = self::$curlInfo['redirect_url'];

        $logger->debug("### Call getReport url [".$redirectUrl."] filter [".$filter."] format [".$format."]", __METHOD__, __LINE__);
        $reportResults = self::getReport($redirectUrl, $filter, $format);
        $logger->debug("### reportResults from getReport() [".var_export($reportResults, true)."]", __METHOD__, __LINE__);

        $totalTime = $reportResults['total_time'];

        # "Plugin" & Wizard data, that needs to be filtered
        if (is_array($reportResults)) {
            if (!array_key_exists('error', $reportResults)) {
                if (!empty($reportResults)) {
                    # This is also for the Wizard lists.
                    if ($format == 'json') {
                        $data = $reportResults;

                        # Handle field limiting (filtering), i.e., only return required fields.
                        if (!empty(self::$fields)) {
                            $data = $reportResults;
                            $data = json_decode($data[0], true);
                            $filteredData = array();

                            foreach ($data['value'] as $record) {
                                foreach (self::$fields as $field => $keep) {
                                    $filteredData[] = array($field => $record[$field]);
                                }
                            }

                            $data = json_encode(array('value' => $filteredData));
                            $logger->debug("data ".var_export($data, true), __METHOD__, __LINE__);
                        } else {
                            $data = $reportResults[0];
                            $logger->debug("data ".var_export($data, true), __METHOD__, __LINE__);
                        }

                        $logger->debug("json result data ".var_export($data, true), __METHOD__, __LINE__);
                        $reportResults = array('0' => $data);
                    }

                    $processedResults = array_merge($reportResults, array("http_code" => self::$curlInfo['http_code'], "status_cd" => true, "status_msg" => "Success connecting to Microsoft Graph API", "total_time" => $totalTime));
                } else {
                    $processedResults = array_merge($reportResults, array("http_code" => self::$curlInfo['http_code'], "status_cd" => true, "status_msg" => "Success connecting to Microsoft Graph API", "total_time" => $totalTime));
                }
            }
        # "Wizard" data.
        } else {
            $processedResults = $reportResults;
        }

        $logger->debug("### return processedResults", __METHOD__, __LINE__);
        return $processedResults;
    }

    protected function save($filename, $data) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);
        $logger->info("fileName [".$filename."]", __METHOD__, __LINE__);
        $logger->debug("data [".$data."]", __METHOD__, __LINE__);

        try {
            $returnCode = file_put_contents($filename, $data);

        } catch (Exception $exception) {
            $logger->error("Failed to save data from last run! code [$exception->getCode()] message [$exception->getMessage()]", __METHOD__, __LINE__);
            $logger->debug($exception->getTraceAsString(), __METHOD__, __LINE__);
        }

        if ($returnCode === false) {
            $logger->error("FAILED TO CREATE/WRITE FILE", __METHOD__, __LINE__);
        }

        $logger->debug("file_put_contents: returnCode [$returnCode]", __METHOD__, __LINE__);
    }

    protected function getSavedReport($filename) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $fileHandle = fopen($filename, 'c+');
        $fileContents = file_get_contents($filename);

        fclose($fileHandle);

        if ($fileContents === FALSE) throw new Exception("Run file does not exist!");

        return file_get_contents($filename);
    }

    /**
     * Get the data from a redirect url.
     *
     * At some point requesting the format, application/json will be an option.  For now, the reports are CSV.
     *
     * @param string $reportUrl The special url to get the requested report.
     * @param array $filter     ['User Principal Name', 'lxxxx@nagios.com'] - the key is the field name to filter by.
     *
     * @return string $report
     */
    # FIX DOCUMENTATION ^ and function signature!!!
    function getReport($redirectUrl, $filter=null, $format=null) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $filename = self::getFileName();
        $logger->debug("filename [".$filename."]", __METHOD__, __LINE__);
        $logger->info("redirectUrl [".$redirectUrl."]", __METHOD__, __LINE__);

        $report = null;

        # Check if we need a new report, otherwise use the saved version.
        # If we are using test data, load here.
        if (empty(self::$testurl) && self::$isCurrent) {
            $logger->info("Retrieving saved report", __METHOD__, __LINE__);
            $report = self::getSavedReport($filename);
        } else {
            $logger->info("Retrieve NEW report", __METHOD__, __LINE__);
            $report = self::getNewReport($redirectUrl, $format, $filename);
        }

        $logger->debug("Report ".var_export($report, true), __METHOD__, __LINE__);
        $response = self::processReport($report);

        if (!is_array($response)) {
            # Generally for Wizard list
            if (self::$type != 'list' && $format == 'json') {
                return json_encode($response);
            }

            $processedResults = array_merge($response, array("status_cd" => true, "status_msg" => "Success connecting to Microsoft Graph API"));
            self::$response = $response;
        } else if (array_key_exists('error', $response)) {
            $processedResults = array_merge($response, array("status_cd" => true, "status_msg" => "Error connecting to Microsoft Graph API, code {".$response['error']['code']."}, message {".$response['error']['message']."}"));
        } else {
            $processedResults = array("status_cd" => false, "status_msg" => "Failed to connect to Microsoft Graph API");
        }

        $processedResults = array_merge($response, array("total_time" => self::$connectTime));

        $logger->debug("processedResults: ".var_export($processedResults, true), __METHOD__, __LINE__);
        return $processedResults;
    }

    /**
     * Get the data from a redirect url.
     *
     * At some point requesting the format, application/json will be an option.  For now, the reports are CSV.
     *
     * @param string $reportUrl The special url to get the requested report.
     * @param array $filter     ['User Principal Name', 'lxxxx@nagios.com'] - the key is the field name to filter by.
     *
     * @return string $report
     */
    function getNewReport($reportUrl, $format, $filename) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        $curlHandle = curl_init();

        $header = array(
            'Host: graph.microsoft.com',
            'Accept: application/json',
            'Authorization: Bearer '.self::$token);
        $logger->debug("### header [".var_export($header, true)."]", __METHOD__, __LINE__);
        $logger->info("### reportUrl [".var_export($reportUrl, true)."]", __METHOD__, __LINE__);

        $options = array (
            #CURLOPT_URL => $reportUrl.'?$format=application/json', # Beta only??
            CURLOPT_URL => $reportUrl,
            CURLOPT_HTTPGET => 1,           # For clarity, not required.
            CURLOPT_RETURNTRANSFER => true, # Return response data, as a string.
            CURLOPT_HEADER => false,        # Do NOT include the header in the response.
            #CURLOPT_HTTPHEADER => $header,  # Array of HTTP header fields.
            CURLINFO_HEADER_OUT => true,    # Track the handle's request string.
            CURLOPT_CONNECTTIMEOUT => 10,   # Wait up to 10 seconds, to connect.
            CURLOPT_TIMEOUT => 60,          # Execute cURL request for up to 60 seconds.
        );

        curl_setopt_array($curlHandle, $options);

        self::$curlInfo = curl_getinfo($curlHandle);
        self::$curlError = curl_errno($curlHandle);     # Set again, after curl_exec()
        $logger->debug("curl: ".var_export(self::$curlInfo, true), __METHOD__, __LINE__);

        if (curl_errno($curlHandle)) {
            $logger->error("Error: ".curl_errno($curlHandle), __METHOD__, __LINE__);
            $curlError = self::$curlError;

            if (version_compare(PHP_VERSION, '5.5.0') >= 0) {
                $curlError = curl_strerror(self::$curlError);
            }

            $processedResults = array_merge($response, array('error_code' => self::$curlError, "status_msg" => $curlError));

            return $processedResults;
        } 

        // Request to provided url.
        $csv = curl_exec($curlHandle);
        self::$curlError = curl_errno($curlHandle);     # Set again, after curl_exec()

        $logger->info("csv: ".var_export($csv, true), __METHOD__, __LINE__);

        if (curl_errno($curlHandle)) {
            $logger->error("Error: ".curl_errno($curlHandle), __METHOD__, __LINE__);
            $curlError = self::$curlError;

            if (version_compare(PHP_VERSION, '5.5.0') >= 0) {
                $curlError = curl_strerror(self::$curlError);
            }

            $processedResults = array_merge($csv, array('error_code' => self::$curlError, "status_msg" => $curlError));

            return $processedResults;
        } 

        self::$curlInfo = curl_getinfo($curlHandle);
        $logger->debug("curl: ".var_export(self::$curlInfo, true), __METHOD__, __LINE__);
        curl_close($curlHandle);

        self::$connectTime = round(self::$curlInfo['total_time'], 6);

        # Don't save, if this is test data.
        if (empty(self::$testurl)) {
            self::save($filename, $csv);
        }

        return $csv;
    }

    /**
     * Process the report and filter the results.
     *
     * At some point requesting the format, application/json will be an option.  For now, the reports are CSV.
     *
     * @param string $report    The special url to get the requested report.
     *
     * @return string $report
     */
    function processReport($report) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        ini_set('auto_detect_line_endings', TRUE);

        # Remove unnecessary unicode characters.  In this case \\ufeff
        $csv = preg_replace('/^[\pZ\p{Cc}\x{feff}]+ | [\pZ\p{Cc}\x{feff}]+$/ux', '', $report);

        $csvReport = str_getcsv($csv, "\n");            # Put each line into an array value.
        $header = str_getcsv(array_shift($csvReport));  # Pull the first array entry, as the header values.
        $rows = $csvReport;                             # The rest of the array entries (body).

        $report = null;

        $logger->debug("type [".self::$type."] filterIdx [".self::$filterIdx."] filter [".self::$filter."]", __METHOD__, __LINE__);

        $rowCount = 0;
        $matchCount = 0;

        # This goes through the data, until a match.  Slower the bigger the dataset.  Is this necessary?  Do we need to sort and search the data?
        # Perhaps put the data that is used, at the top or in separate file(s)?
        foreach($rows as $row) {
            $rowCount++;

            # Use the values of the first as the keys and the second for the values of the array.
            $logger->debug("Before adjusting data: row: ".var_export($row, true), __METHOD__, __LINE__);

            # Replace empty elements ',,', False and True by 0 and 1.
            $row = str_replace(',,', ',0,', $row);
            $row = str_replace(',,', ',0,', $row);
            $row = str_replace(',False', ',0', $row);
            $row = str_replace(',True', ',1', $row);

            $logger->info("row: ".var_export($row, true), __METHOD__, __LINE__);

            $combined = array_combine($header, str_getcsv($row));
            $logger->info("combined: ".var_export($combined, true), __METHOD__, __LINE__);

            # If there is a filter, skip everything that doesn't match, to limit the data...
            # Group needs to look at the Group Id, instead of the filter.
            if (self::$type == "group" && array_key_exists('Group Id', $combined) && $combined['Group Id'] != self::$filter) {
                $logger->debug("NO MATCHING GROUP FILTER!!! filter [".self::$filter."] != combined['Group Id'] [".$combined['Group Id']."]", __METHOD__, __LINE__);
                continue;

            } 
            if (self::$type != "group" && !empty(self::$filter) && $combined[self::$filterIdx] != self::$filter) {
                $logger->debug("NO MATCHING FILTER!!! filter [".self::$filter."] != combined[".self::$filterIdx."] [".$combined[self::$filterIdx]."]", __METHOD__, __LINE__);
                continue;
            }

            $report[] = $combined;

            # Matched, so stop looking.
            if (self::$type != 'list') {
                $logger->debug("### FILTER MATCH!!! filter [".self::$filter."] != combined[".self::$filterIdx."]".((!empty(self::$filterIdx) && array_key_exists(self::$filterIdx, $combined)) ? " [".$combined[self::$filterIdx]."]" : ""), __METHOD__, __LINE__);
                $matchCount++;
                break;
            } else {
                $json[] = $combined;
            }
        }

        # Because some of our "lists" are from different sources, than the reports...
        # In case there is no match...
        if (empty($report)) {
            $logger->debug("Empty report", __METHOD__, __LINE__);
            $emptyRow = array_combine($header, array_fill(0, count($header), 0));
            $report[] = $emptyRow;
            $report = array_merge($report, array('rowCount' => $rowCount, 'matchCount' => 0));
        } else {
            if (self::$type == 'list') {
                $jsonReport = json_encode($json);

                # Embed the json in an array, to match what is expected.
                $report = array("0" => '{"value":'.$jsonReport.'}');
            } else {
                $report = array_merge($report, array('rowCount' => $rowCount, 'matchCount' => $matchCount));
            }
        }

        $logger->debug("report: ".var_export($report, true), __METHOD__, __LINE__);
        return $report;
    }
}

class AzureManagementRequest extends BaseRequest {
    const ENDPOINT = "https://management.azure.com/";

    public function __construct($mode, $options) {
        global $logger;
        parent::__construct($mode, $options);

        $logger->debug("", __METHOD__, __LINE__);

        // Query Azure Management database/api.
        self::$urlBase = self::ENDPOINT;

        $logger->debug("### header [".var_export(self::$header, true)."]", __METHOD__, __LINE__);
        $logger->debug("token [".self::$token."] options (Array) stdout [".self::$stdout."] label [".self::$label."] unit [".self::$unit."] modifier [".self::$modifier."] url [".self::$urlBase."] header [".var_export(self::$header, true)."]", __METHOD__, __LINE__);
    }

    public function setRequestHeader($token) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        self::$token = $token;

        self::$header = array(
            #'Host: management.azure.com',
            'Accept: application/json',
            'Content-Type: application/json',
            'Authorization: Bearer '.self::$token);
        $logger->info("### header ".var_export(self::$header, true), __METHOD__, __LINE__);
    }
}

class AzureADRequest extends BaseRequest {
    const ENDPOINT = "https://graph.windows.net/";

    public function __construct($mode, $options) {
        global $logger;
        parent::__construct($mode, $options);

        $logger->debug("", __METHOD__, __LINE__);

        // Query Azure Management database/api.
        self::$urlBase = self::ENDPOINT.$options['tenant'];

        $logger->debug("### header [".var_export(self::$header, true)."]", __METHOD__, __LINE__);
        $logger->debug("token [".self::$token."] options (Array) stdout [".self::$stdout."] label [".self::$label."] unit [".self::$unit."] modifier [".self::$modifier."] url [".self::$urlBase."] header [".var_export(self::$header, true)."]", __METHOD__, __LINE__);
    }

    public function setRequestHeader($token) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        self::$token = $token;

        self::$header = array(
            'Host: graph.windows.net',
            'Content-Type: application/json',
            'Authorization: Bearer '.self::$token);
        $logger->info("### header ".var_export(self::$header, true), __METHOD__, __LINE__);
    }
}

class ManageOfficeRequest extends BaseRequest {
    const ENDPOINT = "https://manage.office.com/";

    public function __construct($mode, $options) {
        global $logger;
        parent::__construct($mode, $options);

        $logger->debug("", __METHOD__, __LINE__);

        // Query Office Management database/api.
        self::$urlBase = self::ENDPOINT.$options['tenant'];

        $logger->debug("### header [".var_export(self::$header, true)."]", __METHOD__, __LINE__);
        $logger->debug("token [".self::$token."] options (Array) stdout [".self::$stdout."] label [".self::$label."] unit [".self::$unit."] modifier [".self::$modifier."] url [".self::$urlBase."] header [".var_export(self::$header, true)."]", __METHOD__, __LINE__);
    }

    public function setRequestHeader($token) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        self::$token = $token;

        self::$header = array(
            #'Host: graph.windows.net',
            'Content-Type: application/json',
            'Authorization: Bearer '.self::$token);
        $logger->info("### header ".var_export(self::$header, true), __METHOD__, __LINE__);
    }
}

class RSSFeed extends BaseRequest {
    const ENDPOINT = "https://manage.office.com/";

    public function __construct($mode, $options) {
        global $logger;
        parent::__construct($mode, $options);

        $logger->debug("", __METHOD__, __LINE__);

        // Query Office Management database/api.
        self::$urlBase = self::ENDPOINT.$options['tenant'];

        $logger->debug("token [".self::$token."] options (Array) stdout [".self::$stdout."] label [".self::$label."] unit [".self::$unit."] modifier [".self::$modifier."] url [".self::$urlBase."] header [".var_export(self::$header, true)."]", __METHOD__, __LINE__);

        $rss = Feed::loadRss('https://phpfashion.com/feed/rss');
        #https://azurestatuscdn.azureedge.net/en-us/status/feed/
    }

    public function setRequestHeader($token) {
        global $logger;
        $logger->debug("", __METHOD__, __LINE__);

        self::$token = $token;

        self::$header = array(
            #'Host: graph.windows.net',
            'Content-Type: application/json',
            'Authorization: Bearer '.self::$token);
        $logger->info("### header ".var_export(self::$header, true), __METHOD__, __LINE__);
    }
}


##########################################################################################################################################
# Constants
##########################################################################################################################################

final class EndpointConstants {
    const GRAPH_SCOPE = "scope=https://graph.microsoft.com/.default";
    const GRAPH_API = "graph.microsoft.com";
    const GRAPH_REPORTS_API = "graph.microsoft.com/reports";  # This is just for the request factory.

    const MANAGEMENT_AZURE_RESOURCE = "resource=https://management.azure.com/.default";
    const MANAGEMENT_AZURE_API = "management.azure.com";

    #const AZURE_AD_RESOURCE = "resource=https://graph.windows.net/.default";
    #const AZURE_AD_RESOURCE = "resource=https://graph.windows.net";
    const AZURE_AD_RESOURCE = "resource=https%3A%2F%2Fgraph.windows.net%2F";
    const AZURE_AD_API = "graph.windows.com";
    
    const MANAGE_OFFICE_API = "manage.office.com";
    const MANAGE_OFFICE_RESOURCE = "resource=https://manage.office.com";
    const MANAGE_OFFICE_SCOPE = "scope=https://manage.office.com/.default";
        # resource=https%3A%2F%2Fmanage.office.com
        # client_id=a6099727-6b7b-482c-b509-1df309acc563
        # redirect_uri= http%3A%2F%2Fwww.mycompany.com%2Fmyapp%2F
        # client_secret={your_client_key}
        # grant_type=authorization_code
        # code= AAABAAAAvPM1KaPlrEqdFSB...
        #
        # grant_type=client_credentials
        # client_id=625bc9f6-3bf6-4b6d-94ba-e97cf07a22de
        # client_secret=qkDwDJlDfig2IpeuUZYKH1Wb8q1V0ju6sILxQQqhJ+s=
        # resource=https%3A%2F%2Fservice.contoso.com%2F

    const AUTH_TOKEN_V1_ENDPOINT = "https://login.microsoftonline.com/@tenant/oauth2/token";        # V1: Azure Activity Directory Endpoint
    const AUTH_TOKEN_V2_ENDPOINT = "https://login.microsoftonline.com/@tenant/oauth2/v2.0/token";   # V2: Microsoft Identity Platform Endpoint
}

########################################################################
# Basic three level verbosity logging.
# Overkill, but could be useful, later.
########################################################################

class Logger {
    #public const DEBUG     = 100;
    const DEBUG     = 100;
    const INFO      = 200;
    const NOTICE    = 250;
    const WARNING   = 300;
    const ERROR     = 400;
    const CRITICAL  = 500;
    const ALERT     = 550;
    const EMERGENCY = 600;

    protected static $levels = array(
        self::DEBUG     => 'DEBUG',
        self::INFO      => 'INFO',
        self::NOTICE    => 'NOTICE',
        self::WARNING   => 'WARNING',
        self::ERROR     => 'ERROR',
        self::CRITICAL  => 'CRITICAL',
        self::ALERT     => 'ALERT',
        self::EMERGENCY => 'EMERGENCY',
    );

    /**
     * @var string
     */
    private $logLevel;
    private $complex;
    private $name;

    public function __construct($channel = '') {
        $this->setName($channel);
        $this->setLogLevel(static::WARNING);
    }

    private function setName($name = '') {
        $this->name = $name;
    }

    public function getLogLevel() {
        return $this->logLevel;
    }

    private function setLogLevel($logLevel = '') {
        $this->logLevel = $logLevel;
    }

    public function complex() {
        $this->complex = true;
    }

    public function verbose() {
        $localLevels = static::$levels;
        $level = $this->getLogLevel();

        while(key($localLevels) !== null && key($localLevels) !== $level) {
            next($localLevels);
        }

        prev($localLevels);

        $level = key($localLevels);

        if (!empty($level)) {
            $this->setLogLevel($level);
        }
    }

    /**
     * Adds a log record at the DEBUG level.
     *
     * @param string $message The log message
     * @param string $caller The calling method (__METHOD__, __LINE__)
     * @param array  $context The log context
     */
    #public function debug($message, array $context = []): void {
    public function debug($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::DEBUG, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function info($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::INFO, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function notice($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::NOTICE, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function warning($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::WARNING, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function error($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::ERROR, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function critical($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {

            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::CRITICAL, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function alert($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::ALERT, (string) $message, (string) $caller, $context, $lineNumber);
    }

    public function emergency($message, $caller = "", $lineNumber = "", array $context = array()) {
        if (empty($caller)) {
            // Get the calling method.
            #$dbt = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS,2);    # DEBUG_BACKTRACE_IGNORE_ARGS unavailable before 5.3.6
            $dbt = debug_backtrace(true);    # PHP 5.3.6-
            $caller = isset($dbt[1]['function']) ? $dbt[1]['function'] : "";
        }

        $this->addRecord(static::EMERGENCY, (string) $message, (string) $caller, $context, $lineNumber);
    }

    /**
     * Adds a log record.
     *
     * @param  int    $level   The logging level
     * @param  string $message The log message
     * @param  string $caller The calling method (__METHOD__ or backtrace)
     * @param  array  $context The log context
     * @return bool   Whether the record has been processed
     */
    #public function addRecord(int $level, string $message, array $context = array()) bool {
    public function addRecord($level = 0, $message, $caller = "", $context = array(), $lineNumber = "") {
        // check if we need to handle this message.
        // Only proceed if the current message's level is equal to or greater than the minimum level to display.
        if ($level < $this->getLogLevel()) {
            return;
        }

        $record = array(
            'message' => $message,
            'context' => $context,
            'level' => $level,
            'level_name' => static::getLevelName($level),
            'channel' => $this->name,
            #'datetime' => new DateTimeImmutable($this->microsecondTimestamps, $this->timezone),
            #'datetime' => new DateTime("now", date_default_timezone_get()),
            'datetime' => microtime(true), #new DateTime("now", $dateTime->getTimezone()),
            'extra' => array(),
            'line' => $lineNumber,
        );

        #echo("[".$record['datetime']."] ".$record['channel'].".".$record['level'].": ".$record['message']." {".$record['context']."} [".$record['extra']."]".PHP_EOL);
        if ($this->complex) {
            echo("[".$this->udate("Ymd H:i:s.u T", $record['datetime'])."] ".(($record['channel']) ? $record['channel']."." : "").$record['level_name'].": ".((!empty($caller)) ? "[".$caller."] " : "").$record['message'].(!empty($record['line']) ? " (".$record['line'].")" : "").((!empty($record['context'][0])) ? " {".$record['context'][0]."}" : " []").((!empty($record['extra'])) ? " [".$record['extra']."]" : " []").PHP_EOL);flush();
    
        // Simple
        } else {
            echo($record['level_name'].": ".((!empty($caller)) ? "[".$caller."] " : "").$record['message'].(!empty($record['line']) ? " (".$record['line'].")" : "").PHP_EOL);flush();
        }

        return true;
    }

    /**
     * Gets the name of the logging level.
     *
     * @throws \Psr\Log\InvalidArgumentException If level is not defined
     */
    #public static function getLevelName(int $level): string {
    public static function getLevelName($level) {
        if (!isset(static::$levels[$level])) {
            throw new InvalidArgumentException('Level "'.$level.'" is not defined, use one of: '.implode(', ', array_keys(static::$levels)));
        }

        return static::$levels[$level];
    }

    private static function udate($format = 'u', $utimestamp = null) {
        if (is_null($utimestamp))
            $utimestamp = microtime(true);

        $timestamp = floor($utimestamp);
        $milliseconds = round(($utimestamp - $timestamp) * 1000000);

        return date(preg_replace('`(?<!\\\\)u`', $milliseconds, $format), $timestamp);
    }
}   // Logger Class

########################################################################
# Misc. helper functions
########################################################################

/** Echo a message and exit with a status. */
function nagios_exit($message, $status=0) {
    global $logger;
    global $overrideErrors;
    $logger->debug("", __METHOD__, __LINE__);

    echo($message);flush();

    # Make sure the $status is a number.
    if (!is_int($status)) {
        $logger->critical("##### EXIT STATUS CODE MUST BE NUMERIC!!!! [$status]", __METHOD__, __LINE__);
    }

    # Make sure there is an EOL.
    if (!strstr($message, PHP_EOL)) {
        echo PHP_EOL;
    }

    # This is for running multiple tests, so all the tests run, even with failures.
    if ($overrideErrors) {
        return;
    }

    exit($status);
}

########################################################################
# main() - Begin...
########################################################################

main();

?>
