#!/usr/bin/env python3
import sys
from sys import argv
from subprocess import Popen
from subprocess import PIPE
import argparse

argument_parser = argparse.ArgumentParser()

argument_parser.add_argument( '-H',
                     '--host',
                     required=True,
                     type=str,
                     help='The host you wish to run a plugin against.')

argument_parser.add_argument( '-u',
                     '--user',
                     required=True,
                     type=str,
                     help='Username for connecting to the remote system.')


argument_parser.add_argument( '-a',
                     '--args',
                     required=False,
                     type=str,
                     help='Arguments to be sent to the plugin. e.g. -warning 80 -critical 90'
)

arguments = argument_parser.parse_args(argv[1:])
plugin_code = """

<#
.DESCRIPTION
A PowerShell based plugin for Nagios and Nagios-like systems. This plugin checks various metrics related to volumes on Windows systems. Typically this is the plugin people are looking for when they want to know the space usage of a specific drive letter in Windows. The rest of the world calls them logical volumes, but even in Windows, they are different from physical disks.
Remember, thresholds must be breached before they are thrown.
E.g. numwarning 10 will need the number of files to be 11 or higher to throw a WARNING.
.SYNOPSIS
This plugin checks various metrics related to volumes on Windows systems.
.NOTES
This plugin checks various metrics related to volumes on Windows systems. Typically this is the plugin people are looking for when they want to know the space usage of a specific drive letter in Windows.
.PARAMETER volumename
Default is C:\. Important to note that currently this does not work on NTFS mount points...yet.
.PARAMETER outputType
MB, GB, TB, %. How do you want the output reported? Megabytes, Gigabytes, Terabytes, Percent.
.PARAMETER metric
Used or Available. Do you want to see the space used or the space available?
.PARAMETER warning
The amount of reported space before you throw a WARNING.
.PARAMETER critical
The amount of reported space before you throw a CRITICAL.
.EXAMPLE
PS> .\check_volume.ps1 -volumename C:\ -outputType GB -metric Used
.EXAMPLE
PS> .\check_volume.ps1 -volumename D:\ -outputType GB -metric Available -warning 40 -critical 20
#>
param(
    [Parameter(Mandatory=$false)][string]$volumename = 'C:\',
    [Parameter(Mandatory=$false)][ValidateSet('MB', 'GB', 'TB', '%')][string]$outputType = '%',
    [parameter(Mandatory=$false)][ValidateSet('Used', 'Available')][string]$metric = 'Used',

    [Parameter(Mandatory=$false)][int]$warning = $null,
    [Parameter(Mandatory=$false)][int]$critical = $null
)


$message = "Nothing changed the status output!"
$exitcode = 3

function processCheck {
    param (
        [Parameter(Mandatory=$true)][int]$checkResult,
        [Parameter(Mandatory=$true)][int]$warningThresh,
        [Parameter(Mandatory=$true)][int]$criticalThresh,
        [Parameter(Mandatory=$false)][string]$returnMessage
    )

    [array]$returnArray

    if ((!$criticalThresh) -and (!$warningThresh) ) {

        $returnArray = @(0, "OK: $returnMessage")
    }
    elseif ($checkResult -gt $criticalThresh) {

        $returnArray = @(2, "CRITICAL: $returnMessage")
    }
    elseif ($checkResult -le $criticalThresh -and $checkResult -gt $warningThresh) {

        $returnArray = @(1, "WARNING: $returnMessage")
    }
    else {

        $returnArray = @(0, "OK: $returnMessage")
    }

    return $returnArray

}

if (!$volumename.EndsWith('\')) {
    $volumename = $volumename + "\"
}

$volumeresult = 0
$volume = (Get-WmiObject -Class Win32_Volume | where -Property DriveType -in -value 3,6 | where -Property Name -eq -value "$volumename")

if ($metric -eq "Used") {
    $volumeresult = $volume.Capacity - $volume.FreeSpace
    }
else {
    $volumeresult = $volume.FreeSpace
}


switch ($outputType) {
    'MB' {
            $volumeresult = [math]::Round($volumeresult / 1024 / 1024)
            }
    'GB' {
            $volumeresult = [math]::Round($volumeresult / 1024 / 1024 / 1024)
            }
    'TB' {
            $volumeresult = [math]::Round($volumeresult / 1024 / 1024 / 1024 / 1024)
            }
    '%'{
            $volumeresult = [math]::Round(($volume.FreeSpace / $volume.Capacity) * 100, 2)
            }
}

if ($volumename.Length -le 3) {
    $volumename = $volumename.Substring(0,1)
}
else {
    $volumename = $volumename.Substring(3, ($volumename.Length - 3))
    $volumename = $volumename.Substring(0, ($volumename.Length - 1))
}

$processArray = processCheck -checkResult $volumeresult `
                             -warningThresh $warning `
                             -criticalThresh $critical `
                             -returnMessage "$volumename $metric space is $volumeresult $outputType | '$volumename $metric Space'=$volumeresult$outputType;$warning;$critical"
$exitcode = $processArray[1]
$exitMessage = $processArray[2]

write-host "$exitMessage,$exitcode"

exit $exitcode

"""

echo_process = ""
arguments_length = 0
if arguments.args is not None:
    echo_process = Popen(["echo", "function checkplugin {\n", plugin_code, " }\n", "checkplugin ", arguments.args, "\n"], stdout=PIPE)
    arguments_length = len(arguments.args) + 1
else:
    echo_process = Popen(["echo", "function checkplugin {\n", plugin_code, " }\n", "checkplugin  \n"], stdout=PIPE)


ssh_process = Popen(["ssh", "-T", "-l", arguments.user, arguments.host, "powershell.exe"], stdin=echo_process.stdout, stdout=PIPE)
echo_process.stdout.close()
process_output = [ssh_process.communicate(), ssh_process.returncode]

decoded_stdout = process_output[0][0].decode()

if(process_output[1] == 255):
    print("CRITICAL: Connection to host failed. Check that the nagios user can passwordlessly connect to the host.")
    sys.exit(2)     

output_substring = decoded_stdout[(decoded_stdout.find("checkplugin  ") + 18 + arguments_length):(len(decoded_stdout) - 1)].rstrip()
split_output_substring = output_substring.split(',')

exit_status_code = int(split_output_substring[-1])
exit_message = ','.join(split_output_substring[:-1]) 
print(exit_message)
sys.exit(exit_status_code)