#!/usr/bin/env php
<?php

/*****************************************************************************************************************************************
 *
 * openai_check.php - A /Nagios plugin to check various OpenAI usage metrics
 *
 * Copyright (c) 2023 Nagios Enterprises, LLC (Phred White pwhite@nagios.com>)
 *
 * Notes:
 *
 *   This plugin has accessors for all important OpenAI administrative endpoints, and some utilities
 *   for aggregating data from multiple requests.
 *
 *   v1.0.0 only provides the metric 'total_cost' which provides the cost in dollars for a given time period.
 *   It allows you to set warning and critical thresholds.
 *
 *   Things to solve:
 *   Most other metrics require multiple requests, persisting of data, and aggregation. This is tricky because
 *   the API is limited to 5 resuests per minute. If you want to get 30 days of token usage, you need 6 minutes
 *   just to get the data.
 *
 *
 * License Information:
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *****************************************************************************************************************************************/

$PROGRAM = 'openai_check.php';
define("PROGRAM", $PROGRAM);
$VERSION = '1.0.0';
define("VERSION", $VERSION);
define("STATUS_OK", 0);
define("STATUS_WARNING", 1);
define("STATUS_CRITICAL", 2);
define("STATUS_UNKNOWN", 3);
define("DEBUG", false);


// Specify arguments
// All argument values will be accessible via their long name regardless of which format is used in the request.
// Params are: $short, $long, $default, $required, $help, $example, $type (string - requires a value, or flag - no value alowed)
// $help and $example are used to build the usage info.

$args = new Args();

$args->addArg(new ARG('k', 'api_key', '', true, 'Your OpenAI API Key', 'sk-f6FUpWVtGPQgAaykm...', 'string'));
$args->addArg(new ARG('m', 'metric', '', true, 'The OpenAI metric you want to monitor', 'total_cost', 'string'));
$args->addArg(new ARG('s', 'sess_tkn', '', true, 'The session token from your browser', 'sess-f6FUpWVtGPQgA...', 'string'));
$args->addArg(new ARG('o', 'org_id', '', true, 'Your OpenAI organization ID', 'org-POADNV...', 'string'));
$args->addArg(new ARG('t', 'period', '', false, 'The period for which you want to aggregate data in unit or n_unit format (e.g. 4_week)', '5_day, week, 2_month', 'string'));
$args->addArg(new ARG('c', 'crit', '', true, 'Critical threshold value', '100', 'string'));
$args->addArg(new ARG('w', 'warn', '', true, 'Warning threshold value', '90', 'string'));

// Additional notes for usage output
$args->usage_notes = <<<USAGE_NOTES
Note: All arguments are required. Currently, the only metric supported is total cost for a specific period of time.
period should be day, week or month, which means the total cost for the preceding day, month or week will be used.

IMPORTANT! The maximum timeperiod is 100 days, which is a little over 3 months, or 14 weeks.

USAGE_NOTES;

// Process the arguments.
// If there are errors, they will be output along with usage block, and the execution will end.
// If successfull, the values will be available in the $args->opts array via the long name
if (!$args->makeOpts()) {
    return;
}


/************************************************************************
 * Process metrics and return response
 ************************************************************************/
switch ($args->opts['metric']) {
    case 'total_cost':

        // Create an openAI accessor object
        $oai = new openAI('', $args->opts['api_key'], $args->opts['org_id'], $args->opts['sess_tkn'], true, 'v1', DEBUG);

        // Get metric
        $total_cost = $oai->getTotalCost($args->opts['period']);

        // Convert from cents to dollars and round
        $total_cost = round($total_cost/100,2);

        // Return response
        $crit = $args->opts['crit'];
        $warn = $args->opts['warn'];
        $period = $args->opts['period'];
        if ($total_cost > $crit) {
            nagios_exit("CRITICAL - OpenAI total cost, \$$total_cost, has exceeded \$$crit for the specified period ($period).\n\n", STATUS_CRITICAL);
        } elseif ($total_cost > $warn) {
            nagios_exit("WARNING - OpenAI total cost, \$$total_cost, has exceeded \$$warn for the specified period ($period).\n\n", STATUS_WARNING);
        } else {
            nagios_exit("OK - OpenAI total cost, \$$total_cost, is below thresholds for the specified period ($period).\n\n", STATUS_OK);
        }
        break;


    case 'requests':
    case 'context_tokens':
    case 'generated_tokens':

        // Create an openAI accessor object
        $oai = new openAI('', $args->opts['api_key'], $args->opts['org_id'], $args->opts['sess_tkn'], true, 'v1', DEBUG);

        // Convert plugin metric name to OpenAI name
        $oia_metrics = ['requests' => 'n_requests',
            'context_tokens' => 'n_context_tokens_total',
            'gnerated' => 'n_generated',
            'generated_tokens' => 'n_generated_tokens_total'
        ];

        $metric_pretty = str_replace('_', ' ', $args->opts['metric']);
        $oia_metric = $oia_metrics[$args->opts['metric']];

        // Get metric
        $sums = $oai->getDailyTokens();
        $metric_total = $sums[$oia_metric];

        // Return response
        $crit = $args->opts['crit'];
        $warn = $args->opts['warn'];
        if ($metric_total > $crit) {
            nagios_exit("CRITICAL - OpenAI $metric_pretty, $metric_total, has exceeded $crit for the specified period (today).\n\n", STATUS_CRITICAL);
        } elseif ($metric_total > $warn) {
            nagios_exit("WARNING - OpenAI $metric_pretty, $metric_total, has exceeded $warn for the specified period (today).\n\n", STATUS_WARNING);
        } else {
            nagios_exit("OK - OpenAI $metric_pretty, $metric_total, is below thresholds for the specified period (today).\n\n", STATUS_OK);
        }
        break;

    default:
        nagios_exit("Error: {$args->opts['metric']} is not a valid metric.", STATUS_UNKNOWN);

        break;
}

/************************************************************************
 *
 *
 * OpenAI and Request handling classes
 *
 *
 ************************************************************************/


/************************************************************************
 * OpenAI Class
 ************************************************************************/
class openAI {
    public $base_url   = "https://api.openai.com";
    public $api_token  = "";
    public $org_id     = "";
    public $sess_token = "";
    public $uglify     = true;
    public $api_v      = 'v1';
    public $debug      = false;
    public $throttle   = 15; // Seconds to wait before next rqst (max of 5/min)
    public $error       = '';
    public $endpts;

    public function __construct($base_url = '', $api_token = '', $org_id = '', $sess_token = '', $uglify = true, $api_v = '', $debug = '') {
        // Override defaults
        $this->base_url = ($base_url) ? $base_url : $this->base_url;
        $this->api_token = ($api_token) ? $api_token : $this->api_token;
        $this->org_id = ($org_id) ? $org_id : $this->org_id;
        $this->sess_token = ($sess_token) ? $sess_token : $this->sess_token;
        $this->uglify = ($uglify) ? $uglify : $this->uglify;
        $this->api_v = ($api_v) ? $api_v : $this->api_v;
        $this->debug = ($debug) ? $debug : $this->debug;

        $this->setEndPoints();
    }

    /************************************************************************
     * Value methods
     ************************************************************************/

    /**
     * Returns the total cost in cents ($0.01) for a given tim eperiod not to exceed 100 days.
     *
     * @param string $period - number of units _ time unit, e.g. 30_day, 11_week, 2_month.
     *
     * @return float - cost in cents.
     */
    public function getTotalCost($period) {

        // Get dates based on period
        list($startDate, $endDate) = $this->periodToDates($period);
        $days = $this->datesToDays($startDate, $endDate);

        if ($days > 100) {
            nagios_exit("Error: Period exceeds maximum of 100 hundred days ($days days).", STATUS_UNKNOWN);
        }

        // Make the OpenAI request
        $response = $this->getCostForPeriod($startDate, $endDate);
        if (! $response) {
            nagios_exit($this->error, STATUS_UNKNOWN);
        }
        $response = json_decode($response, true);

        return $response['total_usage'];
    }

    /**
     * Returns the total tokens and requests for a date, default is today, not to exceed 100 days n the past.
     *
     * @param string/null $date - in YYYY-MM-DD format.
     *
     * @return array - Total usage for 'n_requests', 'n_context_tokens_total', 'n_generated', 'n_generated_tokens_total'.
     */
    public function getDailyTokens($date = '') {
        $dateTime = new DateTime();
        $today = $dateTime->format("Y-m-d");
        $date = ($date) ? $date : $today;
        $cols = ['n_requests', 'n_context_tokens_total', 'n_generated', 'n_generated_tokens_total'];

        // Make the OpenAI request
        $response = $this->getUsageForDate($date);
        if (! $response) {
            nagios_exit($this->error, STATUS_UNKNOWN);
        }
        $response = json_decode($response, true);

        foreach ($cols as $col) {
            $sums[$col] = 0;
        }

        if ($response['data']) {
            $sums = $this->sumColumns($response['data'], $cols);
        }
        return $sums;
    }


    /************************************************************************
     * End point request methods
     ************************************************************************/

    public function getUsageForDate($date) {
        return $this->doGet($this->endpts->UsageForDate . "?date=$date");
    }

    public function getUsers() {
        return $this->doGet($this->endpts->Users);
    }

    public function getCostForPeriod($startdate, $enddate) {
        return $this->doGet($this->endpts->CostForPeriod . "?start_date=$startdate&end_date=$enddate");
    }

    public function getSubscription() {
        return $this->doGet($this->endpts->Subscription);
    }

    public function getCreditGrants() {
        return $this->doGet($this->endpts->CreditGrants);
    }

    public function getRateLimits() {
        return $this->doGet($this->endpts->RateLimits);
    }

    public function getFeatures() {
        return $this->doGet($this->endpts->Features);
    }

    public function getBilling() {
        return $this->doGet($this->endpts->Billing);
    }

    // This also can be used to check API token
    public function testChat($chat_data = '') {
        if (!$chat_data) {
            $chat_data = '{
                 "model": "gpt-3.5-turbo",
                 "messages": [{"role": "user", "content": "Say this is a test!"}],
                 "temperature": 0.7
            }';
        }
        return $this->doPost($this->endpts->Chat, $chat_data);
    }

    /************************************************************************
     * Raw Request methods
     ************************************************************************/

    public function doGet($endpoint) {
        $rq = new apiRequest($this->base_url, $this->org_id, $this->debug);
        $token =  $this->getToken($endpoint);
        $response = $rq->doRequest($token, 'GET', $endpoint, false, $this->uglify);

        if (!$rq->error) {
            return $response;
        } else {
            $this->error = $this->formatError($rq);
            return false;
        }
    }

    public function doPost($endpoint, $data, $use_org_id = false) {
        $rq = new apiRequest("{$this->base_url}/$endpoint", $this->org_id, $this->debug);
        $token =  $this->getToken($endpoint);
        $response = $rq->doRequest($token, 'POST', $data, $use_org_id, $this->uglify);

        if (!$rq->error) {
            return $response;
        } else {
            $this->error = $this->formatError($rq);
            return false;
        }
    }

    /************************************************************************
     * Utility methods
     ************************************************************************/

    public function setEndPoints() {
        // Define endpoints - having endppoints as properties really helps with testing
        // Call this whenever org_id is set
        $this->endpts = (object) [
            "UsageForDate"  => "{$this->api_v}/usage",
            "Users"         => "{$this->api_v}/organizations/{$this->org_id}/users",
            "CostForPeriod" => "dashboard/billing/usage",
            "Subscription"  => "dashboard/billing/subscription",
            "Billing"       => "account/billing/overview",
            "CreditGrants"  => "dashboard/billing/credit_grants",
            "RateLimits"    => "dashboard/rate_limits",
            "Features"      => "dashboard/organizations/{$this->org_id}/features",
            "Chat"          => "{$this->api_v}/chat/completions",
        ];
    }

    /**
     * Returns the token required for categories of endpoints..
     *
     * @param string $endpoint - URL for endpoint.
     * @return string - token to be used for this endpoint.
     */
    public function getToken($endpoint) {
        $token = strpos($endpoint, 'v1/') !== false ? $this->api_token : $this->sess_token;
        return $token;
    }

    /**
     * Converts period string (e.g. '12_week') into dates
     *
     * @param string   $period -  (4_week, 30_day).
     * @return array           - e.g., [$startDate, $endDate].
     */
    public function periodToDates($period) {
        // Get dates based on period (4_week, 30_day)
        list($cnt, $unit) = $this->parsePeriod($period);

        $dateTime = new DateTime();
        $endDate = $dateTime->format("Y-m-d");
        $dateTime->modify("-$cnt $unit");
        $startDate = $dateTime->format("Y-m-d");
        return [$startDate, $endDate];
    }

    /**
     * Converts returns number of days betwen dates of format  2023-10-31
     *
     * @param string   $startDate
     * @param string   $endDate
     * @return integer           - number of days.
     */
    public function datesToDays($startDate, $endDate) {
        $days = date_diff(new DateTime($endDate), new DateTime($startDate))->days;
        return $days;
    }

    /**
     * Converts period string (e.g. '12_week') into separate values
     * and returns them in an array. E.g. 12_week --> [12, week]
     *
     * @param string   $period -  (4_week, 30_day).
     * @return array           - e.g., [$cnt, $unit].
     */

    public function parsePeriod($period) {
        // Get dates based on period (4_week, 30_day)
        if (strpos($period, '_')) {
            list($cnt, $unit) = explode('_', rtrim($period, 's'));
        } else {
            list($cnt, $unit) = [1, rtrim($period, 's')];
        }
        return [$cnt, $unit];
    }

    /**
     * Calls the provided function with the date for every day for the
     * interval provided, excluding end date.
     *
     * @param string $startdate - first day of period in yyyy-mm-dd format
     * @param string $enddate   - day after last day of period in yyyy-mm-dd format
     * @param string $fncn      - name of method to be called with date in yyyy-mm-dd format
     *                            as only parameter.
     *
     * @return array - array of results with index = date for result.
     */
    public function IterateOnDate($startdate, $enddate, $fncn) {
        $start = new DateTime($startdate);
        $end = new DateTime($enddate);
        $delta = $end->diff($start)->format("%a");

        $interval = DateInterval::createFromDateString('1 day');
        $period = new DatePeriod($start, $interval, $end);

        $results = [];
        foreach ($period as $dt) {
            $date = $dt->format("Y-m-d");
            echo "Requesting $fncn($date)...\n";
            $results[$date] = $this->$fncn($date);
            if ($delta > 4 && $date < $enddate) {
                sleep($this->throttle);
            }
        }
        return $results;
    }

    // Converts simple 2-d array into a table with rows of tab-delimited values and a header row consisting of the keys.
    // $table is passed by reference so that multiple arrays of the same type can be aggregated.
    public function array_to_table($a, &$table) {
        If (!$table) {
            $table = implode("\t", array_keys($a[0])) ."\n";
        }
        foreach($a as $row) {
            $row[array_keys($row)[1]] = date("Y-m-d h:i:s", $row[array_keys($row)[1]]);
            $table .= implode("\t", $row) ."\n";
        }
    }

    public function sumColumns($a, $cols) {
        $result = [];
        foreach ($cols as $col) {            
            $array_sum = 0;
            foreach($a as $array_vals){
                $array_sum += $array_vals[$col];
            }
            $result[$col] = $array_sum;
        }
        return $result;
    }

    // Format an error for display
    public function formatError($rq_obj, $debug = false) {
        $response = "\nError: {$rq_obj->error}\n";

        if ($debug) {
            $response .= "Url: {$rq_obj->url}\n";
            $response .= "Header: " . print_r($rq_obj->header, true) . "\n";
        }
        return $response;
    }
}

/************************************************************************
 * Request Class
 ************************************************************************/
class apiRequest {
    public $header = [];
    public $url;
    public $curl_obj;
    public $curl_timeout = 60; //Seconds
    public $curl_connect_timeout = 10; //Seconds
    public $org_id;
    public $error = '';
    public $response ='';
    public $response_info ='';
    public $debug;

    public function __construct($base_url, $org_id = '', $debug = false) {
        $this->url = $base_url;
        $this->org_id = $org_id;
        $this->debug = $debug;
    }

    // $data is either a uri param string or a json post
    public function doRequest($token, $verb = 'GET', $data='', $use_org_id = false, $uglify = false) {
        $this->makeHeader($token, $use_org_id);
        $this->error = '';

        switch ( $verb ) {
            case 'GET':
                $this->curl_obj = curl_init($this->url .= "/$data");
                break;

            case'POST':
                $this->curl_obj = curl_init($this->url);
                curl_setopt($this->curl_obj, CURLOPT_POST, true);
                curl_setopt($this->curl_obj, CURLOPT_POSTFIELDS, $data);
                break;

            default:
                echo "Warning: Unimplemented verb $verb. We only know how to do GET and POST";
        }

        curl_setopt($this->curl_obj, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($this->curl_obj, CURLOPT_CONNECTTIMEOUT, $this->curl_connect_timeout);
        curl_setopt($this->curl_obj, CURLOPT_TIMEOUT, $this->curl_timeout);
        curl_setopt($this->curl_obj, CURLOPT_HTTPHEADER, $this->header);

        $this->response = curl_exec($this->curl_obj);

        // If the curl call itself failed, capture error and return false
        if (! $this->response) {
            $this->error = curl_error($this->curl_obj);
            if ($this->debug) {
                echo "\nthis->error: ".print_r($this->error,true)."\n";
            }
            curl_close($this->curl_obj);
            return false;
        }

        $this->response_info = curl_getinfo($this->curl_obj);
        curl_close($this->curl_obj);

        if ($this->debug) {
            echo "\nthis->response: ".print_r($this->response,true)."\n";
            echo "\nthis->response_info: ".print_r($this->response_info,true)."\n";
        }

        // Check if reponse has HTTP error, or API error.
        if ($this->has_error()) {
            return false;
        }

        if ($uglify) {
            return $this->uglify($this->response) . PHP_EOL;
        }

        return $this->response . PHP_EOL;
    }

    /************************************************************************
     * Request Utility methods
     ************************************************************************/

    protected function makeHeader($token, $use_org_id = false) {
        $this->header = [
            "Content-Type: application/json",
            "Authorization: Bearer $token",
            "User-Agent: Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/16.0 Safari/605.1.15",
            "Accept: */*",
            "Connection: keep-alive",
            "Cache-Control: no-cache",
        ];

        if ($use_org_id) {
            $this->header[] = "OpenAI-Organization: $this->org_id";
        }

        if ($this->debug) {
            echo "\nthis->header: ".print_r($this->header,true)."\n";
        }
    }

    public function uglify($json) {
        return json_encode(json_decode($json));
    }

    /**
     * Checks response for error conditions. If an error is found, $this->error is set with message, and true is returned.
     * If an error is found, $this->error is set with message, and true is returned.
     *
     * @return bool - false if no error is found, otherwise true.
     */
    public function has_error() {

        // Check for HTTP errors
        if($this->response_info['http_code'] == '0') {
            $this->error = "The server at {$this->url} is not responding.";
            return true;
        }

        if (! $this->response) {
            $this->error = "The server at {$this->url} returned an empty response (HTTP Status: $this->response_info['http_code']).";
            return true;

        }

        // Check for Endpoint errors
        $response_array = json_decode($this->response, true);
        if (isset($response_array['error'])) {
            $this->error = "{$response_array['error']['message']} Error type: {$response_array['error']['type']}  (HTTP Status: {$this->response_info['http_code']})";
            return true;
        }

        if($this->response_info['http_code'] != '200') {
            $this->error = "The server at {$this->url} returned HTTP Status: {$this->response_info['http_code']}, but no error information was included.";
            return true;
        }

        return false;
    }
}


/************************************************************************
 *
 *
 * Command line argument processing classes
 *
 *
 ************************************************************************/

class Arg {
    public $short;
    public $long;
    public $required = true;
    public $default = '';
    public $help = '';
    public $example = '';
    public $type = 'string'; // Could be string, flag

    public function __construct($short, $long, $default = '', $required = false, $help = '', $example = '', $type = 'string') {
        $this->short = trim($short, '-');
        $this->long = trim(trim($long, '-'), '-');
        $this->default = $default;
        $this->required = $required;
        $this->help = $help;
        $this->example = $example;
        $this->type = $type;
    }
}

class Args {
    protected $definitions = [];
    public $short = '';
    public $long = [];
    public $required = [];
    public $usage_options = '';
    public $usage_example = '';
    public $usage_notes = '';
    public $opts = [];
    public $debug = false;

    public function __construct() {
        // TBD
    }

    public function addArg($arg) {
        if ($this->debug) {
            echo "\narg: ".print_r($arg,true)."\n";
        }
        if ($arg->short && strpos($this->short, $arg->short) !== false) {
            echo "{$arg['short']} is a duplicate argument";
            return;
        }
        if (in_array($arg->long, $this->long)) {
            echo "{$arg['long']} is a duplicate argument";
            return;
        }
        if ($arg->short) {
            $this->short .= $arg->short;
            $this->short .= ($arg->type!='flag') ? ':' : '';
        }
        if ($arg->long) {
            $long = $arg->long;
            $long .= ($arg->type!='flag') ? ':' : '';
            $this->long[] = $long;
        }
        if ($arg->required) {
            $this->required[] = ['short' => $arg->short, 'long' => $arg->long];
        }

        $this->definitions[] = $arg;

        // Reset in case args added after getHelp, makeOpts are called
        $this->help = '';
        $this->opts = '';
    }

    public function makeOpts() {
        if (empty($this->opts)) {
            $this->opts = getopt($this->short, $this->long);
        }

        if ($this->debug) {
            echo "\nthis->opts raw: ".print_r($this->opts,true)."\n";
        }

        $errors = '';
        foreach ($this->required as $rq) {
            if (
                ($rq['short'] && (!$rq['long'] || !$this->isOpt($rq['long'])) && !$this->isOpt($rq['short'])) ||
                ($rq['long'] && (!$rq['short'] || !$this->isOpt($rq['short'])) && !$this->isOpt($rq['long']))
            ) {
                $errors .= "\tError: ";
                $errors .= ($rq['short'] && $rq['long']) ? "Either " : "Parameter ";
                $errors .= $rq['short'] ? "-{$rq['short']} " : "";
                $errors .= ($rq['short'] && $rq['long']) ? "or " : "";
                $errors .= $rq['long'] ? "--{$rq['long']} " : "";
                $errors .= "is required!\n";
            }
        }
        if ($errors) {
            $errors = "\n*******************************************************\n$errors";
            $errors .= "*******************************************************\n";
            echo $errors;
            echo $this->getUsage();
            return false;
        }

        // Make all opts accessible by long name
        foreach ($this->definitions as $arg) {
            if ($this->isOpt($arg->short) && $arg->long) {
                $this->opts[$arg->long] = $this->opts[$arg->short];
            }
        }
        if ($this->debug) {
            echo "\nthis->opts2 final: ".print_r($this->opts,true)."\n";
        }
        return $this->opts;
    }


    // Build usage options section. E.g.
    //    -T           - Required: Description of parameter
    //    -S|--secret  - Optional: Description of parameter

    public function getUsageOptions() {
        $this->usage_options = "";
        if (!$this->usage_options) {
            foreach ($this->definitions as $def) {
                $this->usage_options .= "\t";
                $this->usage_options .= $def->short ? "-{$def->short}" : "";
                $this->usage_options .= $def->short && $def->long ? "|" : "";
                $this->usage_options .= $def->long ? "--{$def->long} " : "\t";
                $this->usage_options .= "\t\t- " . ($def->required ? "Required" : "Optional") . ": {$def->help}\n";
            }
        }
        return $this->usage_options;
    }

    // Build command line example/spec. E.g.
    //    "processargs.php  --appid <appid> -T <tenant> [-S | --secret <api_token>]"
    public function getUsageExample() {
        if (!$this->usage_example) {
            $this->usage_example .= '';
            foreach ($this->definitions as $def) {
                $example = '';
                $example .= $def->short ? "-{$def->short}" : "";
                $example .= $def->short && $def->long ? " | " : "";
                $example .= $def->long ? "--{$def->long}" : "";
                $example .= ($def->type != 'flag' && $def->example) ? " <{$def->example}>" : "";
                $this->usage_example .= " " . ($def->required ? "$example" : "[$example]") . "";
            }
        }
        return $this->usage_example;
    }
    public function getUsage() {
        $PROGRAM = PROGRAM;
        $example = $this->getUsageExample();
        $options = $this->getUsageOptions();
        $notes = $this->usage_notes;
        $usage = <<<USAGE

    Usage: {$PROGRAM} {$example}"

    Options:

    {$options}

    {$notes}

USAGE;

        return $usage;
    }

    public function isOpt($arg) {
        return in_array($arg, array_keys($this->opts));
    }
}


/************************************************************************
 * Plugin Utility methods
 ************************************************************************/

function nagios_exit($stdout='', $exitcode=0) {
    print($stdout);
    exit($exitcode);
}
