#!/usr/bin/perl
#
# Copyright (c) 2017 Mellanox Technologies. All rights reserved.
#
# This Software is licensed under one of the following licenses:
#
# 1) under the terms of the "Common Public License 1.0" a copy of which is
#	available from the Open Source Initiative, see
#	http://www.opensource.org/licenses/cpl.php.
#
# 2) under the terms of the "The BSD License" a copy of which is
#	available from the Open Source Initiative, see
#	http://www.opensource.org/licenses/bsd-license.php.
#
# 3) under the terms of the "GNU General Public License (GPL) Version 2" a
#	copy of which is available from the Open Source Initiative, see
#	http://www.opensource.org/licenses/gpl-license.php.
#
# Licensee has the right to choose one of the above licenses.
#
# Redistributions of source code must retain the above copyright
# notice and one of the license notices.
#
# Redistributions in binary form must reproduce both the above copyright
# notice, one of the license notices in the documentation
# and/or other materials provided with the distribution.


use strict;
use File::Basename;
use File::Path;
use File::Compare;
use File::Copy;
use File::Temp;
use Term::ANSIColor qw(:constants);
$ENV{"LANG"} = "C";

my $RC = 0;
my $PREREQUISIT = "172";
my $NO_HARDWARE = "171";
my $DEVICE_INI_MISSING = "2";

if ($<) {
	print RED "Only root can run $0", RESET "\n";
	exit $PREREQUISIT;
}
$| = 1;
my $WDIR	= dirname(`readlink -f $0`);
chdir $WDIR;
my $ERROR = "1";

my $TMPDIR  = '/tmp';
my $log = "";
my $hca_self_test = "/usr/bin/hca_self_test.ofed";
my $reset = 0;
my %fw_info = ();
my $arch = `uname -m`;
chomp $arch;
if ($arch =~ /i.86/) {
	$arch = "i686";
}
my $mlxfwmanager_tool = "";

my $firmware_directory = "$WDIR/firmware";
my $conf = "$WDIR/mlnx-fw-updater.conf";
my $force_firmware_update = 0;
my $quiet = 0;
my $verbose = 0;
my $disable_bf = 1;
my %exclude_pci = ();
# path to bin images dir to use instead of default bins included in mlxfwmanger
my $image_dir = "";
my $image_dir_given = 0;

sub which($)
{
	my $prog = shift;
	my @path = split /[:]/, $ENV{'PATH'};
	foreach my $dir (@path) {
		return "$dir/$prog" if (-x "$dir/$prog");
	}
	return "";
}

sub update_fw_version_in_hca_self_test
{
	my $dev = shift @_;
	my $fwver = shift @_;
	my @lines;
	open(FWCONF, "$hca_self_test");
	while(<FWCONF>) {
		push @lines, $_;
	}
	close FWCONF;
	open(FWCONF, ">$hca_self_test");
	foreach my $line (@lines) {
		chomp $line;
		if ($line =~ /^$dev/) {
			print FWCONF "$dev=v$fwver\n";
		} else {
			print FWCONF "$line\n";
		}
	}
	close FWCONF;
}

sub logMsg
{
	my $msg = shift @_;
	open (OUT, ">> $TMPDIR/tmplog");
	print OUT "$msg\n";
	close OUT;
}

sub printNlog
{
	my $msg = shift @_;
	print "$msg\n";
	logMsg "$msg";
}

sub printNlogYellow
{
	my $msg = shift @_;
	print YELLOW "$msg", RESET "\n";
	logMsg "$msg";
}

sub printNlogRED
{
	my $msg = shift @_;
	print RED "$msg", RESET "\n";
	logMsg "$msg";
}

sub logNexit
{
	my $rc = shift;

	# append RC to log file
	logMsg("EXIT_STATUS: $rc");

	# rename log file
	system("/bin/mv $TMPDIR/tmplog $log >/dev/null 2>&1");

	exit $rc;
}

#
# update FW on devices
#

my @SkipCards = (
	{pattern => 'virtual',		name => 'a Virtual Function'},
	{pattern => 'PCI bridge',	name => 'a PCI bridge'},
	{pattern => 'Innova-2',		name => 'an Innova-2 card'},
);

my @SkipUnsupportedCards = (
	{pattern => 'ConnectX-8',	name => 'a ConnectX-8 card'},
);

my @SkipBFCards = (
	{pattern => 'BlueField',	name => 'a BlueField card'},
);


sub load_device_firmware($$$) {
	my $ibdev = shift;
	my $psid = shift;
	my $cmd = shift;

	my ($tmp_fh, $tmp_fn) = File::Temp::tempfile();
	my $tmpdir = File::Temp::tempdir(CLEANUP => 1);
	$cmd = "-L $tmp_fn $cmd --sfx-extract-dir $tmpdir";
	my $cmd1 = "$mlxfwmanager_tool $cmd -d $ibdev";
	if (exists $fw_info{$psid}) {
		$cmd1 = "$fw_info{$psid} $cmd -d $ibdev";
	}
	print "Running: $cmd1\n" if $verbose;
	system("$cmd1");
	if ($? >> 8 or $? & 127) {
		$RC = 1;
	}
	close($tmp_fn);
	system("/bin/cat $tmp_fn >> $TMPDIR/tmplog 2>/dev/null");
	system("/bin/rm -f $tmp_fn >/dev/null 2>&1");
}

my %CEDAR_PSIDS = (
	'MT_0000000891' => 1,
	'LNV0000000064' => 1,
);

sub is_device_cedar($) {
	my $psid = undef;
	my $pci_id = shift;
	my $cmd = "$mlxfwmanager_tool -d $pci_id --query";
	my $handle;
	open $handle, "$cmd |" || return 0; # MFT fails or missing.
	while(<$handle>) {
		next unless /^\s*PSID:/;
		s/\s*PSID: *//;
		chomp;
		$psid = $_;
		last;
	}
	close($handle) || return 0; # MFT failed
	return 0 unless $psid;
	return 1 if exists $CEDAR_PSIDS{$psid};
	return 0;
}

sub check_skip_cards($$$@) {
	my $ibdev = shift;
	my $devDesc = shift;
	my $print_unsupported = shift;

	my $cond = $print_unsupported or $verbose;

	my $skip_device = 0;
	foreach my $skip (@_) {
		if ($devDesc =~ /$$skip{'pattern'}/i) {
			print "Skipping $$skip{'name'}: $ibdev\n" if $cond;
			$skip_device = 1;
			last;
		}
	}
	return $skip_device;
}

sub check_and_update_FW
{
	print "Attempting to perform Firmware update...\n" if not $quiet;

	my $cmd = "-y";
	if ($force_firmware_update) {
		$cmd .= " --force";
	}
	if ($quiet) {
		$cmd .= " -o /dev/null";
	}
	if ($image_dir_given) {
		$cmd .= " --image-dir $image_dir";
	}

	# Start MST to support Secure Boot environment that does not provides access to sysfs configuration space
	system("mst start > /dev/null 2>&1");

	# loop on Mellanox devices
	my %processedSN = ();
	my $founddevs = 0;
	my @devs_to_update = ();
	print "Running: lspci -d 15b3: -s.0 2>/dev/null | cut -d\" \" -f\"1\"\n" if $verbose;
	for my $ibdev ( `lspci -d 15b3: -s.0 2>/dev/null | cut -d" " -f"1"` ) {
		$founddevs = 1;
		chomp $ibdev;

		# Skip Virtual Functions
		my $devDesc = `lspci -s $ibdev 2>/dev/null`;
		my $devPci = `lspci -s $ibdev 2>/dev/null | cut -d " " -f1`;
		chomp $devPci;
		chomp $devDesc;
		next if (check_skip_cards $ibdev, $devDesc, 0, @SkipCards);
		next if (check_skip_cards $ibdev, $devDesc, 1, @SkipUnsupportedCards);
		if ($disable_bf) {
			next if (check_skip_cards $ibdev, $devDesc, 1, @SkipBFCards);
		}

		if (exists $exclude_pci{"$ibdev"}) {
			printNlogYellow ("Skipping Firmware update for device $ibdev as set in conf file...");
			next;
		}

		# Some devices can appear multiple times with different pci IDs.
		# Prevent burning same physical device more than once by looking at the Serial Numbers.

		my $devSN = `lspci -s $ibdev -vv 2>/dev/null | grep SN | head -1 | cut -d":" -f"2" 2>/dev/null | sed -e 's/\\s*//g'`;
		chomp $devSN;
		if ($devSN ne "") {
			if (not exists $processedSN{"$devSN"}) {
				$processedSN{"$devSN"} = 1;
				print "Processing device $ibdev (SN: $devSN) for first time...\n" if $verbose;
			} elsif ($processedSN{"$devSN"} eq "CEDAR") {
				# Third / fourth time. Skip a needless flint call
				print "Processing ceder device $ibdev (SN: $devSN) again...\n" if $verbose;
			} elsif (is_device_cedar $devPci) {
				$processedSN{"$devSN"} = "CEDAR";
				print "Processing ceder device $ibdev (SN: $devSN) for the second time...\n" if $verbose;
			} else {
				print "Skipping already processed device $ibdev (SN: $devSN)...\n" if $verbose;
				next;
			}
		}

		print "Running: $mlxfwmanager_tool --clear-semaphore -d $ibdev > /dev/null 2>&1\n" if $verbose;
		system("$mlxfwmanager_tool --clear-semaphore -d $ibdev > /dev/null 2>&1");
		print "running $mlxfwmanager_tool -d $ibdev --query 2>/dev/null | grep PSID: | awk '{print \$NF}'\n" if $verbose;
		my $psid = `$mlxfwmanager_tool -d $ibdev --query 2>/dev/null | grep PSID: | awk '{print \$NF}'`;
		chomp $psid;
		if (exists $fw_info{$psid} or $psid eq "" or $psid =~ /psid/i) {
			# we have FW for this device,
			# or we couldn't open the device so let it print the error message.
			push @devs_to_update, [$ibdev, $psid];
		} else {
			$RC = $DEVICE_INI_MISSING;
			# we don't have FW for this device
			if (not $image_dir_given) {
				printNlogYellow ("\nThe firmware for this device is not distributed inside Mellanox driver: $ibdev (PSID: $psid)");
				printNlogYellow ("To obtain firmware for this device, please contact your HW vendor.\n");
			} else {
				printNlogYellow ("\nThe firmware for device $ibdev (PSID: $psid) is not available in the provided image dir '$image_dir'.");
			}
		}
	}
	my @children = ();
	foreach (@devs_to_update) {
		my ($ibdev, $psid) = @$_;
		my $pid = fork();
		if ($pid == 0) {
			load_device_firmware($ibdev, $psid, $cmd);
			exit 0;
		}
		push @children, $pid;
		sleep(1);
	}
	foreach (@children) {
		wait
	}
	if (@devs_to_update) {
		# There are confusing messages about log files:
		print "Real log file: $log\n" unless ($quiet);
	}

	if (not $founddevs) {
		printNlog "No devices found!\n";
		logNexit $NO_HARDWARE;
	}

	if (`grep "Query failed" $TMPDIR/tmplog 2>/dev/null`) {
		$RC = 1;
	}
	if ($RC) {
		print RED "Failed to update Firmware.", RESET "\n";
		print RED "See $log", RESET "\n";
	}
	if (`grep -E "FW.*N/A" $TMPDIR/tmplog 2>/dev/null`) {
		$RC = $DEVICE_INI_MISSING;
	}
	if (`grep -E "Updating FW.*Done" $TMPDIR/tmplog 2>/dev/null`) {
		$reset = 1;
	}
}

sub check_if_bluefield() {
	if (which "bfver") {
		print "BlueField DPU, may load BlueField firmware.\n" if $verbose;
		$disable_bf = 0;
	}
}

# get list of available FW and PSIDs
sub init_fw_info
{
	my $flags = "";
	if ($image_dir_given) {
		$flags .= " --image-dir $image_dir";
	}

	check_if_bluefield();
	print("Initializing...\n");
	# Loop over all possible mlxfwmanager packages
	for my $fwpkg (`/bin/ls $firmware_directory/mlxfwmanager* `) {
		chomp $fwpkg;
		print "Scanning $fwpkg\n" if $verbose;
		# set mlxfwmanager_tool to the first one we see, it will be used as tool
		# that we don't care about its content
		if ($mlxfwmanager_tool eq "") {
			$mlxfwmanager_tool = $fwpkg;
		}

		my @content = `$fwpkg -l $flags 2>/dev/null`;
		foreach my $line ( @content ) {
			chomp $line;
			next if ($line !~ /FW/);
			my $fwver = "";
			if ($line =~ /.*\s([A-Za-z0-9_]+)\s*FW ([0-9.]+)\s.*/) {
				# save a path to the tool that provides FW for this PSID
				$fw_info{$1} = $fwpkg;
				$fwver = $2;
			}
			# update hca_self_test.ofed
			if (-f "$hca_self_test") {
				if ($line =~ /ConnectX-3/ and $line !~ /Pro/) {
					update_fw_version_in_hca_self_test("CX3_FW_NEEDED", $fwver);
				} elsif ($line =~ /ConnectX-3 Pro/) {
					update_fw_version_in_hca_self_test("CX3_PRO_FW_NEEDED", $fwver);
				} elsif ($line =~ /ConnectX-2/) {
					update_fw_version_in_hca_self_test("HERMON_FW_NEEDED", $fwver);
				} elsif ($line =~ /Connect-IB/) {
					update_fw_version_in_hca_self_test("CONNECTIB_FW_NEEDED", $fwver);
				} elsif ($line =~ /ConnectX-4 Lx/) {
					update_fw_version_in_hca_self_test("CONNECTX4LX_FW_NEEDED", $fwver);
				} elsif ($line =~ /ConnectX-4/) {
					update_fw_version_in_hca_self_test("CONNECTX4_FW_NEEDED", $fwver);
				} elsif ($line =~ /ConnectX-5/) {
					update_fw_version_in_hca_self_test("CONNECTX5_FW_NEEDED", $fwver);
				}
			}
		}
	}
}

sub read_conf
{
	if (not -f "$conf") {
		printNlog("-W- $conf doen't exist!");
		return;
	}

	open(CONF, "$conf");
	if (tell(CONF) == -1) {
		printNlogRED("-E- Can't open $conf for read!");
		logNexit 1;
	}
	while (<CONF>) {
		my $line = $_;
		chomp $line;
		next if ($line =~ /^#/);

		if ($line =~ /MLNX_EXCLUDE_DEVICES/) {
			$line =~ s/.*=//g;
			$line =~ s/"//g;
			for my $item (split(',', $line)) {
				$exclude_pci{"$item"} = 1;
				print "PCI $item will not be udpdated\n" if $verbose;
			}
		}
	}
	close(CONF);
}

sub update_pci_ids
{
	my $line = "";
	my $pci_ids_file;
	my $compressed = 0;

	if (-e "/usr/share/hwdata/pci.ids") {
		$pci_ids_file = "/usr/share/hwdata/pci.ids";
	} elsif (-e "/usr/share/misc/pci.ids") {
		$pci_ids_file = "/usr/share/misc/pci.ids";
	} elsif (-e "/usr/share/pci.ids") {
		$pci_ids_file = "/usr/share/pci.ids";
	} elsif (-e "/usr/share/hwdata/pci.ids.gz") {
		$pci_ids_file = "/usr/share/hwdata/pci.ids.gz";
		$compressed = 1;
	} elsif (-e "/usr/share/misc/pci.ids.gz") {
		$pci_ids_file = "/usr/share/misc/pci.ids.gz";
		$compressed = 1;
	} elsif (-e "/usr/share/pci.ids.gz") {
		$pci_ids_file = "/usr/share/pci.ids.gz";
		$compressed = 1;
	} else {
		print "Could not locate pci.ids file.\n";
		return;
	}

	if (not -w "$pci_ids_file") {
		print "$pci_ids_file is read-only.\n";
		return;
	}

	system("/bin/rm -rf ${pci_ids_file}.new ${pci_ids_file}.old >/dev/null 2>/dev/null");
	if ($compressed) {
		system("/bin/cp -f ${pci_ids_file} ${pci_ids_file}.old.gz");
		system("gzip -d ${pci_ids_file}.old.gz")
	} else {
		system("/bin/cp -f ${pci_ids_file} ${pci_ids_file}.old");
	}

	open(PCI_IDS, "${pci_ids_file}.old");
	open(PCI_IDS_NEW, ">${pci_ids_file}.new");
	while(<PCI_IDS>) {
		$line = $_;
		chomp $line;
		if ($line =~ /1017.*MT2.*Connect/) {
			print PCI_IDS_NEW "\t1017  MT27800 Family [ConnectX-5]\n";
		} elsif ($line =~ /1018.*MT2.*Connect/) {
			print PCI_IDS_NEW "\t1018  MT27800 Family [ConnectX-5 Virtual Function]\n";
		} elsif ($line =~ /1019.*MT2.*Connect/) {
			print PCI_IDS_NEW "\t1019  MT28800 Family [ConnectX-5 Ex]\n";
		} elsif ($line =~ /101a.*MT2.*Connect/) {
			print PCI_IDS_NEW "\t101a  MT28800 Family [ConnectX-5 Ex Virtual Function]\n";
		} else {
			print PCI_IDS_NEW "$line\n";
		}
	}
	close PCI_IDS;
	close PCI_IDS_NEW;
	if (compare("${pci_ids_file}.old", "${pci_ids_file}.new") != 0) {
		if ($compressed) {
			system("gzip ${pci_ids_file}.new");
			system("/bin/mv -f ${pci_ids_file}.new.gz ${pci_ids_file}");
		} else {
			system("/bin/mv -f ${pci_ids_file}.new ${pci_ids_file}");
		}
		print("Updated $pci_ids_file\n");
	}
	system("/bin/rm -rf ${pci_ids_file}.new ${pci_ids_file}.old >/dev/null 2>/dev/null");
}

sub usage
{
	print GREEN;
	print "\n Usage: $0 [OPTIONS]\n";

	print "\n Options";
	print "\n        --force-fw-update             Force firmware update";
	print "\n        --fw-dir                      Path to firmware directory with mlnxfwmanager files (Default: $firmware_directory)";
	print "\n        --fw-image-dir                Firmware images directory to use instead of default package content.";
	print "\n        --tmpdir                      Change tmp directory. (Default: $TMPDIR)";
	print "\n        --log                         Path to log file (Default: $TMPDIR/mlnx_fw_update.log)";
	print "\n        -v                            Verbose";
	print "\n        -q                            Set quiet - no messages will be printed";
	print "\n";
	print "\n Note: To exclude devices from the automatic FW update procedure upon system boot\n refer to the conf file: $conf ";
	print "\n";
	print RESET "\n\n";
}

## main
while ( $#ARGV >= 0 ) {
	my $cmd_flag = shift(@ARGV);

	if ( $cmd_flag eq "--force-fw-update" ) {
		$force_firmware_update = 1;
	} elsif ( $cmd_flag eq "--fw-dir" ) {
		$firmware_directory = shift(@ARGV);
	} elsif ( $cmd_flag eq "--fw-image-dir" ) {
		$image_dir = shift(@ARGV);
		$image_dir_given = 1;
	} elsif ( $cmd_flag eq "--tmpdir" ) {
		$TMPDIR = shift(@ARGV);
	} elsif ( $cmd_flag eq "--log" ) {
		$log = shift(@ARGV);
	} elsif ( $cmd_flag eq "--disable-bf" ) {
		$disable_bf = 1;
	} elsif ( $cmd_flag eq "--enable-sriov" ) {
		print YELLOW "WARNING: '--enable-sriov' flag is deprecated!", RESET "\n";
		print YELLOW "To configure SR-IOV, please use the 'mlxconfig' or 'mstconfig' utility.", RESET "\n\n";
	} elsif ( $cmd_flag eq "-q" ) {
		$quiet = 1;
	} elsif ( $cmd_flag eq "-v" ) {
		$verbose = 1;
	} else {
		&usage();
		logNexit 1;
	}
}

$log = "$TMPDIR/mlnx_fw_update.log" if ($log eq "");

if ($image_dir_given and not -e $image_dir) {
	printNlogRED("-E- Provided image-dir does not exist: '$image_dir'!");
	logNexit 1;
}

# set path to the mlxfwmanager
if (`/bin/ls $firmware_directory/mlxfwmanager* ` eq "") {
	print RED "Error: mlxfwmanager doesn't exist at $firmware_directory! Cannot perform firmware update.", RESET "\n";
	logNexit $DEVICE_INI_MISSING;
}

update_pci_ids();
read_conf();
init_fw_info();
check_and_update_FW();
print GREEN "Please reboot your system for the changes to take effect.", RESET "\n"  if ($reset and not $quiet);

logNexit $RC;
