/*
 * Copyright (c) 2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

/*
 * Abstract:
 *	Module for Link as a Service configuration for NVLink systems
 *
 * Author:
 *	Or Nechemia, NVIDIA
 *	Sasha Minchiu, NVIDIA
 */

#ifndef OSM_NVLINK_PRTN_H
#define OSM_NVLINK_PRTN_H

#include <iba/ib_types.h>
#include <complib/cl_types_osd.h>
#include <complib/cl_dispatcher.h>
#include <opensm/osm_log.h>
#include <opensm/osm_nvlink.h>
#include <opensm/osm_gpu.h>
#include <opensm/osm_partition.h>

#ifdef __cplusplus
#  define BEGIN_C_DECLS extern "C" {
#  define END_C_DECLS }
#else                           /* !__cplusplus */
#  define BEGIN_C_DECLS
#  define END_C_DECLS
#endif                          /* __cplusplus */

BEGIN_C_DECLS

#define OSM_NVL_PRTN_ZERO_PREV_RAIL_FILTER		TRUE
#define OSM_NVL_PRTN_SET_NEW_RAIL_FILTER		FALSE

static inline void osm_nvl_prtn_clear_gpu2pkey_db(osm_nvlink_mgr_t *p_mgr)
{
	cl_hashmap_remove_all(&p_mgr->gpu_to_pkey_map);
}

void osm_nvl_prtn_destroy_prtn(osm_nvlink_mgr_t * p_mgr, osm_prtn_t * p_prtn);

osm_prtn_t * osm_nvl_prtn_make_new(osm_nvlink_mgr_t * p_mgr,
				   const char *name, ib_net16_t pkey,
				   uint8_t max_trunk_links,
				   boolean_t is_reroute_req);

/****f* OpenSM: NVLink/osm_nvl_prtn_is_trunk_in_cnd
* NAME
*	osm_nvl_prtn_is_trunk_in_cnd
*
* DESCRIPTION
* 	The function returns TRUE if p_physp is a switch trunk port that is
* 	in Contain and Drain state. Otherwise, FALSE.
*
* SYNOPSIS
*/
boolean_t osm_nvl_prtn_is_trunk_in_cnd(osm_nvlink_mgr_t * p_mgr, osm_physp_t * p_physp);
/*
* PARAMETERS
*	p_mgr
*		Pointer to an osm_nvlink_mgr_t object
*
*	p_physp
*		Pointer to physical port
*
* RETURN VALUES
* 	The function returns TRUE if p_physp is a switch trunk port that is
* 	in Contain and Drain state. Otherwise, FALSE.
*
*********/

/****f* OpenSM: NVLink/osm_nvl_prtn_add_gpu
* NAME
*	osm_nvl_prtn_add_gpu
*
* DESCRIPTION
* 	The function adds all the ports of the GPU to the partition.
*
* SYNOPSIS
*/
int osm_nvl_prtn_add_gpu(osm_nvlink_mgr_t *p_mgr, osm_prtn_t *p_prtn, ib_net64_t node_guid);
/*
* PARAMETERS
*	p_mgr
*		Pointer to an osm_nvlink_mgr_t object
*
*	p_prtn
*		Pointer to an osm_prtn_t object
*
*	node_guid
*		The node GUID of the GPU
*
* RETURN VALUES
* 	Return 0 in case of success
*
*********/

/****f* OpenSM: NVLink/osm_nvl_prtn_get_gpu_ptrn
* NAME
*	osm_nvl_prtn_get_gpu_ptrn
*
* DESCRIPTION
* 	Returns Partition key of the GPU.
*
* SYNOPSIS
*/
ib_net16_t osm_nvl_prtn_get_gpu_prtn(IN osm_nvlink_mgr_t * p_mgr, IN osm_gpu_t * p_gpu);
/*
* PARAMETERS
*	p_mgr
*		Pointer to an osm_nvlink_mgr_t object.
*
* 	p_gpu
*		Pointer to GPU object.
*
* RETURN VALUES
* 	Returns Partition key of the GPU in network order.
*
*********/
typedef enum _osm_nvlink_link_type {
	IB_NVLINK_LINK_UNDEFINED,
	IB_NVLINK_LINK_TRUNK,
	IB_NVLINK_LINK_ACCESS_SW,
	IB_NVLINK_LINK_ACCESS_GPU,
} osm_nvlink_link_type_t;

typedef struct osm_nvlink_link {
	osm_physp_t *p_physp;
	uint8_t type;
} osm_nvlink_link_t;
/*
* FIELDS
* 	p_physp
*		Pointer to osm_physp_t object.
* 	type
* 		Link type (osm_nvlink_link_type_t)
*
* SEE ALSO
* 	osm_nvlink_link_t, osm_nvl_prtn_get_links, osm_nvl_prtn_release_links
*********/

/****f* OpenSM: NVLink/osm_nvl_prtn_get_links
* NAME
*	osm_nvl_prtn_get_links
*
* DESCRIPTION
* 	The function returns array of osm_nvlink_link_type_t objects associate by
* 	specified partition.
*
* SYNOPSIS
*/
osm_nvlink_link_t * osm_nvl_prtn_get_links(osm_subn_t * p_subn, uint16_t pkey);
/*
* PARAMETERS
*	p_subn
*		[in] Pointer to a Subnet object to construct.
*
* 	pkey
* 		Partition
*
* RETURN VALUES
* 	The function return NULL in case of failure.
* 	Otherwise, returns array of osm_nvlink_link_type_t, last element is zero.
*
* NOTES
* 	Returned array of osm_nvlink_link_type_t objects should be released using
* 	osm_nvl_prtn_release_links function.
*
* SEE ALSO
* 	osm_nvlink_link_t, osm_nvl_prtn_get_links, osm_nvl_prtn_release_links
*********/

/****f* OpenSM: NVLink/osm_nvl_prtn_release_links
* NAME
*	osm_nvl_prtn_release_links
*
* DESCRIPTION
* 	The function releases array of osm_nvlink_link_type_t objects previously
* 	allocated by osm_nvl_prtn_get_links.
*
* SYNOPSIS
*/
void osm_nvl_prtn_release_links(IN osm_nvlink_link_t * links);
/*
* PARAMETERS
* 	links
* 		Array of osm_nvl_prtn_get_links objects.
*
* RETURN VALUE
*	This function does not return a value.
*
* SEE ALSO
* 	osm_nvlink_link_t, osm_nvl_prtn_get_links, osm_nvl_prtn_release_links
*/

void osm_nvl_prtn_preprocess(osm_subn_t * p_subn);

void osm_nvl_prtn_laas_process(osm_subn_t * p_subn);

void osm_nvl_prtn_send_rail_filter_of_prtn(osm_subn_t * p_subn, osm_switch_t * p_sw,
					   ib_net16_t pkey, uint8_t rail, uint8_t ingress_block,
					   uint8_t egress_block);

void osm_nvl_prtn_cpy_subn_rail_filter_to_prtn(osm_subn_t * p_subn, osm_switch_t * p_sw,
					       ib_net16_t pkey, uint8_t rail, uint8_t ingress_block,
					       uint8_t egress_block,
					       ib_rail_filter_config_t * p_rail_filter);

void osm_nvl_prtn_reroute_finish(osm_subn_t * p_subn);

boolean_t osm_nvl_prtn_get_gpu_reroute_allowed(osm_nvlink_mgr_t * p_mgr, osm_gpu_t * p_gpu);

END_C_DECLS
#endif				/* ifndef OSM_NVLINK_PRTN_H */
