# Copyright (c) 2004-2010 Mellanox Technologies LTD. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

import ctypes
import os
import platform


class MftCoreDeviceException(Exception):
    pass


DEVICE_INFO = None
DEVICE_INFO_SO = "device_info.so"
DEVICE_INFO_DLL = "device_info.dll"

try:
    from ctypes import *
    if platform.system() == "Windows" or os.name == "nt":
        try:
            DEVICE_INFO = CDLL(DEVICE_INFO_DLL)
        except BaseException:
            DEVICE_INFO = CDLL(os.path.join(os.path.dirname(os.path.realpath(__file__)), DEVICE_INFO_DLL))
    else:
        try:
            DEVICE_INFO = CDLL(DEVICE_INFO_SO)
        except BaseException:
            DEVICE_INFO = CDLL(os.path.join(os.path.dirname(os.path.realpath(__file__)), DEVICE_INFO_SO))
except Exception as exp:
    raise MftCoreDeviceException("Failed to load shared library device_info: %s" % exp)


def get_num_of_devices():
    DEVICE_INFO.get_num_of_devices.restype = c_uint
    return DEVICE_INFO.get_num_of_devices()


def get_num_of_tracer_supported_deveices():
    DEVICE_INFO.get_num_of_tracer_supported_deveices.restype = c_uint
    return DEVICE_INFO.get_num_of_tracer_supported_deveices()


def get_device_name_by_index(i):
    dev_name = ctypes.create_string_buffer(64)
    DEVICE_INFO.get_device_name_by_index.restype = c_void_p
    DEVICE_INFO.get_device_name_by_index(i, dev_name)
    string = dev_name.value.decode("utf-8")
    return string


def get_device_name_by_id(id):
    dev_name = ctypes.create_string_buffer(64)
    DEVICE_INFO.get_device_name_by_id.restype = c_void_p
    DEVICE_INFO.get_device_name_by_id(id, dev_name)
    string = dev_name.value.decode("utf-8")
    return string


def get_device_id_from_str(dev_name):
    DEVICE_INFO.get_device_id_from_str.restype = c_int
    DEVICE_INFO.get_device_id_from_str.argtypes = [c_char_p]
    return DEVICE_INFO.get_device_id_from_str(dev_name.encode("utf-8"))


def get_device_id_from_str_non_default_json_dir(dev_name, json_dir):
    DEVICE_INFO.get_device_id_from_str_non_default_json_dir.restype = c_int
    DEVICE_INFO.get_device_id_from_str_non_default_json_dir.argtypes = [c_char_p, c_char_p]
    return DEVICE_INFO.get_device_id_from_str_non_default_json_dir(dev_name.encode("utf-8"), json_dir.encode("utf-8"))


def get_all_tracer_supported_ids():
    size = get_num_of_tracer_supported_deveices()
    devArray = (c_uint32 * size)()
    DEVICE_INFO.get_all_tracer_supported_ids(cast(devArray, POINTER(c_uint32)))
    return list(devArray)


# Intialization and destruction functions
DEVICE_INFO.create_device_info.argtypes = [ctypes.c_void_p, ctypes.c_uint]
DEVICE_INFO.create_device_info.restype = c_int


def create_device_info(device_info_ptr, device_id):
    return DEVICE_INFO.create_device_info(device_info_ptr, device_id)


DEVICE_INFO.create_device_info_with_json_dir.argtypes = [ctypes.c_void_p, ctypes.c_uint, ctypes.c_char_p]
DEVICE_INFO.create_device_info_with_json_dir.restype = c_int


def create_device_info_with_json_dir(device_info_ptr, device_id, json_dir):
    c_char_p_json_dir = ctypes.c_char_p(json_dir.encode('utf-8'))  # necessary because Python strings are Unicode, while C strings (char*) are arrays of bytes.
    return DEVICE_INFO.create_device_info_with_json_dir(device_info_ptr, device_id, c_char_p_json_dir)


DEVICE_INFO.destroy_device_info.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.destroy_device_info.restype = c_int


def destroy_device_info(device_info_ptr_ref):
    return DEVICE_INFO.destroy_device_info(device_info_ptr_ref)


# Bool functions
DEVICE_INFO.is_nic.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_nic.restype = c_bool


def is_nic(device_info_ptr):
    return DEVICE_INFO.is_nic(device_info_ptr)


DEVICE_INFO.is_switch.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_switch.restype = c_bool


def is_switch(device_info_ptr):
    return DEVICE_INFO.is_switch(device_info_ptr)


DEVICE_INFO.is_retimer.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_retimer.restype = c_bool


def is_retimer(device_info_ptr):
    return DEVICE_INFO.is_retimer(device_info_ptr)


DEVICE_INFO.is_gearbox.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_gearbox.restype = c_bool


def is_gearbox(device_info_ptr):
    return DEVICE_INFO.is_gearbox(device_info_ptr)


DEVICE_INFO.is_4th_gen_nic.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_4th_gen_nic.restype = c_bool


def is_4th_gen_nic(device_info_ptr):
    return DEVICE_INFO.is_4th_gen_nic(device_info_ptr)


DEVICE_INFO.is_5th_gen_nic.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_5th_gen_nic.restype = c_bool


def is_5th_gen_nic(device_info_ptr):
    return DEVICE_INFO.is_5th_gen_nic(device_info_ptr)


DEVICE_INFO.is_fs5.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_fs5.restype = c_bool


def is_fs5(device_info_ptr):
    return DEVICE_INFO.is_fs5(device_info_ptr)


DEVICE_INFO.is_fs4.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_fs4.restype = c_bool


def is_fs4(device_info_ptr):
    return DEVICE_INFO.is_fs4(device_info_ptr)


DEVICE_INFO.is_fs3.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_fs3.restype = c_bool


def is_fs3(device_info_ptr):
    return DEVICE_INFO.is_fs3(device_info_ptr)


DEVICE_INFO.is_fs2.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_fs2.restype = c_bool


def is_fs2(device_info_ptr):
    return DEVICE_INFO.is_fs2(device_info_ptr)


DEVICE_INFO.is_switch_ib.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_switch_ib.restype = c_bool


def is_switch_ib(device_info_ptr):
    return DEVICE_INFO.is_switch_ib(device_info_ptr)


DEVICE_INFO.is_switch_ib2.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_switch_ib2.restype = c_bool


def is_switch_ib2(device_info_ptr):
    return DEVICE_INFO.is_switch_ib2(device_info_ptr)


DEVICE_INFO.fw_str_db_signature_exists.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.fw_str_db_signature_exists.restype = c_bool


def fw_str_db_signature_exists(device_info_ptr):
    return DEVICE_INFO.fw_str_db_signature_exists(device_info_ptr)


DEVICE_INFO.supports_phy_uc.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.supports_phy_uc.restype = c_bool


def supports_phy_uc(device_info_ptr):
    return DEVICE_INFO.supports_phy_uc(device_info_ptr)


DEVICE_INFO.is_dynamic_device_without_irisc_id.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_dynamic_device_without_irisc_id.restype = c_bool


def is_dynamic_device_without_irisc_id(device_info_ptr):
    return DEVICE_INFO.is_dynamic_device_without_irisc_id(device_info_ptr)


DEVICE_INFO.is_dynamic_db_supported.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_dynamic_db_supported.restype = c_bool


def is_dynamic_db_supported(device_info_ptr):
    return DEVICE_INFO.is_dynamic_db_supported(device_info_ptr)


DEVICE_INFO.is_nvRisc.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.is_nvRisc.restype = c_bool


def is_nvRisc(device_info_ptr):
    return DEVICE_INFO.is_nvRisc(device_info_ptr)


# Get device properties functions
DEVICE_INFO.get_max_main_irisc.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_max_main_irisc.restype = c_uint


def get_max_main_irisc(device_info_ptr):
    return DEVICE_INFO.get_max_main_irisc(device_info_ptr)


DEVICE_INFO.get_max_apu.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_max_apu.restype = c_uint


def get_max_apu(device_info_ptr):
    return DEVICE_INFO.get_max_apu(device_info_ptr)


DEVICE_INFO.get_max_num_of_tiles.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_max_num_of_tiles.restype = c_uint


def get_max_num_of_tiles(device_info_ptr):
    return DEVICE_INFO.get_max_num_of_tiles(device_info_ptr)


DEVICE_INFO.get_max_irisc_per_tile.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_max_irisc_per_tile.restype = c_uint


def get_max_irisc_per_tile(device_info_ptr):
    return DEVICE_INFO.get_max_irisc_per_tile(device_info_ptr)


DEVICE_INFO.get_iris_start_addr.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_iris_start_addr.restype = c_uint


def get_iris_start_addr(device_info_ptr):
    return DEVICE_INFO.get_iris_start_addr(device_info_ptr)


DEVICE_INFO.get_iris_step.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_iris_step.restype = c_uint


def get_iris_step(device_info_ptr):
    return DEVICE_INFO.get_iris_step(device_info_ptr)


DEVICE_INFO.get_tile_start.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_tile_start.restype = c_uint


def get_tile_start(device_info_ptr):
    return DEVICE_INFO.get_tile_start(device_info_ptr)


DEVICE_INFO.get_tile_step.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_tile_step.restype = c_uint


def get_tile_step(device_info_ptr):
    return DEVICE_INFO.get_tile_step(device_info_ptr)


DEVICE_INFO.get_apu_start_addr_step.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_apu_start_addr_step.restype = c_uint


def get_apu_start_addr_step(device_info_ptr):
    return DEVICE_INFO.get_apu_start_addr_step(device_info_ptr)


DEVICE_INFO.get_apu_step.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_apu_step.restype = c_uint


def get_apu_step(device_info_ptr):
    return DEVICE_INFO.get_apu_step(device_info_ptr)


DEVICE_INFO.get_apu_max_num_of_steps.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_apu_max_num_of_steps.restype = c_uint


def get_apu_max_num_of_steps(device_info_ptr):
    return DEVICE_INFO.get_apu_max_num_of_steps(device_info_ptr)


DEVICE_INFO.get_device_hw_id.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_device_hw_id.restype = c_uint


def get_device_hw_id(device_info_ptr):
    return DEVICE_INFO.get_device_hw_id(device_info_ptr)


DEVICE_INFO.get_pci_device_id.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_pci_device_id.restype = c_uint


def get_pci_device_id(device_info_ptr):
    return DEVICE_INFO.get_pci_device_id(device_info_ptr)


DEVICE_INFO.get_device_fw_major.argtypes = (ctypes.c_void_p,)
DEVICE_INFO.get_device_fw_major.restype = c_uint


def get_device_fw_major(device_info_ptr):
    return DEVICE_INFO.get_device_fw_major(device_info_ptr)


DEVICE_INFO.get_field_as_int.argtypes = (ctypes.c_void_p, ctypes.c_char_p, ctypes.c_char_p)
DEVICE_INFO.get_field_as_int.restype = c_uint


def get_field_as_int(device_info_ptr):
    return DEVICE_INFO.get_field_as_int(device_info_ptr)


DEVICE_INFO.get_device_fw_name.argtypes = (ctypes.c_void_p, ctypes.c_char_p)
DEVICE_INFO.get_device_fw_name.restype = c_void_p


def get_device_fw_name(device_info):
    dev_name = ctypes.create_string_buffer(64)
    DEVICE_INFO.get_device_fw_name(device_info, dev_name)
    string = dev_name.value.decode("utf-8")
    return string


DEVICE_INFO.get_device_name.argtypes = (ctypes.c_void_p, ctypes.c_char_p)
DEVICE_INFO.get_device_name.restype = c_void_p


def get_device_name(device_info):
    dev_name = ctypes.create_string_buffer(64)
    DEVICE_INFO.get_device_name(device_info, dev_name)
    string = dev_name.value.decode("utf-8")
    return string


DEVICE_INFO.get_device_short_name.argtypes = (ctypes.c_void_p, ctypes.c_char_p)
DEVICE_INFO.get_device_short_name.restype = c_void_p


def get_device_short_name(device_info):
    dev_name = ctypes.create_string_buffer(64)
    DEVICE_INFO.get_device_short_name(device_info, dev_name)
    string = dev_name.value.decode("utf-8")
    return string


DEVICE_INFO.get_field_as_string.argtypes = (ctypes.c_void_p, ctypes.c_char_p, ctypes.c_char_p, ctypes.c_char_p)
DEVICE_INFO.get_field_as_string.restype = c_void_p


def get_field_as_string(device_info, field_name, json_attribute):
    data = ctypes.create_string_buffer(64)
    DEVICE_INFO.get_field_as_string(device_info, data, field_name, json_attribute)
    string = data.value.decode("utf-8")
    return string
