# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

import binascii
import os

from .dumper import PacketDumper
from ..constants import PacketType

__all__ = ['RawPacketDumper', 'RawHexPacketDumper']


class RawPacketDumper(PacketDumper):
    """A concrete class that represents packet dumper for raw data packets."""
    __product_type__ = PacketType.RAW.value

    def dump(self, packets):
        """Dumps a list of raw data packets into a file.

        :param list[RawPacket] packets: A list of packets to be saved
        """
        super().dump(packets)
        buffers = self.get_packets_buffers(packets)
        data = os.linesep.encode('utf-8').join(buffer for buffer in buffers)
        with open(self.path, 'wb') as f:
            f.write(data)


class RawHexPacketDumper(RawPacketDumper):
    """A concrete class that represents packet dumper for raw hexadecimal

    packets."""
    __product_type__ = PacketType.HEX.value

    @classmethod
    def get_packets_buffers(cls, packets):
        """Returns list of packets buffers from the given packets.

        :param packets: List of packets
        :return: packets buffers
        :rtype: List[bytes]
        """
        buffers = super().get_packets_buffers(packets)
        return [binascii.b2a_hex(buffer) for buffer in buffers]
