# Copyright (C) Nov 2020 Mellanox Technologies Ltd. All rights reserved.
# Copyright (c) 2021 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This software is available to you under a choice of one of two
# licenses.  You may choose to be licensed under the terms of the GNU
# General Public License (GPL) Version 2, available from the file
# COPYING in the main directory of this source tree, or the
# OpenIB.org BSD license below:
#
#     Redistribution and use in source and binary forms, with or
#     without modification, are permitted provided that the following
#     conditions are met:
#
#      - Redistributions of source code must retain the above
#        copyright notice, this list of conditions and the following
#        disclaimer.
#
#      - Redistributions in binary form must reproduce the above
#        copyright notice, this list of conditions and the following
#        disclaimer in the documentation and/or other materials
#        provided with the distribution.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
# --

import os
from collections.abc import Iterable
from errno import ENOENT

from ..factory import Product, FactoryMeta

__all__ = ['PacketLoader']


class PacketLoader(Iterable, Product, metaclass=FactoryMeta):
    """An abstract class that represents packet loader.

    :param str path: The path to a file which contains packets
    """
    def __init__(self, path, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.path = path

    def load(self):
        """Loads a list of packets from a file.

        :return: A list of packets.
        :rtype: list[Packet]
        """
        return [packet for packet in self]

    @property
    def path(self):
        """The path to an input file.

        :getter: Gets the path to an input file
        :setter: Sets the path to an input file
        :type: str
        """
        return self._path

    @path.setter
    def path(self, path):
        if os.path.isfile(path) is False:
            raise FileNotFoundError(ENOENT, os.strerror(ENOENT), path)
        self._path = path
