#!/usr/bin/env bash
###############################################################################
#
# Bash Remediation Script for CIS Aliyun Linux 2 Benchmark for Level 1
#
# Profile Description:
# This profile defines a baseline that aligns to the "Level 1"
# configuration from the Center for Internet Security® Aliyun
# Linux 2 Benchmark™, v1.0.0, released 08-16-2019.
# This profile includes Center for Internet Security®
# Aliyun Linux 2 CIS Benchmarks™ content.
#
# Profile ID:  xccdf_org.ssgproject.content_profile_cis_l1
# Benchmark ID:  xccdf_org.ssgproject.content_benchmark_ALINUX-2
# Benchmark Version:  0.1.70
# XCCDF Version:  1.2
#
# This file can be generated by OpenSCAP using:
# $ oscap xccdf generate fix --profile xccdf_org.ssgproject.content_profile_cis_l1 --fix-type bash ssg-alinux2-ds.xml
#
# This Bash Remediation Script is generated from an XCCDF profile without preliminary evaluation.
# It attempts to fix every selected rule, even if the system is already compliant.
#
# How to apply this Bash Remediation Script:
# $ sudo ./remediation-script.sh
#
###############################################################################

###############################################################################
# BEGIN fix (1 / 219) for 'xccdf_org.ssgproject.content_rule_disable_prelink'
###############################################################################
(>&2 echo "Remediating rule 1/219: 'xccdf_org.ssgproject.content_rule_disable_prelink'")
# prelink not installed
if test -e /etc/sysconfig/prelink -o -e /usr/sbin/prelink; then
    if grep -q ^PRELINKING /etc/sysconfig/prelink
    then
        sed -i 's/^PRELINKING[:blank:]*=[:blank:]*[:alpha:]*/PRELINKING=no/' /etc/sysconfig/prelink
    else
        printf '\n' >> /etc/sysconfig/prelink
        printf '%s\n' '# Set PRELINKING=no per security requirements' 'PRELINKING=no' >> /etc/sysconfig/prelink
    fi

    # Undo previous prelink changes to binaries if prelink is available.
    if test -x /usr/sbin/prelink; then
        /usr/sbin/prelink -ua
    fi
fi

# END fix for 'xccdf_org.ssgproject.content_rule_disable_prelink'

###############################################################################
# BEGIN fix (2 / 219) for 'xccdf_org.ssgproject.content_rule_package_aide_installed'
###############################################################################
(>&2 echo "Remediating rule 2/219: 'xccdf_org.ssgproject.content_rule_package_aide_installed'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_package_aide_installed' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_package_aide_installed'

###############################################################################
# BEGIN fix (3 / 219) for 'xccdf_org.ssgproject.content_rule_aide_build_database'
###############################################################################
(>&2 echo "Remediating rule 3/219: 'xccdf_org.ssgproject.content_rule_aide_build_database'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_aide_build_database' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_aide_build_database'

###############################################################################
# BEGIN fix (4 / 219) for 'xccdf_org.ssgproject.content_rule_aide_periodic_cron_checking'
###############################################################################
(>&2 echo "Remediating rule 4/219: 'xccdf_org.ssgproject.content_rule_aide_periodic_cron_checking'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_aide_periodic_cron_checking' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_aide_periodic_cron_checking'

###############################################################################
# BEGIN fix (5 / 219) for 'xccdf_org.ssgproject.content_rule_partition_for_tmp'
###############################################################################
(>&2 echo "Remediating rule 5/219: 'xccdf_org.ssgproject.content_rule_partition_for_tmp'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_partition_for_tmp' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_partition_for_tmp'

###############################################################################
# BEGIN fix (6 / 219) for 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_globally_activated'
###############################################################################
(>&2 echo "Remediating rule 6/219: 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_globally_activated'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_globally_activated' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_globally_activated'

###############################################################################
# BEGIN fix (7 / 219) for 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_never_disabled'
###############################################################################
(>&2 echo "Remediating rule 7/219: 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_never_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_never_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_never_disabled'

###############################################################################
# BEGIN fix (8 / 219) for 'xccdf_org.ssgproject.content_rule_banner_etc_issue'
###############################################################################
(>&2 echo "Remediating rule 8/219: 'xccdf_org.ssgproject.content_rule_banner_etc_issue'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_banner_etc_issue' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_banner_etc_issue'

###############################################################################
# BEGIN fix (9 / 219) for 'xccdf_org.ssgproject.content_rule_banner_etc_issue_net'
###############################################################################
(>&2 echo "Remediating rule 9/219: 'xccdf_org.ssgproject.content_rule_banner_etc_issue_net'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

remote_login_banner_text='^(Authorized[\s\n]+uses[\s\n]+only\.[\s\n]+All[\s\n]+activity[\s\n]+may[\s\n]+be[\s\n]+monitored[\s\n]+and[\s\n]+reported\.|^(?!.*(\\|fedora|rhel|sle|ubuntu)).*)$'


# Multiple regexes transform the banner regex into a usable banner
# 0 - Remove anchors around the banner text
remote_login_banner_text=$(echo "$remote_login_banner_text" | sed 's/^\^\(.*\)\$$/\1/g')
# 1 - Keep only the first banners if there are multiple
#    (dod_banners contains the long and short banner)
remote_login_banner_text=$(echo "$remote_login_banner_text" | sed 's/^(\(.*\.\)|.*)$/\1/g')
# 2 - Add spaces ' '. (Transforms regex for "space or newline" into a " ")
remote_login_banner_text=$(echo "$remote_login_banner_text" | sed 's/\[\\s\\n\]+/ /g')
# 3 - Adds newlines. (Transforms "(?:\[\\n\]+|(?:\\n)+)" into "\n")
remote_login_banner_text=$(echo "$remote_login_banner_text" | sed 's/(?:\[\\n\]+|(?:\\\\n)+)/\n/g')
# 4 - Remove any leftover backslash. (From any parethesis in the banner, for example).
remote_login_banner_text=$(echo "$remote_login_banner_text" | sed 's/\\//g')
formatted=$(echo "$remote_login_banner_text" | fold -sw 80)

cat <<EOF >/etc/issue.net
$formatted
EOF

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_banner_etc_issue_net'

###############################################################################
# BEGIN fix (10 / 219) for 'xccdf_org.ssgproject.content_rule_banner_etc_motd'
###############################################################################
(>&2 echo "Remediating rule 10/219: 'xccdf_org.ssgproject.content_rule_banner_etc_motd'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_banner_etc_motd' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_banner_etc_motd'

###############################################################################
# BEGIN fix (11 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_issue'
###############################################################################
(>&2 echo "Remediating rule 11/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_issue'")
chgrp 0 /etc/issue

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_issue'

###############################################################################
# BEGIN fix (12 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_motd'
###############################################################################
(>&2 echo "Remediating rule 12/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_motd'")
chgrp 0 /etc/motd

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_motd'

###############################################################################
# BEGIN fix (13 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_etc_issue'
###############################################################################
(>&2 echo "Remediating rule 13/219: 'xccdf_org.ssgproject.content_rule_file_owner_etc_issue'")
chown 0 /etc/issue

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_etc_issue'

###############################################################################
# BEGIN fix (14 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_etc_motd'
###############################################################################
(>&2 echo "Remediating rule 14/219: 'xccdf_org.ssgproject.content_rule_file_owner_etc_motd'")
chown 0 /etc/motd

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_etc_motd'

###############################################################################
# BEGIN fix (15 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_issue'
###############################################################################
(>&2 echo "Remediating rule 15/219: 'xccdf_org.ssgproject.content_rule_file_permissions_etc_issue'")





chmod u-xs,g-xws,o-xwt /etc/issue

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_issue'

###############################################################################
# BEGIN fix (16 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_motd'
###############################################################################
(>&2 echo "Remediating rule 16/219: 'xccdf_org.ssgproject.content_rule_file_permissions_etc_motd'")





chmod u-xs,g-xws,o-xwt /etc/motd

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_motd'

###############################################################################
# BEGIN fix (17 / 219) for 'xccdf_org.ssgproject.content_rule_account_password_pam_faillock_password_auth'
###############################################################################
(>&2 echo "Remediating rule 17/219: 'xccdf_org.ssgproject.content_rule_account_password_pam_faillock_password_auth'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_account_password_pam_faillock_password_auth' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_account_password_pam_faillock_password_auth'

###############################################################################
# BEGIN fix (18 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_password_pam_pwhistory_remember_password_auth'
###############################################################################
(>&2 echo "Remediating rule 18/219: 'xccdf_org.ssgproject.content_rule_accounts_password_pam_pwhistory_remember_password_auth'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_accounts_password_pam_pwhistory_remember_password_auth' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_password_pam_pwhistory_remember_password_auth'

###############################################################################
# BEGIN fix (19 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_password_pam_pwhistory_remember_system_auth'
###############################################################################
(>&2 echo "Remediating rule 19/219: 'xccdf_org.ssgproject.content_rule_accounts_password_pam_pwhistory_remember_system_auth'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_accounts_password_pam_pwhistory_remember_system_auth' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_password_pam_pwhistory_remember_system_auth'

###############################################################################
# BEGIN fix (20 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_password_pam_minclass'
###############################################################################
(>&2 echo "Remediating rule 20/219: 'xccdf_org.ssgproject.content_rule_accounts_password_pam_minclass'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_accounts_password_pam_minclass' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_password_pam_minclass'

###############################################################################
# BEGIN fix (21 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_password_pam_minlen'
###############################################################################
(>&2 echo "Remediating rule 21/219: 'xccdf_org.ssgproject.content_rule_accounts_password_pam_minlen'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_accounts_password_pam_minlen' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_password_pam_minlen'

###############################################################################
# BEGIN fix (22 / 219) for 'xccdf_org.ssgproject.content_rule_set_password_hashing_algorithm_systemauth'
###############################################################################
(>&2 echo "Remediating rule 22/219: 'xccdf_org.ssgproject.content_rule_set_password_hashing_algorithm_systemauth'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q pam; then

if [ -e "/etc/pam.d/system-auth" ] ; then
    PAM_FILE_PATH="/etc/pam.d/system-auth"
    if [ -f /usr/bin/authselect ]; then
        
        if ! authselect check; then
        echo "
        authselect integrity check failed. Remediation aborted!
        This remediation could not be applied because an authselect profile was not selected or the selected profile is not intact.
        It is not recommended to manually edit the PAM files when authselect tool is available.
        In cases where the default authselect profile does not cover a specific demand, a custom authselect profile is recommended."
        exit 1
        fi

        CURRENT_PROFILE=$(authselect current -r | awk '{ print $1 }')
        # If not already in use, a custom profile is created preserving the enabled features.
        if [[ ! $CURRENT_PROFILE == custom/* ]]; then
            ENABLED_FEATURES=$(authselect current | tail -n+3 | awk '{ print $2 }')
            authselect create-profile hardening -b $CURRENT_PROFILE
            CURRENT_PROFILE="custom/hardening"
            
            authselect apply-changes -b --backup=before-hardening-custom-profile
            authselect select $CURRENT_PROFILE
            for feature in $ENABLED_FEATURES; do
                authselect enable-feature $feature;
            done
            
            authselect apply-changes -b --backup=after-hardening-custom-profile
        fi
        PAM_FILE_NAME=$(basename "/etc/pam.d/system-auth")
        PAM_FILE_PATH="/etc/authselect/$CURRENT_PROFILE/$PAM_FILE_NAME"

        authselect apply-changes -b
    fi
    if ! grep -qP '^\s*password\s+'"sufficient"'\s+pam_unix.so\s*.*' "$PAM_FILE_PATH"; then
            # Line matching group + control + module was not found. Check group + module.
            if [ "$(grep -cP '^\s*password\s+.*\s+pam_unix.so\s*' "$PAM_FILE_PATH")" -eq 1 ]; then
                # The control is updated only if one single line matches.
                sed -i -E --follow-symlinks 's/^(\s*password\s+).*(\bpam_unix.so.*)/\1'"sufficient"' \2/' "$PAM_FILE_PATH"
            else
                echo 'password    '"sufficient"'    pam_unix.so' >> "$PAM_FILE_PATH"
            fi
        fi
        # Check the option
        if ! grep -qP '^\s*password\s+'"sufficient"'\s+pam_unix.so\s*.*\ssha512\b' "$PAM_FILE_PATH"; then
            sed -i -E --follow-symlinks '/\s*password\s+'"sufficient"'\s+pam_unix.so.*/ s/$/ sha512/' "$PAM_FILE_PATH"
        fi
    if [ -f /usr/bin/authselect ]; then
        
        authselect apply-changes -b
    fi
else
    echo "/etc/pam.d/system-auth was not found" >&2
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_set_password_hashing_algorithm_systemauth'

###############################################################################
# BEGIN fix (23 / 219) for 'xccdf_org.ssgproject.content_rule_require_emergency_target_auth'
###############################################################################
(>&2 echo "Remediating rule 23/219: 'xccdf_org.ssgproject.content_rule_require_emergency_target_auth'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

service_file="/usr/lib/systemd/system/emergency.service"


sulogin='/bin/sh -c "/sbin/sulogin; /usr/bin/systemctl --fail --no-block default"'


if grep "^ExecStart=.*" "$service_file" ; then
    sed -i "s%^ExecStart=.*%ExecStart=-$sulogin%" "$service_file"
else
    echo "ExecStart=-$sulogin" >> "$service_file"
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_require_emergency_target_auth'

###############################################################################
# BEGIN fix (24 / 219) for 'xccdf_org.ssgproject.content_rule_require_singleuser_auth'
###############################################################################
(>&2 echo "Remediating rule 24/219: 'xccdf_org.ssgproject.content_rule_require_singleuser_auth'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

service_file="/usr/lib/systemd/system/rescue.service"

sulogin='/bin/sh -c "/sbin/sulogin; /usr/bin/systemctl --fail --no-block default"'

if grep "^ExecStart=.*" "$service_file" ; then
    sed -i "s%^ExecStart=.*%ExecStart=-$sulogin%" "$service_file"
else
    echo "ExecStart=-$sulogin" >> "$service_file"
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_require_singleuser_auth'

###############################################################################
# BEGIN fix (25 / 219) for 'xccdf_org.ssgproject.content_rule_account_unique_id'
###############################################################################
(>&2 echo "Remediating rule 25/219: 'xccdf_org.ssgproject.content_rule_account_unique_id'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_account_unique_id' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_account_unique_id'

###############################################################################
# BEGIN fix (26 / 219) for 'xccdf_org.ssgproject.content_rule_group_unique_id'
###############################################################################
(>&2 echo "Remediating rule 26/219: 'xccdf_org.ssgproject.content_rule_group_unique_id'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_group_unique_id' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_group_unique_id'

###############################################################################
# BEGIN fix (27 / 219) for 'xccdf_org.ssgproject.content_rule_group_unique_name'
###############################################################################
(>&2 echo "Remediating rule 27/219: 'xccdf_org.ssgproject.content_rule_group_unique_name'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_group_unique_name' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_group_unique_name'

###############################################################################
# BEGIN fix (28 / 219) for 'xccdf_org.ssgproject.content_rule_account_disable_post_pw_expiration'
###############################################################################
(>&2 echo "Remediating rule 28/219: 'xccdf_org.ssgproject.content_rule_account_disable_post_pw_expiration'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_account_disable_post_pw_expiration' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_account_disable_post_pw_expiration'

###############################################################################
# BEGIN fix (29 / 219) for 'xccdf_org.ssgproject.content_rule_account_unique_name'
###############################################################################
(>&2 echo "Remediating rule 29/219: 'xccdf_org.ssgproject.content_rule_account_unique_name'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_account_unique_name' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_account_unique_name'

###############################################################################
# BEGIN fix (30 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_maximum_age_login_defs'
###############################################################################
(>&2 echo "Remediating rule 30/219: 'xccdf_org.ssgproject.content_rule_accounts_maximum_age_login_defs'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q shadow-utils; then

var_accounts_maximum_age_login_defs='365'


grep -q ^PASS_MAX_DAYS /etc/login.defs && \
  sed -i "s/PASS_MAX_DAYS.*/PASS_MAX_DAYS     $var_accounts_maximum_age_login_defs/g" /etc/login.defs
if ! [ $? -eq 0 ]; then
    echo "PASS_MAX_DAYS      $var_accounts_maximum_age_login_defs" >> /etc/login.defs
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_maximum_age_login_defs'

###############################################################################
# BEGIN fix (31 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_minimum_age_login_defs'
###############################################################################
(>&2 echo "Remediating rule 31/219: 'xccdf_org.ssgproject.content_rule_accounts_minimum_age_login_defs'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q shadow-utils; then

var_accounts_minimum_age_login_defs='1'


grep -q ^PASS_MIN_DAYS /etc/login.defs && \
  sed -i "s/PASS_MIN_DAYS.*/PASS_MIN_DAYS     $var_accounts_minimum_age_login_defs/g" /etc/login.defs
if ! [ $? -eq 0 ]; then
    echo "PASS_MIN_DAYS      $var_accounts_minimum_age_login_defs" >> /etc/login.defs
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_minimum_age_login_defs'

###############################################################################
# BEGIN fix (32 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_password_warn_age_login_defs'
###############################################################################
(>&2 echo "Remediating rule 32/219: 'xccdf_org.ssgproject.content_rule_accounts_password_warn_age_login_defs'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q shadow-utils; then

var_accounts_password_warn_age_login_defs='7'


grep -q ^PASS_WARN_AGE /etc/login.defs && \
sed -i "s/PASS_WARN_AGE.*/PASS_WARN_AGE\t$var_accounts_password_warn_age_login_defs/g" /etc/login.defs
if ! [ $? -eq 0 ]
then
  echo -e "PASS_WARN_AGE\t$var_accounts_password_warn_age_login_defs" >> /etc/login.defs
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_password_warn_age_login_defs'

###############################################################################
# BEGIN fix (33 / 219) for 'xccdf_org.ssgproject.content_rule_gid_passwd_group_same'
###############################################################################
(>&2 echo "Remediating rule 33/219: 'xccdf_org.ssgproject.content_rule_gid_passwd_group_same'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_gid_passwd_group_same' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_gid_passwd_group_same'

###############################################################################
# BEGIN fix (34 / 219) for 'xccdf_org.ssgproject.content_rule_no_forward_files'
###############################################################################
(>&2 echo "Remediating rule 34/219: 'xccdf_org.ssgproject.content_rule_no_forward_files'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_no_forward_files' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_no_forward_files'

###############################################################################
# BEGIN fix (35 / 219) for 'xccdf_org.ssgproject.content_rule_no_legacy_plus_entries_etc_passwd'
###############################################################################
(>&2 echo "Remediating rule 35/219: 'xccdf_org.ssgproject.content_rule_no_legacy_plus_entries_etc_passwd'")

if grep -q '^\+' /etc/passwd; then
# backup old file to /etc/passwd-
	cp /etc/passwd /etc/passwd-
	sed -i '/^\+.*$/d' /etc/passwd
fi

# END fix for 'xccdf_org.ssgproject.content_rule_no_legacy_plus_entries_etc_passwd'

###############################################################################
# BEGIN fix (36 / 219) for 'xccdf_org.ssgproject.content_rule_no_legacy_plus_entries_etc_shadow'
###############################################################################
(>&2 echo "Remediating rule 36/219: 'xccdf_org.ssgproject.content_rule_no_legacy_plus_entries_etc_shadow'")

if grep -q '^\+' /etc/shadow; then
# backup old file to /etc/shadow-
	cp /etc/shadow /etc/shadow-
	sed -i '/^\+.*$/d' /etc/shadow
fi

# END fix for 'xccdf_org.ssgproject.content_rule_no_legacy_plus_entries_etc_shadow'

###############################################################################
# BEGIN fix (37 / 219) for 'xccdf_org.ssgproject.content_rule_no_netrc_files'
###############################################################################
(>&2 echo "Remediating rule 37/219: 'xccdf_org.ssgproject.content_rule_no_netrc_files'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_no_netrc_files' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_no_netrc_files'

###############################################################################
# BEGIN fix (38 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_no_uid_except_zero'
###############################################################################
(>&2 echo "Remediating rule 38/219: 'xccdf_org.ssgproject.content_rule_accounts_no_uid_except_zero'")
awk -F: '$3 == 0 && $1 != "root" { print $1 }' /etc/passwd | xargs --no-run-if-empty --max-lines=1 passwd -l

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_no_uid_except_zero'

###############################################################################
# BEGIN fix (39 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_root_gid_zero'
###############################################################################
(>&2 echo "Remediating rule 39/219: 'xccdf_org.ssgproject.content_rule_accounts_root_gid_zero'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_accounts_root_gid_zero' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_root_gid_zero'

###############################################################################
# BEGIN fix (40 / 219) for 'xccdf_org.ssgproject.content_rule_no_shelllogin_for_systemaccounts'
###############################################################################
(>&2 echo "Remediating rule 40/219: 'xccdf_org.ssgproject.content_rule_no_shelllogin_for_systemaccounts'")

readarray -t systemaccounts < <(awk -F: '($3 < 1000 && $3 != root \
  && $7 != "\/sbin\/shutdown" && $7 != "\/sbin\/halt" && $7 != "\/bin\/sync") \
  { print $1 }' /etc/passwd)

for systemaccount in "${systemaccounts[@]}"; do
    usermod -s /sbin/nologin "$systemaccount"
done

# END fix for 'xccdf_org.ssgproject.content_rule_no_shelllogin_for_systemaccounts'

###############################################################################
# BEGIN fix (41 / 219) for 'xccdf_org.ssgproject.content_rule_use_pam_wheel_for_su'
###############################################################################
(>&2 echo "Remediating rule 41/219: 'xccdf_org.ssgproject.content_rule_use_pam_wheel_for_su'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_use_pam_wheel_for_su' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_use_pam_wheel_for_su'

###############################################################################
# BEGIN fix (42 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_tmout'
###############################################################################
(>&2 echo "Remediating rule 42/219: 'xccdf_org.ssgproject.content_rule_accounts_tmout'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

var_accounts_tmout='900'


# if 0, no occurence of tmout found, if 1, occurence found
tmout_found=0


for f in /etc/profile /etc/profile.d/*.sh; do

    if grep --silent '^[^#].*TMOUT' $f; then
        sed -i -E "s/^(.*)TMOUT\s*=\s*(\w|\$)*(.*)$/declare -xr TMOUT=$var_accounts_tmout\3/g" $f
        tmout_found=1
    fi
done

if [ $tmout_found -eq 0 ]; then
        echo -e "\n# Set TMOUT to $var_accounts_tmout per security requirements" >> /etc/profile.d/tmout.sh
        echo "declare -xr TMOUT=$var_accounts_tmout" >> /etc/profile.d/tmout.sh
        echo "readonly TMOUT" >> /etc/profile.d/tmout.sh
        echo "export TMOUT" >> /etc/profile.d/tmout.sh
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_tmout'

###############################################################################
# BEGIN fix (43 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_user_dot_no_world_writable_programs'
###############################################################################
(>&2 echo "Remediating rule 43/219: 'xccdf_org.ssgproject.content_rule_accounts_user_dot_no_world_writable_programs'")

readarray -t world_writable_files < <(find / -xdev -type f -perm -0002 2> /dev/null)
readarray -t interactive_home_dirs < <(awk -F':' '{ if ($3 >= 1000 && $3 != 65534) print $6 }' /etc/passwd)

for world_writable in "${world_writable_files[@]}"; do
    for homedir in "${interactive_home_dirs[@]}"; do
        if grep -q -d skip "$world_writable" "$homedir"/.*; then
            chmod o-w $world_writable
            break
        fi
    done
done

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_user_dot_no_world_writable_programs'

###############################################################################
# BEGIN fix (44 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_user_interactive_home_directory_exists'
###############################################################################
(>&2 echo "Remediating rule 44/219: 'xccdf_org.ssgproject.content_rule_accounts_user_interactive_home_directory_exists'")

for user in $(awk -F':' '{ if ($3 >= 1000 && $3 != 65534) print $1}' /etc/passwd); do
    mkhomedir_helper $user 0077;
done

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_user_interactive_home_directory_exists'

###############################################################################
# BEGIN fix (45 / 219) for 'xccdf_org.ssgproject.content_rule_file_ownership_home_directories'
###############################################################################
(>&2 echo "Remediating rule 45/219: 'xccdf_org.ssgproject.content_rule_file_ownership_home_directories'")

awk -F':' '{ if ($3 >= 1000 && $3 != 65534) system("chown -f " $3" "$6) }' /etc/passwd

# END fix for 'xccdf_org.ssgproject.content_rule_file_ownership_home_directories'

###############################################################################
# BEGIN fix (46 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_home_directories'
###############################################################################
(>&2 echo "Remediating rule 46/219: 'xccdf_org.ssgproject.content_rule_file_permissions_home_directories'")

for home_dir in $(awk -F':' '{ if ($3 >= 1000 && $3 != 65534) print $6 }' /etc/passwd); do
    # Only update the permissions when necessary. This will avoid changing the inode timestamp when
    # the permission is already defined as expected, therefore not impacting in possible integrity
    # check systems that also check inodes timestamps.
    find "$home_dir" -maxdepth 0 -perm /7027 -exec chmod u-s,g-w-s,o=- {} \;
done

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_home_directories'

###############################################################################
# BEGIN fix (47 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_root_path_dirs_no_write'
###############################################################################
(>&2 echo "Remediating rule 47/219: 'xccdf_org.ssgproject.content_rule_accounts_root_path_dirs_no_write'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_accounts_root_path_dirs_no_write' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_root_path_dirs_no_write'

###############################################################################
# BEGIN fix (48 / 219) for 'xccdf_org.ssgproject.content_rule_root_path_no_dot'
###############################################################################
(>&2 echo "Remediating rule 48/219: 'xccdf_org.ssgproject.content_rule_root_path_no_dot'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_root_path_no_dot' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_root_path_no_dot'

###############################################################################
# BEGIN fix (49 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_umask_etc_bashrc'
###############################################################################
(>&2 echo "Remediating rule 49/219: 'xccdf_org.ssgproject.content_rule_accounts_umask_etc_bashrc'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q bash; then

var_accounts_user_umask='027'






grep -q "^\s*umask" /etc/bashrc && \
  sed -i -E -e "s/^(\s*umask).*/\1 $var_accounts_user_umask/g" /etc/bashrc
if ! [ $? -eq 0 ]; then
    echo "umask $var_accounts_user_umask" >> /etc/bashrc
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_umask_etc_bashrc'

###############################################################################
# BEGIN fix (50 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_umask_etc_login_defs'
###############################################################################
(>&2 echo "Remediating rule 50/219: 'xccdf_org.ssgproject.content_rule_accounts_umask_etc_login_defs'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q shadow-utils; then

var_accounts_user_umask='027'


# Strip any search characters in the key arg so that the key can be replaced without
# adding any search characters to the config file.
stripped_key=$(sed 's/[\^=\$,;+]*//g' <<< "^UMASK")

# shellcheck disable=SC2059
printf -v formatted_output "%s %s" "$stripped_key" "$var_accounts_user_umask"

# If the key exists, change it. Otherwise, add it to the config_file.
# We search for the key string followed by a word boundary (matched by \>),
# so if we search for 'setting', 'setting2' won't match.
if LC_ALL=C grep -q -m 1 -i -e "^UMASK\\>" "/etc/login.defs"; then
    escaped_formatted_output=$(sed -e 's|/|\\/|g' <<< "$formatted_output")
    LC_ALL=C sed -i --follow-symlinks "s/^UMASK\\>.*/$escaped_formatted_output/gi" "/etc/login.defs"
else
    if [[ -s "/etc/login.defs" ]] && [[ -n "$(tail -c 1 -- "/etc/login.defs" || true)" ]]; then
        LC_ALL=C sed -i --follow-symlinks '$a'\\ "/etc/login.defs"
    fi
    printf '%s\n' "$formatted_output" >> "/etc/login.defs"
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_umask_etc_login_defs'

###############################################################################
# BEGIN fix (51 / 219) for 'xccdf_org.ssgproject.content_rule_accounts_umask_etc_profile'
###############################################################################
(>&2 echo "Remediating rule 51/219: 'xccdf_org.ssgproject.content_rule_accounts_umask_etc_profile'")

var_accounts_user_umask='027'


readarray -t profile_files < <(find /etc/profile.d/ -type f -name '*.sh' -or -name 'sh.local')

for file in "${profile_files[@]}" /etc/profile; do
  grep -qE '^[^#]*umask' "$file" && sed -i -E "s/^(\s*umask\s*)[0-7]+/\1$var_accounts_user_umask/g" "$file"
done

if ! grep -qrE '^[^#]*umask' /etc/profile*; then
  echo "umask $var_accounts_user_umask" >> /etc/profile
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_umask_etc_profile'

###############################################################################
# BEGIN fix (52 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_grub2_cfg'
###############################################################################
(>&2 echo "Remediating rule 52/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_grub2_cfg'")
# Remediation is applicable only in certain platforms
if [ ! -d /sys/firmware/efi ] && rpm --quiet -q grub2-common && { [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; }; then

chgrp 0 /boot/grub2/grub.cfg

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_grub2_cfg'

###############################################################################
# BEGIN fix (53 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_grub2_cfg'
###############################################################################
(>&2 echo "Remediating rule 53/219: 'xccdf_org.ssgproject.content_rule_file_owner_grub2_cfg'")
# Remediation is applicable only in certain platforms
if [ ! -d /sys/firmware/efi ] && rpm --quiet -q grub2-common && { [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; }; then

chown 0 /boot/grub2/grub.cfg

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_grub2_cfg'

###############################################################################
# BEGIN fix (54 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_grub2_cfg'
###############################################################################
(>&2 echo "Remediating rule 54/219: 'xccdf_org.ssgproject.content_rule_file_permissions_grub2_cfg'")
# Remediation is applicable only in certain platforms
if [ ! -d /sys/firmware/efi ] && rpm --quiet -q grub2-common && { [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; }; then

chmod u-xs,g-xwrs,o-xwrt /boot/grub2/grub.cfg

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_grub2_cfg'

###############################################################################
# BEGIN fix (55 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_efi_grub2_cfg'
###############################################################################
(>&2 echo "Remediating rule 55/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_efi_grub2_cfg'")
# Remediation is applicable only in certain platforms
if [ -d /sys/firmware/efi ] && rpm --quiet -q grub2-common && { [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; }; then

chgrp 0 /boot/grub2/grub.cfg

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_efi_grub2_cfg'

###############################################################################
# BEGIN fix (56 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_efi_grub2_cfg'
###############################################################################
(>&2 echo "Remediating rule 56/219: 'xccdf_org.ssgproject.content_rule_file_owner_efi_grub2_cfg'")
# Remediation is applicable only in certain platforms
if [ -d /sys/firmware/efi ] && rpm --quiet -q grub2-common && { [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; }; then

chown 0 /boot/grub2/grub.cfg

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_efi_grub2_cfg'

###############################################################################
# BEGIN fix (57 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_efi_grub2_cfg'
###############################################################################
(>&2 echo "Remediating rule 57/219: 'xccdf_org.ssgproject.content_rule_file_permissions_efi_grub2_cfg'")
# Remediation is applicable only in certain platforms
if [ -d /sys/firmware/efi ] && rpm --quiet -q grub2-common && { [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; }; then

chmod u-s,g-xwrs,o-xwrt /boot/grub2/grub.cfg

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_efi_grub2_cfg'

###############################################################################
# BEGIN fix (58 / 219) for 'xccdf_org.ssgproject.content_rule_package_rsyslog_installed'
###############################################################################
(>&2 echo "Remediating rule 58/219: 'xccdf_org.ssgproject.content_rule_package_rsyslog_installed'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_package_rsyslog_installed' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_package_rsyslog_installed'

###############################################################################
# BEGIN fix (59 / 219) for 'xccdf_org.ssgproject.content_rule_service_rsyslog_enabled'
###############################################################################
(>&2 echo "Remediating rule 59/219: 'xccdf_org.ssgproject.content_rule_service_rsyslog_enabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_rsyslog_enabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_rsyslog_enabled'

###############################################################################
# BEGIN fix (60 / 219) for 'xccdf_org.ssgproject.content_rule_rsyslog_files_permissions'
###############################################################################
(>&2 echo "Remediating rule 60/219: 'xccdf_org.ssgproject.content_rule_rsyslog_files_permissions'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

# List of log file paths to be inspected for correct permissions
# * Primarily inspect log file paths listed in /etc/rsyslog.conf
RSYSLOG_ETC_CONFIG="/etc/rsyslog.conf"
# * And also the log file paths listed after rsyslog's $IncludeConfig directive
#   (store the result into array for the case there's shell glob used as value of IncludeConfig)
readarray -t OLD_INC < <(grep -e "\$IncludeConfig[[:space:]]\+[^[:space:];]\+" /etc/rsyslog.conf | cut -d ' ' -f 2)
readarray -t RSYSLOG_INCLUDE_CONFIG < <(for INCPATH in "${OLD_INC[@]}"; do eval printf '%s\\n' "${INCPATH}"; done)
readarray -t NEW_INC < <(awk '/)/{f=0} /include\(/{f=1} f{nf=gensub("^(include\\(|\\s*)file=\"(\\S+)\".*","\\2",1); if($0!=nf){print nf}}' /etc/rsyslog.conf)
readarray -t RSYSLOG_INCLUDE < <(for INCPATH in "${NEW_INC[@]}"; do eval printf '%s\\n' "${INCPATH}"; done)

# Declare an array to hold the final list of different log file paths
declare -a LOG_FILE_PATHS

# Array to hold all rsyslog config entries
RSYSLOG_CONFIGS=()
RSYSLOG_CONFIGS=("${RSYSLOG_ETC_CONFIG}" "${RSYSLOG_INCLUDE_CONFIG[@]}" "${RSYSLOG_INCLUDE[@]}")

# Get full list of files to be checked
# RSYSLOG_CONFIGS may contain globs such as
# /etc/rsyslog.d/*.conf /etc/rsyslog.d/*.frule
# So, loop over the entries in RSYSLOG_CONFIGS and use find to get the list of included files.
RSYSLOG_CONFIG_FILES=()
for ENTRY in "${RSYSLOG_CONFIGS[@]}"
do
	# If directory, rsyslog will search for config files in recursively.
	# However, files in hidden sub-directories or hidden files will be ignored.
	if [ -d "${ENTRY}" ]
	then
		readarray -t FINDOUT < <(find "${ENTRY}" -not -path '*/.*' -type f)
		RSYSLOG_CONFIG_FILES+=("${FINDOUT[@]}")
	elif [ -f "${ENTRY}" ]
	then
		RSYSLOG_CONFIG_FILES+=("${ENTRY}")
	else
		echo "Invalid include object: ${ENTRY}"
	fi
done

# Browse each file selected above as containing paths of log files
# ('/etc/rsyslog.conf' and '/etc/rsyslog.d/*.conf' in the default configuration)
for LOG_FILE in "${RSYSLOG_CONFIG_FILES[@]}"
do
	# From each of these files extract just particular log file path(s), thus:
	# * Ignore lines starting with space (' '), comment ('#"), or variable syntax ('$') characters,
	# * Ignore empty lines,
	# * Strip quotes and closing brackets from paths.
	# * Ignore paths that match /dev|/etc.*\.conf, as those are paths, but likely not log files
	# * From the remaining valid rows select only fields constituting a log file path
	# Text file column is understood to represent a log file path if and only if all of the
	# following are met:
	# * it contains at least one slash '/' character,
	# * it is preceded by space
	# * it doesn't contain space (' '), colon (':'), and semicolon (';') characters
	# Search log file for path(s) only in case it exists!
	if [[ -f "${LOG_FILE}" ]]
	then
		NORMALIZED_CONFIG_FILE_LINES=$(sed -e "/^[#|$]/d" "${LOG_FILE}")
		LINES_WITH_PATHS=$(grep '[^/]*\s\+\S*/\S\+$' <<< "${NORMALIZED_CONFIG_FILE_LINES}")
		FILTERED_PATHS=$(awk '{if(NF>=2&&($NF~/^\//||$NF~/^-\//)){sub(/^-\//,"/",$NF);print $NF}}' <<< "${LINES_WITH_PATHS}")
		CLEANED_PATHS=$(sed -e "s/[\"')]//g; /\\/etc.*\.conf/d; /\\/dev\\//d" <<< "${FILTERED_PATHS}")
		MATCHED_ITEMS=$(sed -e "/^$/d" <<< "${CLEANED_PATHS}")
		# Since above sed command might return more than one item (delimited by newline), split
		# the particular matches entries into new array specific for this log file
		readarray -t ARRAY_FOR_LOG_FILE <<< "$MATCHED_ITEMS"
		# Concatenate the two arrays - previous content of $LOG_FILE_PATHS array with
		# items from newly created array for this log file
		LOG_FILE_PATHS+=("${ARRAY_FOR_LOG_FILE[@]}")
		# Delete the temporary array
		unset ARRAY_FOR_LOG_FILE
	fi
done

# Check for RainerScript action log format which might be also multiline so grep regex is a bit
# curly:
# extract possibly multiline action omfile expressions
# extract File="logfile" expression
# match only "logfile" expression
for LOG_FILE in "${RSYSLOG_CONFIG_FILES[@]}"
do
	ACTION_OMFILE_LINES=$(grep -ozP "action\s*\(\s*type\s*=\s*\"omfile\"[^\)]*\)" "${LOG_FILE}")
	OMFILE_LINES=$(echo "${ACTION_OMFILE_LINES}"| grep -aoP "File\s*=\s*\"([/[:alnum:][:punct:]]*)\"\s*\)")
	LOG_FILE_PATHS+=("$(echo "${OMFILE_LINES}"| grep -oE "\"([/[:alnum:][:punct:]]*)\""|tr -d "\"")")
done

# Ensure the correct attribute if file exists
FILE_CMD="chmod"
for LOG_FILE_PATH in "${LOG_FILE_PATHS[@]}"
do
	# Sanity check - if particular $LOG_FILE_PATH is empty string, skip it from further processing
	if [ -z "$LOG_FILE_PATH" ]
	then
		continue
	fi
	$FILE_CMD "0640" "$LOG_FILE_PATH"
done

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_rsyslog_files_permissions'

###############################################################################
# BEGIN fix (61 / 219) for 'xccdf_org.ssgproject.content_rule_rsyslog_remote_loghost'
###############################################################################
(>&2 echo "Remediating rule 61/219: 'xccdf_org.ssgproject.content_rule_rsyslog_remote_loghost'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_rsyslog_remote_loghost' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_rsyslog_remote_loghost'

###############################################################################
# BEGIN fix (62 / 219) for 'xccdf_org.ssgproject.content_rule_configured_firewalld_default_deny'
###############################################################################
(>&2 echo "Remediating rule 62/219: 'xccdf_org.ssgproject.content_rule_configured_firewalld_default_deny'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_configured_firewalld_default_deny' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_configured_firewalld_default_deny'

###############################################################################
# BEGIN fix (63 / 219) for 'xccdf_org.ssgproject.content_rule_ensure_firewall_rules_for_open_ports'
###############################################################################
(>&2 echo "Remediating rule 63/219: 'xccdf_org.ssgproject.content_rule_ensure_firewall_rules_for_open_ports'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_ensure_firewall_rules_for_open_ports' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_ensure_firewall_rules_for_open_ports'

###############################################################################
# BEGIN fix (64 / 219) for 'xccdf_org.ssgproject.content_rule_set_loopback_traffic'
###############################################################################
(>&2 echo "Remediating rule 64/219: 'xccdf_org.ssgproject.content_rule_set_loopback_traffic'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_set_loopback_traffic' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_set_loopback_traffic'

###############################################################################
# BEGIN fix (65 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_ra'
###############################################################################
(>&2 echo "Remediating rule 65/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_ra'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_ra' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_ra'

###############################################################################
# BEGIN fix (66 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_redirects'
###############################################################################
(>&2 echo "Remediating rule 66/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_redirects'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_redirects' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_redirects'

###############################################################################
# BEGIN fix (67 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_source_route'
###############################################################################
(>&2 echo "Remediating rule 67/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_source_route'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_source_route' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_accept_source_route'

###############################################################################
# BEGIN fix (68 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_forwarding'
###############################################################################
(>&2 echo "Remediating rule 68/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_forwarding'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_forwarding' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_all_forwarding'

###############################################################################
# BEGIN fix (69 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_ra'
###############################################################################
(>&2 echo "Remediating rule 69/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_ra'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_ra' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_ra'

###############################################################################
# BEGIN fix (70 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_redirects'
###############################################################################
(>&2 echo "Remediating rule 70/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_redirects'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_redirects' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_redirects'

###############################################################################
# BEGIN fix (71 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_source_route'
###############################################################################
(>&2 echo "Remediating rule 71/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_source_route'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_source_route' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv6_conf_default_accept_source_route'

###############################################################################
# BEGIN fix (72 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_accept_redirects'
###############################################################################
(>&2 echo "Remediating rule 72/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_accept_redirects'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_accept_redirects' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_accept_redirects'

###############################################################################
# BEGIN fix (73 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_accept_source_route'
###############################################################################
(>&2 echo "Remediating rule 73/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_accept_source_route'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_accept_source_route' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_accept_source_route'

###############################################################################
# BEGIN fix (74 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_log_martians'
###############################################################################
(>&2 echo "Remediating rule 74/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_log_martians'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_log_martians' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_log_martians'

###############################################################################
# BEGIN fix (75 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_rp_filter'
###############################################################################
(>&2 echo "Remediating rule 75/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_rp_filter'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_rp_filter' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_rp_filter'

###############################################################################
# BEGIN fix (76 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_secure_redirects'
###############################################################################
(>&2 echo "Remediating rule 76/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_secure_redirects'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_secure_redirects' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_secure_redirects'

###############################################################################
# BEGIN fix (77 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_accept_redirects'
###############################################################################
(>&2 echo "Remediating rule 77/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_accept_redirects'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_accept_redirects' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_accept_redirects'

###############################################################################
# BEGIN fix (78 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_accept_source_route'
###############################################################################
(>&2 echo "Remediating rule 78/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_accept_source_route'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_accept_source_route' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_accept_source_route'

###############################################################################
# BEGIN fix (79 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_log_martians'
###############################################################################
(>&2 echo "Remediating rule 79/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_log_martians'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_log_martians' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_log_martians'

###############################################################################
# BEGIN fix (80 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_rp_filter'
###############################################################################
(>&2 echo "Remediating rule 80/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_rp_filter'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_rp_filter' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_rp_filter'

###############################################################################
# BEGIN fix (81 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_secure_redirects'
###############################################################################
(>&2 echo "Remediating rule 81/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_secure_redirects'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_secure_redirects' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_secure_redirects'

###############################################################################
# BEGIN fix (82 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_icmp_echo_ignore_broadcasts'
###############################################################################
(>&2 echo "Remediating rule 82/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_icmp_echo_ignore_broadcasts'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_icmp_echo_ignore_broadcasts' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_icmp_echo_ignore_broadcasts'

###############################################################################
# BEGIN fix (83 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_icmp_ignore_bogus_error_responses'
###############################################################################
(>&2 echo "Remediating rule 83/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_icmp_ignore_bogus_error_responses'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_icmp_ignore_bogus_error_responses' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_icmp_ignore_bogus_error_responses'

###############################################################################
# BEGIN fix (84 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_tcp_syncookies'
###############################################################################
(>&2 echo "Remediating rule 84/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_tcp_syncookies'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_tcp_syncookies' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_tcp_syncookies'

###############################################################################
# BEGIN fix (85 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_send_redirects'
###############################################################################
(>&2 echo "Remediating rule 85/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_send_redirects'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_send_redirects' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_all_send_redirects'

###############################################################################
# BEGIN fix (86 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_send_redirects'
###############################################################################
(>&2 echo "Remediating rule 86/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_send_redirects'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_send_redirects' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_conf_default_send_redirects'

###############################################################################
# BEGIN fix (87 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_ip_forward'
###############################################################################
(>&2 echo "Remediating rule 87/219: 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_ip_forward'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_ip_forward' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_net_ipv4_ip_forward'

###############################################################################
# BEGIN fix (88 / 219) for 'xccdf_org.ssgproject.content_rule_kernel_module_dccp_disabled'
###############################################################################
(>&2 echo "Remediating rule 88/219: 'xccdf_org.ssgproject.content_rule_kernel_module_dccp_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_kernel_module_dccp_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_kernel_module_dccp_disabled'

###############################################################################
# BEGIN fix (89 / 219) for 'xccdf_org.ssgproject.content_rule_kernel_module_rds_disabled'
###############################################################################
(>&2 echo "Remediating rule 89/219: 'xccdf_org.ssgproject.content_rule_kernel_module_rds_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_kernel_module_rds_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_kernel_module_rds_disabled'

###############################################################################
# BEGIN fix (90 / 219) for 'xccdf_org.ssgproject.content_rule_kernel_module_sctp_disabled'
###############################################################################
(>&2 echo "Remediating rule 90/219: 'xccdf_org.ssgproject.content_rule_kernel_module_sctp_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_kernel_module_sctp_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_kernel_module_sctp_disabled'

###############################################################################
# BEGIN fix (91 / 219) for 'xccdf_org.ssgproject.content_rule_kernel_module_tipc_disabled'
###############################################################################
(>&2 echo "Remediating rule 91/219: 'xccdf_org.ssgproject.content_rule_kernel_module_tipc_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_kernel_module_tipc_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_kernel_module_tipc_disabled'

###############################################################################
# BEGIN fix (92 / 219) for 'xccdf_org.ssgproject.content_rule_dir_perms_world_writable_sticky_bits'
###############################################################################
(>&2 echo "Remediating rule 92/219: 'xccdf_org.ssgproject.content_rule_dir_perms_world_writable_sticky_bits'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_dir_perms_world_writable_sticky_bits' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_dir_perms_world_writable_sticky_bits'

###############################################################################
# BEGIN fix (93 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_world_writable'
###############################################################################
(>&2 echo "Remediating rule 93/219: 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_world_writable'")

find / -xdev -type f -perm -002 -exec chmod o-w {} \;

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_world_writable'

###############################################################################
# BEGIN fix (94 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_ungroupowned'
###############################################################################
(>&2 echo "Remediating rule 94/219: 'xccdf_org.ssgproject.content_rule_file_permissions_ungroupowned'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_file_permissions_ungroupowned' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_ungroupowned'

###############################################################################
# BEGIN fix (95 / 219) for 'xccdf_org.ssgproject.content_rule_no_files_unowned_by_user'
###############################################################################
(>&2 echo "Remediating rule 95/219: 'xccdf_org.ssgproject.content_rule_no_files_unowned_by_user'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_no_files_unowned_by_user' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_no_files_unowned_by_user'

###############################################################################
# BEGIN fix (96 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_group'
###############################################################################
(>&2 echo "Remediating rule 96/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_group'")
chgrp 0 /etc/group-

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_group'

###############################################################################
# BEGIN fix (97 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_gshadow'
###############################################################################
(>&2 echo "Remediating rule 97/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_gshadow'")
chgrp 0 /etc/gshadow-

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_gshadow'

###############################################################################
# BEGIN fix (98 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_passwd'
###############################################################################
(>&2 echo "Remediating rule 98/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_passwd'")
chgrp 0 /etc/passwd-

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_passwd'

###############################################################################
# BEGIN fix (99 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_shadow'
###############################################################################
(>&2 echo "Remediating rule 99/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_shadow'")
chgrp 0 /etc/shadow-

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_backup_etc_shadow'

###############################################################################
# BEGIN fix (100 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_group'
###############################################################################
(>&2 echo "Remediating rule 100/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_group'")
chgrp 0 /etc/group

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_group'

###############################################################################
# BEGIN fix (101 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_gshadow'
###############################################################################
(>&2 echo "Remediating rule 101/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_gshadow'")
chgrp 0 /etc/gshadow

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_gshadow'

###############################################################################
# BEGIN fix (102 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_passwd'
###############################################################################
(>&2 echo "Remediating rule 102/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_passwd'")
chgrp 0 /etc/passwd

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_passwd'

###############################################################################
# BEGIN fix (103 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_shadow'
###############################################################################
(>&2 echo "Remediating rule 103/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_shadow'")
chgrp 0 /etc/shadow

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_etc_shadow'

###############################################################################
# BEGIN fix (104 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_group'
###############################################################################
(>&2 echo "Remediating rule 104/219: 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_group'")
chown 0 /etc/group-

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_group'

###############################################################################
# BEGIN fix (105 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_gshadow'
###############################################################################
(>&2 echo "Remediating rule 105/219: 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_gshadow'")
chown 0 /etc/gshadow-

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_gshadow'

###############################################################################
# BEGIN fix (106 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_passwd'
###############################################################################
(>&2 echo "Remediating rule 106/219: 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_passwd'")
chown 0 /etc/passwd-

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_passwd'

###############################################################################
# BEGIN fix (107 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_shadow'
###############################################################################
(>&2 echo "Remediating rule 107/219: 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_shadow'")
chown 0 /etc/shadow-

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_backup_etc_shadow'

###############################################################################
# BEGIN fix (108 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_etc_group'
###############################################################################
(>&2 echo "Remediating rule 108/219: 'xccdf_org.ssgproject.content_rule_file_owner_etc_group'")
chown 0 /etc/group

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_etc_group'

###############################################################################
# BEGIN fix (109 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_etc_gshadow'
###############################################################################
(>&2 echo "Remediating rule 109/219: 'xccdf_org.ssgproject.content_rule_file_owner_etc_gshadow'")
chown 0 /etc/gshadow

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_etc_gshadow'

###############################################################################
# BEGIN fix (110 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_etc_passwd'
###############################################################################
(>&2 echo "Remediating rule 110/219: 'xccdf_org.ssgproject.content_rule_file_owner_etc_passwd'")
chown 0 /etc/passwd

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_etc_passwd'

###############################################################################
# BEGIN fix (111 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_etc_shadow'
###############################################################################
(>&2 echo "Remediating rule 111/219: 'xccdf_org.ssgproject.content_rule_file_owner_etc_shadow'")
chown 0 /etc/shadow

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_etc_shadow'

###############################################################################
# BEGIN fix (112 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_group'
###############################################################################
(>&2 echo "Remediating rule 112/219: 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_group'")





chmod u-xs,g-xws,o-xwt /etc/group-

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_group'

###############################################################################
# BEGIN fix (113 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_gshadow'
###############################################################################
(>&2 echo "Remediating rule 113/219: 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_gshadow'")





chmod u-xwrs,g-xwrs,o-xwrt /etc/gshadow-

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_gshadow'

###############################################################################
# BEGIN fix (114 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_passwd'
###############################################################################
(>&2 echo "Remediating rule 114/219: 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_passwd'")





chmod u-xs,g-xws,o-xwt /etc/passwd-

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_passwd'

###############################################################################
# BEGIN fix (115 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_shadow'
###############################################################################
(>&2 echo "Remediating rule 115/219: 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_shadow'")





chmod u-xwrs,g-xwrs,o-xwrt /etc/shadow-

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_backup_etc_shadow'

###############################################################################
# BEGIN fix (116 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_group'
###############################################################################
(>&2 echo "Remediating rule 116/219: 'xccdf_org.ssgproject.content_rule_file_permissions_etc_group'")





chmod u-xs,g-xws,o-xwt /etc/group

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_group'

###############################################################################
# BEGIN fix (117 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_gshadow'
###############################################################################
(>&2 echo "Remediating rule 117/219: 'xccdf_org.ssgproject.content_rule_file_permissions_etc_gshadow'")





chmod u-xwrs,g-xwrs,o-xwrt /etc/gshadow

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_gshadow'

###############################################################################
# BEGIN fix (118 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_passwd'
###############################################################################
(>&2 echo "Remediating rule 118/219: 'xccdf_org.ssgproject.content_rule_file_permissions_etc_passwd'")





chmod u-xs,g-xws,o-xwt /etc/passwd

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_passwd'

###############################################################################
# BEGIN fix (119 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_shadow'
###############################################################################
(>&2 echo "Remediating rule 119/219: 'xccdf_org.ssgproject.content_rule_file_permissions_etc_shadow'")





chmod u-xwrs,g-xwrs,o-xwrt /etc/shadow

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_shadow'

###############################################################################
# BEGIN fix (120 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_var_log'
###############################################################################
(>&2 echo "Remediating rule 120/219: 'xccdf_org.ssgproject.content_rule_file_permissions_var_log'")





find -H /var/log/ -maxdepth 1 -perm /u+s,g+ws,o+wt -type d -exec chmod u-s,g-ws,o-wt {} \;

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_var_log'

###############################################################################
# BEGIN fix (121 / 219) for 'xccdf_org.ssgproject.content_rule_service_autofs_disabled'
###############################################################################
(>&2 echo "Remediating rule 121/219: 'xccdf_org.ssgproject.content_rule_service_autofs_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_autofs_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_autofs_disabled'

###############################################################################
# BEGIN fix (122 / 219) for 'xccdf_org.ssgproject.content_rule_kernel_module_squashfs_disabled'
###############################################################################
(>&2 echo "Remediating rule 122/219: 'xccdf_org.ssgproject.content_rule_kernel_module_squashfs_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_kernel_module_squashfs_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_kernel_module_squashfs_disabled'

###############################################################################
# BEGIN fix (123 / 219) for 'xccdf_org.ssgproject.content_rule_mount_option_dev_shm_nodev'
###############################################################################
(>&2 echo "Remediating rule 123/219: 'xccdf_org.ssgproject.content_rule_mount_option_dev_shm_nodev'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

function perform_remediation {
    


    mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" /dev/shm)"

    # If the mount point is not in /etc/fstab, get previous mount options from /etc/mtab
    if ! grep -q "$mount_point_match_regexp" /etc/fstab; then
        # runtime opts without some automatic kernel/userspace-added defaults
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/mtab | head -1 |  awk '{print $4}' \
                    | sed -E "s/(rw|defaults|seclabel|nodev)(,|$)//g;s/,$//")
        [ "$previous_mount_opts" ] && previous_mount_opts+=","
        # In iso9660 filesystems mtab could describe a "blocksize" value, this should be reflected in
        # fstab as "block".  The next variable is to satisfy shellcheck SC2050.
        fs_type="tmpfs"
        if [  "$fs_type" == "iso9660" ] ; then
            previous_mount_opts=$(sed 's/blocksize=/block=/' <<< "$previous_mount_opts")
        fi
        echo "tmpfs /dev/shm tmpfs defaults,${previous_mount_opts}nodev 0 0" >> /etc/fstab
    # If the mount_opt option is not already in the mount point's /etc/fstab entry, add it
    elif ! grep "$mount_point_match_regexp" /etc/fstab | grep -q "nodev"; then
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/fstab | awk '{print $4}')
        sed -i "s|\(${mount_point_match_regexp}.*${previous_mount_opts}\)|\1,nodev|" /etc/fstab
    fi


    if mkdir -p "/dev/shm"; then
        if mountpoint -q "/dev/shm"; then
            mount -o remount --target "/dev/shm"
        fi
    fi
}

perform_remediation

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_mount_option_dev_shm_nodev'

###############################################################################
# BEGIN fix (124 / 219) for 'xccdf_org.ssgproject.content_rule_mount_option_dev_shm_noexec'
###############################################################################
(>&2 echo "Remediating rule 124/219: 'xccdf_org.ssgproject.content_rule_mount_option_dev_shm_noexec'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

function perform_remediation {
    


    mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" /dev/shm)"

    # If the mount point is not in /etc/fstab, get previous mount options from /etc/mtab
    if ! grep -q "$mount_point_match_regexp" /etc/fstab; then
        # runtime opts without some automatic kernel/userspace-added defaults
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/mtab | head -1 |  awk '{print $4}' \
                    | sed -E "s/(rw|defaults|seclabel|noexec)(,|$)//g;s/,$//")
        [ "$previous_mount_opts" ] && previous_mount_opts+=","
        # In iso9660 filesystems mtab could describe a "blocksize" value, this should be reflected in
        # fstab as "block".  The next variable is to satisfy shellcheck SC2050.
        fs_type="tmpfs"
        if [  "$fs_type" == "iso9660" ] ; then
            previous_mount_opts=$(sed 's/blocksize=/block=/' <<< "$previous_mount_opts")
        fi
        echo "tmpfs /dev/shm tmpfs defaults,${previous_mount_opts}noexec 0 0" >> /etc/fstab
    # If the mount_opt option is not already in the mount point's /etc/fstab entry, add it
    elif ! grep "$mount_point_match_regexp" /etc/fstab | grep -q "noexec"; then
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/fstab | awk '{print $4}')
        sed -i "s|\(${mount_point_match_regexp}.*${previous_mount_opts}\)|\1,noexec|" /etc/fstab
    fi


    if mkdir -p "/dev/shm"; then
        if mountpoint -q "/dev/shm"; then
            mount -o remount --target "/dev/shm"
        fi
    fi
}

perform_remediation

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_mount_option_dev_shm_noexec'

###############################################################################
# BEGIN fix (125 / 219) for 'xccdf_org.ssgproject.content_rule_mount_option_dev_shm_nosuid'
###############################################################################
(>&2 echo "Remediating rule 125/219: 'xccdf_org.ssgproject.content_rule_mount_option_dev_shm_nosuid'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

function perform_remediation {
    


    mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" /dev/shm)"

    # If the mount point is not in /etc/fstab, get previous mount options from /etc/mtab
    if ! grep -q "$mount_point_match_regexp" /etc/fstab; then
        # runtime opts without some automatic kernel/userspace-added defaults
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/mtab | head -1 |  awk '{print $4}' \
                    | sed -E "s/(rw|defaults|seclabel|nosuid)(,|$)//g;s/,$//")
        [ "$previous_mount_opts" ] && previous_mount_opts+=","
        # In iso9660 filesystems mtab could describe a "blocksize" value, this should be reflected in
        # fstab as "block".  The next variable is to satisfy shellcheck SC2050.
        fs_type="tmpfs"
        if [  "$fs_type" == "iso9660" ] ; then
            previous_mount_opts=$(sed 's/blocksize=/block=/' <<< "$previous_mount_opts")
        fi
        echo "tmpfs /dev/shm tmpfs defaults,${previous_mount_opts}nosuid 0 0" >> /etc/fstab
    # If the mount_opt option is not already in the mount point's /etc/fstab entry, add it
    elif ! grep "$mount_point_match_regexp" /etc/fstab | grep -q "nosuid"; then
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/fstab | awk '{print $4}')
        sed -i "s|\(${mount_point_match_regexp}.*${previous_mount_opts}\)|\1,nosuid|" /etc/fstab
    fi


    if mkdir -p "/dev/shm"; then
        if mountpoint -q "/dev/shm"; then
            mount -o remount --target "/dev/shm"
        fi
    fi
}

perform_remediation

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_mount_option_dev_shm_nosuid'

###############################################################################
# BEGIN fix (126 / 219) for 'xccdf_org.ssgproject.content_rule_mount_option_home_nodev'
###############################################################################
(>&2 echo "Remediating rule 126/219: 'xccdf_org.ssgproject.content_rule_mount_option_home_nodev'")
# Remediation is applicable only in certain platforms
if ( [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ] && findmnt --kernel "/home" > /dev/null || findmnt --fstab "/home" > /dev/null ); then

function perform_remediation {
    
        mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" "/home")"

    grep "$mount_point_match_regexp" -q /etc/fstab \
        || { echo "The mount point '/home' is not even in /etc/fstab, so we can't set up mount options" >&2;
                echo "Not remediating, because there is no record of /home in /etc/fstab" >&2; return 1; }
    


    mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" /home)"

    # If the mount point is not in /etc/fstab, get previous mount options from /etc/mtab
    if ! grep -q "$mount_point_match_regexp" /etc/fstab; then
        # runtime opts without some automatic kernel/userspace-added defaults
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/mtab | head -1 |  awk '{print $4}' \
                    | sed -E "s/(rw|defaults|seclabel|nodev)(,|$)//g;s/,$//")
        [ "$previous_mount_opts" ] && previous_mount_opts+=","
        # In iso9660 filesystems mtab could describe a "blocksize" value, this should be reflected in
        # fstab as "block".  The next variable is to satisfy shellcheck SC2050.
        fs_type=""
        if [  "$fs_type" == "iso9660" ] ; then
            previous_mount_opts=$(sed 's/blocksize=/block=/' <<< "$previous_mount_opts")
        fi
        echo " /home  defaults,${previous_mount_opts}nodev 0 0" >> /etc/fstab
    # If the mount_opt option is not already in the mount point's /etc/fstab entry, add it
    elif ! grep "$mount_point_match_regexp" /etc/fstab | grep -q "nodev"; then
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/fstab | awk '{print $4}')
        sed -i "s|\(${mount_point_match_regexp}.*${previous_mount_opts}\)|\1,nodev|" /etc/fstab
    fi


    if mkdir -p "/home"; then
        if mountpoint -q "/home"; then
            mount -o remount --target "/home"
        fi
    fi
}

perform_remediation

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_mount_option_home_nodev'

###############################################################################
# BEGIN fix (127 / 219) for 'xccdf_org.ssgproject.content_rule_mount_option_tmp_nodev'
###############################################################################
(>&2 echo "Remediating rule 127/219: 'xccdf_org.ssgproject.content_rule_mount_option_tmp_nodev'")
# Remediation is applicable only in certain platforms
if ( [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ] && findmnt --kernel "/tmp" > /dev/null || findmnt --fstab "/tmp" > /dev/null ); then

function perform_remediation {
    
        mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" "/tmp")"

    grep "$mount_point_match_regexp" -q /etc/fstab \
        || { echo "The mount point '/tmp' is not even in /etc/fstab, so we can't set up mount options" >&2;
                echo "Not remediating, because there is no record of /tmp in /etc/fstab" >&2; return 1; }
    


    mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" /tmp)"

    # If the mount point is not in /etc/fstab, get previous mount options from /etc/mtab
    if ! grep -q "$mount_point_match_regexp" /etc/fstab; then
        # runtime opts without some automatic kernel/userspace-added defaults
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/mtab | head -1 |  awk '{print $4}' \
                    | sed -E "s/(rw|defaults|seclabel|nodev)(,|$)//g;s/,$//")
        [ "$previous_mount_opts" ] && previous_mount_opts+=","
        # In iso9660 filesystems mtab could describe a "blocksize" value, this should be reflected in
        # fstab as "block".  The next variable is to satisfy shellcheck SC2050.
        fs_type=""
        if [  "$fs_type" == "iso9660" ] ; then
            previous_mount_opts=$(sed 's/blocksize=/block=/' <<< "$previous_mount_opts")
        fi
        echo " /tmp  defaults,${previous_mount_opts}nodev 0 0" >> /etc/fstab
    # If the mount_opt option is not already in the mount point's /etc/fstab entry, add it
    elif ! grep "$mount_point_match_regexp" /etc/fstab | grep -q "nodev"; then
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/fstab | awk '{print $4}')
        sed -i "s|\(${mount_point_match_regexp}.*${previous_mount_opts}\)|\1,nodev|" /etc/fstab
    fi


    if mkdir -p "/tmp"; then
        if mountpoint -q "/tmp"; then
            mount -o remount --target "/tmp"
        fi
    fi
}

perform_remediation

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_mount_option_tmp_nodev'

###############################################################################
# BEGIN fix (128 / 219) for 'xccdf_org.ssgproject.content_rule_mount_option_tmp_noexec'
###############################################################################
(>&2 echo "Remediating rule 128/219: 'xccdf_org.ssgproject.content_rule_mount_option_tmp_noexec'")
# Remediation is applicable only in certain platforms
if ( [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ] && findmnt --kernel "/tmp" > /dev/null || findmnt --fstab "/tmp" > /dev/null ); then

function perform_remediation {
    
        mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" "/tmp")"

    grep "$mount_point_match_regexp" -q /etc/fstab \
        || { echo "The mount point '/tmp' is not even in /etc/fstab, so we can't set up mount options" >&2;
                echo "Not remediating, because there is no record of /tmp in /etc/fstab" >&2; return 1; }
    


    mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" /tmp)"

    # If the mount point is not in /etc/fstab, get previous mount options from /etc/mtab
    if ! grep -q "$mount_point_match_regexp" /etc/fstab; then
        # runtime opts without some automatic kernel/userspace-added defaults
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/mtab | head -1 |  awk '{print $4}' \
                    | sed -E "s/(rw|defaults|seclabel|noexec)(,|$)//g;s/,$//")
        [ "$previous_mount_opts" ] && previous_mount_opts+=","
        # In iso9660 filesystems mtab could describe a "blocksize" value, this should be reflected in
        # fstab as "block".  The next variable is to satisfy shellcheck SC2050.
        fs_type=""
        if [  "$fs_type" == "iso9660" ] ; then
            previous_mount_opts=$(sed 's/blocksize=/block=/' <<< "$previous_mount_opts")
        fi
        echo " /tmp  defaults,${previous_mount_opts}noexec 0 0" >> /etc/fstab
    # If the mount_opt option is not already in the mount point's /etc/fstab entry, add it
    elif ! grep "$mount_point_match_regexp" /etc/fstab | grep -q "noexec"; then
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/fstab | awk '{print $4}')
        sed -i "s|\(${mount_point_match_regexp}.*${previous_mount_opts}\)|\1,noexec|" /etc/fstab
    fi


    if mkdir -p "/tmp"; then
        if mountpoint -q "/tmp"; then
            mount -o remount --target "/tmp"
        fi
    fi
}

perform_remediation

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_mount_option_tmp_noexec'

###############################################################################
# BEGIN fix (129 / 219) for 'xccdf_org.ssgproject.content_rule_mount_option_tmp_nosuid'
###############################################################################
(>&2 echo "Remediating rule 129/219: 'xccdf_org.ssgproject.content_rule_mount_option_tmp_nosuid'")
# Remediation is applicable only in certain platforms
if ( [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ] && findmnt --kernel "/tmp" > /dev/null || findmnt --fstab "/tmp" > /dev/null ); then

function perform_remediation {
    
        mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" "/tmp")"

    grep "$mount_point_match_regexp" -q /etc/fstab \
        || { echo "The mount point '/tmp' is not even in /etc/fstab, so we can't set up mount options" >&2;
                echo "Not remediating, because there is no record of /tmp in /etc/fstab" >&2; return 1; }
    


    mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" /tmp)"

    # If the mount point is not in /etc/fstab, get previous mount options from /etc/mtab
    if ! grep -q "$mount_point_match_regexp" /etc/fstab; then
        # runtime opts without some automatic kernel/userspace-added defaults
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/mtab | head -1 |  awk '{print $4}' \
                    | sed -E "s/(rw|defaults|seclabel|nosuid)(,|$)//g;s/,$//")
        [ "$previous_mount_opts" ] && previous_mount_opts+=","
        # In iso9660 filesystems mtab could describe a "blocksize" value, this should be reflected in
        # fstab as "block".  The next variable is to satisfy shellcheck SC2050.
        fs_type=""
        if [  "$fs_type" == "iso9660" ] ; then
            previous_mount_opts=$(sed 's/blocksize=/block=/' <<< "$previous_mount_opts")
        fi
        echo " /tmp  defaults,${previous_mount_opts}nosuid 0 0" >> /etc/fstab
    # If the mount_opt option is not already in the mount point's /etc/fstab entry, add it
    elif ! grep "$mount_point_match_regexp" /etc/fstab | grep -q "nosuid"; then
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/fstab | awk '{print $4}')
        sed -i "s|\(${mount_point_match_regexp}.*${previous_mount_opts}\)|\1,nosuid|" /etc/fstab
    fi


    if mkdir -p "/tmp"; then
        if mountpoint -q "/tmp"; then
            mount -o remount --target "/tmp"
        fi
    fi
}

perform_remediation

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_mount_option_tmp_nosuid'

###############################################################################
# BEGIN fix (130 / 219) for 'xccdf_org.ssgproject.content_rule_mount_option_var_tmp_nodev'
###############################################################################
(>&2 echo "Remediating rule 130/219: 'xccdf_org.ssgproject.content_rule_mount_option_var_tmp_nodev'")
# Remediation is applicable only in certain platforms
if ( [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ] && findmnt --kernel "/var/tmp" > /dev/null || findmnt --fstab "/var/tmp" > /dev/null ); then

function perform_remediation {
    
        mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" "/var/tmp")"

    grep "$mount_point_match_regexp" -q /etc/fstab \
        || { echo "The mount point '/var/tmp' is not even in /etc/fstab, so we can't set up mount options" >&2;
                echo "Not remediating, because there is no record of /var/tmp in /etc/fstab" >&2; return 1; }
    


    mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" /var/tmp)"

    # If the mount point is not in /etc/fstab, get previous mount options from /etc/mtab
    if ! grep -q "$mount_point_match_regexp" /etc/fstab; then
        # runtime opts without some automatic kernel/userspace-added defaults
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/mtab | head -1 |  awk '{print $4}' \
                    | sed -E "s/(rw|defaults|seclabel|nodev)(,|$)//g;s/,$//")
        [ "$previous_mount_opts" ] && previous_mount_opts+=","
        # In iso9660 filesystems mtab could describe a "blocksize" value, this should be reflected in
        # fstab as "block".  The next variable is to satisfy shellcheck SC2050.
        fs_type=""
        if [  "$fs_type" == "iso9660" ] ; then
            previous_mount_opts=$(sed 's/blocksize=/block=/' <<< "$previous_mount_opts")
        fi
        echo " /var/tmp  defaults,${previous_mount_opts}nodev 0 0" >> /etc/fstab
    # If the mount_opt option is not already in the mount point's /etc/fstab entry, add it
    elif ! grep "$mount_point_match_regexp" /etc/fstab | grep -q "nodev"; then
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/fstab | awk '{print $4}')
        sed -i "s|\(${mount_point_match_regexp}.*${previous_mount_opts}\)|\1,nodev|" /etc/fstab
    fi


    if mkdir -p "/var/tmp"; then
        if mountpoint -q "/var/tmp"; then
            mount -o remount --target "/var/tmp"
        fi
    fi
}

perform_remediation

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_mount_option_var_tmp_nodev'

###############################################################################
# BEGIN fix (131 / 219) for 'xccdf_org.ssgproject.content_rule_mount_option_var_tmp_noexec'
###############################################################################
(>&2 echo "Remediating rule 131/219: 'xccdf_org.ssgproject.content_rule_mount_option_var_tmp_noexec'")
# Remediation is applicable only in certain platforms
if ( [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ] && findmnt --kernel "/var/tmp" > /dev/null || findmnt --fstab "/var/tmp" > /dev/null ); then

function perform_remediation {
    
        mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" "/var/tmp")"

    grep "$mount_point_match_regexp" -q /etc/fstab \
        || { echo "The mount point '/var/tmp' is not even in /etc/fstab, so we can't set up mount options" >&2;
                echo "Not remediating, because there is no record of /var/tmp in /etc/fstab" >&2; return 1; }
    


    mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" /var/tmp)"

    # If the mount point is not in /etc/fstab, get previous mount options from /etc/mtab
    if ! grep -q "$mount_point_match_regexp" /etc/fstab; then
        # runtime opts without some automatic kernel/userspace-added defaults
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/mtab | head -1 |  awk '{print $4}' \
                    | sed -E "s/(rw|defaults|seclabel|noexec)(,|$)//g;s/,$//")
        [ "$previous_mount_opts" ] && previous_mount_opts+=","
        # In iso9660 filesystems mtab could describe a "blocksize" value, this should be reflected in
        # fstab as "block".  The next variable is to satisfy shellcheck SC2050.
        fs_type=""
        if [  "$fs_type" == "iso9660" ] ; then
            previous_mount_opts=$(sed 's/blocksize=/block=/' <<< "$previous_mount_opts")
        fi
        echo " /var/tmp  defaults,${previous_mount_opts}noexec 0 0" >> /etc/fstab
    # If the mount_opt option is not already in the mount point's /etc/fstab entry, add it
    elif ! grep "$mount_point_match_regexp" /etc/fstab | grep -q "noexec"; then
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/fstab | awk '{print $4}')
        sed -i "s|\(${mount_point_match_regexp}.*${previous_mount_opts}\)|\1,noexec|" /etc/fstab
    fi


    if mkdir -p "/var/tmp"; then
        if mountpoint -q "/var/tmp"; then
            mount -o remount --target "/var/tmp"
        fi
    fi
}

perform_remediation

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_mount_option_var_tmp_noexec'

###############################################################################
# BEGIN fix (132 / 219) for 'xccdf_org.ssgproject.content_rule_mount_option_var_tmp_nosuid'
###############################################################################
(>&2 echo "Remediating rule 132/219: 'xccdf_org.ssgproject.content_rule_mount_option_var_tmp_nosuid'")
# Remediation is applicable only in certain platforms
if ( [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ] && findmnt --kernel "/var/tmp" > /dev/null || findmnt --fstab "/var/tmp" > /dev/null ); then

function perform_remediation {
    
        mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" "/var/tmp")"

    grep "$mount_point_match_regexp" -q /etc/fstab \
        || { echo "The mount point '/var/tmp' is not even in /etc/fstab, so we can't set up mount options" >&2;
                echo "Not remediating, because there is no record of /var/tmp in /etc/fstab" >&2; return 1; }
    


    mount_point_match_regexp="$(printf "^[[:space:]]*[^#].*[[:space:]]%s[[:space:]]" /var/tmp)"

    # If the mount point is not in /etc/fstab, get previous mount options from /etc/mtab
    if ! grep -q "$mount_point_match_regexp" /etc/fstab; then
        # runtime opts without some automatic kernel/userspace-added defaults
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/mtab | head -1 |  awk '{print $4}' \
                    | sed -E "s/(rw|defaults|seclabel|nosuid)(,|$)//g;s/,$//")
        [ "$previous_mount_opts" ] && previous_mount_opts+=","
        # In iso9660 filesystems mtab could describe a "blocksize" value, this should be reflected in
        # fstab as "block".  The next variable is to satisfy shellcheck SC2050.
        fs_type=""
        if [  "$fs_type" == "iso9660" ] ; then
            previous_mount_opts=$(sed 's/blocksize=/block=/' <<< "$previous_mount_opts")
        fi
        echo " /var/tmp  defaults,${previous_mount_opts}nosuid 0 0" >> /etc/fstab
    # If the mount_opt option is not already in the mount point's /etc/fstab entry, add it
    elif ! grep "$mount_point_match_regexp" /etc/fstab | grep -q "nosuid"; then
        previous_mount_opts=$(grep "$mount_point_match_regexp" /etc/fstab | awk '{print $4}')
        sed -i "s|\(${mount_point_match_regexp}.*${previous_mount_opts}\)|\1,nosuid|" /etc/fstab
    fi


    if mkdir -p "/var/tmp"; then
        if mountpoint -q "/var/tmp"; then
            mount -o remount --target "/var/tmp"
        fi
    fi
}

perform_remediation

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_mount_option_var_tmp_nosuid'

###############################################################################
# BEGIN fix (133 / 219) for 'xccdf_org.ssgproject.content_rule_coredump_disable_backtraces'
###############################################################################
(>&2 echo "Remediating rule 133/219: 'xccdf_org.ssgproject.content_rule_coredump_disable_backtraces'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q systemd; then

if [ -e "/etc/systemd/coredump.conf" ] ; then
    
    LC_ALL=C sed -i "/^\s*ProcessSizeMax\s*=\s*/Id" "/etc/systemd/coredump.conf"
else
    touch "/etc/systemd/coredump.conf"
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/systemd/coredump.conf"

cp "/etc/systemd/coredump.conf" "/etc/systemd/coredump.conf.bak"
# Insert at the end of the file
printf '%s\n' "ProcessSizeMax=0" >> "/etc/systemd/coredump.conf"
# Clean up after ourselves.
rm "/etc/systemd/coredump.conf.bak"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_coredump_disable_backtraces'

###############################################################################
# BEGIN fix (134 / 219) for 'xccdf_org.ssgproject.content_rule_coredump_disable_storage'
###############################################################################
(>&2 echo "Remediating rule 134/219: 'xccdf_org.ssgproject.content_rule_coredump_disable_storage'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q systemd; then

if [ -e "/etc/systemd/coredump.conf" ] ; then
    
    LC_ALL=C sed -i "/^\s*Storage\s*=\s*/Id" "/etc/systemd/coredump.conf"
else
    touch "/etc/systemd/coredump.conf"
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/systemd/coredump.conf"

cp "/etc/systemd/coredump.conf" "/etc/systemd/coredump.conf.bak"
# Insert at the end of the file
printf '%s\n' "Storage=none" >> "/etc/systemd/coredump.conf"
# Clean up after ourselves.
rm "/etc/systemd/coredump.conf.bak"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_coredump_disable_storage'

###############################################################################
# BEGIN fix (135 / 219) for 'xccdf_org.ssgproject.content_rule_disable_users_coredumps'
###############################################################################
(>&2 echo "Remediating rule 135/219: 'xccdf_org.ssgproject.content_rule_disable_users_coredumps'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_disable_users_coredumps' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_disable_users_coredumps'

###############################################################################
# BEGIN fix (136 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_fs_suid_dumpable'
###############################################################################
(>&2 echo "Remediating rule 136/219: 'xccdf_org.ssgproject.content_rule_sysctl_fs_suid_dumpable'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_fs_suid_dumpable' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_fs_suid_dumpable'

###############################################################################
# BEGIN fix (137 / 219) for 'xccdf_org.ssgproject.content_rule_sysctl_kernel_randomize_va_space'
###############################################################################
(>&2 echo "Remediating rule 137/219: 'xccdf_org.ssgproject.content_rule_sysctl_kernel_randomize_va_space'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sysctl_kernel_randomize_va_space' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sysctl_kernel_randomize_va_space'

###############################################################################
# BEGIN fix (138 / 219) for 'xccdf_org.ssgproject.content_rule_service_avahi-daemon_disabled'
###############################################################################
(>&2 echo "Remediating rule 138/219: 'xccdf_org.ssgproject.content_rule_service_avahi-daemon_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_avahi-daemon_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_avahi-daemon_disabled'

###############################################################################
# BEGIN fix (139 / 219) for 'xccdf_org.ssgproject.content_rule_service_crond_enabled'
###############################################################################
(>&2 echo "Remediating rule 139/219: 'xccdf_org.ssgproject.content_rule_service_crond_enabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_crond_enabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_crond_enabled'

###############################################################################
# BEGIN fix (140 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_d'
###############################################################################
(>&2 echo "Remediating rule 140/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_d'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.d/ -maxdepth 1 -type d -exec chgrp 0 {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_d'

###############################################################################
# BEGIN fix (141 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_daily'
###############################################################################
(>&2 echo "Remediating rule 141/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_daily'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.daily/ -maxdepth 1 -type d -exec chgrp 0 {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_daily'

###############################################################################
# BEGIN fix (142 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_hourly'
###############################################################################
(>&2 echo "Remediating rule 142/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_hourly'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.hourly/ -maxdepth 1 -type d -exec chgrp 0 {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_hourly'

###############################################################################
# BEGIN fix (143 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_monthly'
###############################################################################
(>&2 echo "Remediating rule 143/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_monthly'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.monthly/ -maxdepth 1 -type d -exec chgrp 0 {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_monthly'

###############################################################################
# BEGIN fix (144 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_weekly'
###############################################################################
(>&2 echo "Remediating rule 144/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_weekly'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.weekly/ -maxdepth 1 -type d -exec chgrp 0 {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_weekly'

###############################################################################
# BEGIN fix (145 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_crontab'
###############################################################################
(>&2 echo "Remediating rule 145/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_crontab'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chgrp 0 /etc/crontab

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_crontab'

###############################################################################
# BEGIN fix (146 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_cron_d'
###############################################################################
(>&2 echo "Remediating rule 146/219: 'xccdf_org.ssgproject.content_rule_file_owner_cron_d'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.d/ -maxdepth 1 -type d -exec chown 0 {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_cron_d'

###############################################################################
# BEGIN fix (147 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_cron_daily'
###############################################################################
(>&2 echo "Remediating rule 147/219: 'xccdf_org.ssgproject.content_rule_file_owner_cron_daily'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.daily/ -maxdepth 1 -type d -exec chown 0 {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_cron_daily'

###############################################################################
# BEGIN fix (148 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_cron_hourly'
###############################################################################
(>&2 echo "Remediating rule 148/219: 'xccdf_org.ssgproject.content_rule_file_owner_cron_hourly'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.hourly/ -maxdepth 1 -type d -exec chown 0 {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_cron_hourly'

###############################################################################
# BEGIN fix (149 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_cron_monthly'
###############################################################################
(>&2 echo "Remediating rule 149/219: 'xccdf_org.ssgproject.content_rule_file_owner_cron_monthly'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.monthly/ -maxdepth 1 -type d -exec chown 0 {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_cron_monthly'

###############################################################################
# BEGIN fix (150 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_cron_weekly'
###############################################################################
(>&2 echo "Remediating rule 150/219: 'xccdf_org.ssgproject.content_rule_file_owner_cron_weekly'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.weekly/ -maxdepth 1 -type d -exec chown 0 {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_cron_weekly'

###############################################################################
# BEGIN fix (151 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_crontab'
###############################################################################
(>&2 echo "Remediating rule 151/219: 'xccdf_org.ssgproject.content_rule_file_owner_crontab'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chown 0 /etc/crontab

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_crontab'

###############################################################################
# BEGIN fix (152 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_d'
###############################################################################
(>&2 echo "Remediating rule 152/219: 'xccdf_org.ssgproject.content_rule_file_permissions_cron_d'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.d/ -maxdepth 1 -perm /u+s,g+xwrs,o+xwrt -type d -exec chmod u-s,g-xwrs,o-xwrt {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_d'

###############################################################################
# BEGIN fix (153 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_daily'
###############################################################################
(>&2 echo "Remediating rule 153/219: 'xccdf_org.ssgproject.content_rule_file_permissions_cron_daily'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.daily/ -maxdepth 1 -perm /u+s,g+xwrs,o+xwrt -type d -exec chmod u-s,g-xwrs,o-xwrt {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_daily'

###############################################################################
# BEGIN fix (154 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_hourly'
###############################################################################
(>&2 echo "Remediating rule 154/219: 'xccdf_org.ssgproject.content_rule_file_permissions_cron_hourly'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.hourly/ -maxdepth 1 -perm /u+s,g+xwrs,o+xwrt -type d -exec chmod u-s,g-xwrs,o-xwrt {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_hourly'

###############################################################################
# BEGIN fix (155 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_monthly'
###############################################################################
(>&2 echo "Remediating rule 155/219: 'xccdf_org.ssgproject.content_rule_file_permissions_cron_monthly'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.monthly/ -maxdepth 1 -perm /u+s,g+xwrs,o+xwrt -type d -exec chmod u-s,g-xwrs,o-xwrt {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_monthly'

###############################################################################
# BEGIN fix (156 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_weekly'
###############################################################################
(>&2 echo "Remediating rule 156/219: 'xccdf_org.ssgproject.content_rule_file_permissions_cron_weekly'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/cron.weekly/ -maxdepth 1 -perm /u+s,g+xwrs,o+xwrt -type d -exec chmod u-s,g-xwrs,o-xwrt {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_weekly'

###############################################################################
# BEGIN fix (157 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_crontab'
###############################################################################
(>&2 echo "Remediating rule 157/219: 'xccdf_org.ssgproject.content_rule_file_permissions_crontab'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chmod u-xs,g-xwrs,o-xwrt /etc/crontab

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_crontab'

###############################################################################
# BEGIN fix (158 / 219) for 'xccdf_org.ssgproject.content_rule_file_at_deny_not_exist'
###############################################################################
(>&2 echo "Remediating rule 158/219: 'xccdf_org.ssgproject.content_rule_file_at_deny_not_exist'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

#!/bin/bash



    if [[ -f  /etc/at.deny ]]; then
        rm /etc/at.deny
    fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_at_deny_not_exist'

###############################################################################
# BEGIN fix (159 / 219) for 'xccdf_org.ssgproject.content_rule_file_cron_deny_not_exist'
###############################################################################
(>&2 echo "Remediating rule 159/219: 'xccdf_org.ssgproject.content_rule_file_cron_deny_not_exist'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

#!/bin/bash



    if [[ -f  /etc/cron.deny ]]; then
        rm /etc/cron.deny
    fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_cron_deny_not_exist'

###############################################################################
# BEGIN fix (160 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_at_allow'
###############################################################################
(>&2 echo "Remediating rule 160/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_at_allow'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chgrp 0 /etc/at.allow

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_at_allow'

###############################################################################
# BEGIN fix (161 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_allow'
###############################################################################
(>&2 echo "Remediating rule 161/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_allow'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chgrp 0 /etc/cron.allow

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_cron_allow'

###############################################################################
# BEGIN fix (162 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_at_allow'
###############################################################################
(>&2 echo "Remediating rule 162/219: 'xccdf_org.ssgproject.content_rule_file_owner_at_allow'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chown 0 /etc/at.allow

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_at_allow'

###############################################################################
# BEGIN fix (163 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_cron_allow'
###############################################################################
(>&2 echo "Remediating rule 163/219: 'xccdf_org.ssgproject.content_rule_file_owner_cron_allow'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chown 0 /etc/cron.allow

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_cron_allow'

###############################################################################
# BEGIN fix (164 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_at_allow'
###############################################################################
(>&2 echo "Remediating rule 164/219: 'xccdf_org.ssgproject.content_rule_file_permissions_at_allow'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chmod u-xs,g-xws,o-xwrt /etc/at.allow

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_at_allow'

###############################################################################
# BEGIN fix (165 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_allow'
###############################################################################
(>&2 echo "Remediating rule 165/219: 'xccdf_org.ssgproject.content_rule_file_permissions_cron_allow'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chmod u-xs,g-xws,o-xwrt /etc/cron.allow

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_cron_allow'

###############################################################################
# BEGIN fix (166 / 219) for 'xccdf_org.ssgproject.content_rule_service_dhcpd_disabled'
###############################################################################
(>&2 echo "Remediating rule 166/219: 'xccdf_org.ssgproject.content_rule_service_dhcpd_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_dhcpd_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_dhcpd_disabled'

###############################################################################
# BEGIN fix (167 / 219) for 'xccdf_org.ssgproject.content_rule_service_named_disabled'
###############################################################################
(>&2 echo "Remediating rule 167/219: 'xccdf_org.ssgproject.content_rule_service_named_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_named_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_named_disabled'

###############################################################################
# BEGIN fix (168 / 219) for 'xccdf_org.ssgproject.content_rule_service_vsftpd_disabled'
###############################################################################
(>&2 echo "Remediating rule 168/219: 'xccdf_org.ssgproject.content_rule_service_vsftpd_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_vsftpd_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_vsftpd_disabled'

###############################################################################
# BEGIN fix (169 / 219) for 'xccdf_org.ssgproject.content_rule_service_httpd_disabled'
###############################################################################
(>&2 echo "Remediating rule 169/219: 'xccdf_org.ssgproject.content_rule_service_httpd_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_httpd_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_httpd_disabled'

###############################################################################
# BEGIN fix (170 / 219) for 'xccdf_org.ssgproject.content_rule_package_dovecot_removed'
###############################################################################
(>&2 echo "Remediating rule 170/219: 'xccdf_org.ssgproject.content_rule_package_dovecot_removed'")

# CAUTION: This remediation script will remove dovecot
#	   from the system, and may remove any packages
#	   that depend on dovecot. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

if rpm -q --quiet "dovecot" ; then

    yum remove -y "dovecot"

fi

# END fix for 'xccdf_org.ssgproject.content_rule_package_dovecot_removed'

###############################################################################
# BEGIN fix (171 / 219) for 'xccdf_org.ssgproject.content_rule_package_openldap-clients_removed'
###############################################################################
(>&2 echo "Remediating rule 171/219: 'xccdf_org.ssgproject.content_rule_package_openldap-clients_removed'")

# CAUTION: This remediation script will remove openldap-clients
#	   from the system, and may remove any packages
#	   that depend on openldap-clients. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

if rpm -q --quiet "openldap-clients" ; then

    yum remove -y "openldap-clients"

fi

# END fix for 'xccdf_org.ssgproject.content_rule_package_openldap-clients_removed'

###############################################################################
# BEGIN fix (172 / 219) for 'xccdf_org.ssgproject.content_rule_service_slapd_disabled'
###############################################################################
(>&2 echo "Remediating rule 172/219: 'xccdf_org.ssgproject.content_rule_service_slapd_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_slapd_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_slapd_disabled'

###############################################################################
# BEGIN fix (173 / 219) for 'xccdf_org.ssgproject.content_rule_postfix_network_listening_disabled'
###############################################################################
(>&2 echo "Remediating rule 173/219: 'xccdf_org.ssgproject.content_rule_postfix_network_listening_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_postfix_network_listening_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_postfix_network_listening_disabled'

###############################################################################
# BEGIN fix (174 / 219) for 'xccdf_org.ssgproject.content_rule_service_rpcbind_disabled'
###############################################################################
(>&2 echo "Remediating rule 174/219: 'xccdf_org.ssgproject.content_rule_service_rpcbind_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_rpcbind_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_rpcbind_disabled'

###############################################################################
# BEGIN fix (175 / 219) for 'xccdf_org.ssgproject.content_rule_service_nfs_disabled'
###############################################################################
(>&2 echo "Remediating rule 175/219: 'xccdf_org.ssgproject.content_rule_service_nfs_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_nfs_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_nfs_disabled'

###############################################################################
# BEGIN fix (176 / 219) for 'xccdf_org.ssgproject.content_rule_service_ntpd_enabled'
###############################################################################
(>&2 echo "Remediating rule 176/219: 'xccdf_org.ssgproject.content_rule_service_ntpd_enabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_ntpd_enabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_ntpd_enabled'

###############################################################################
# BEGIN fix (177 / 219) for 'xccdf_org.ssgproject.content_rule_chronyd_run_as_chrony_user'
###############################################################################
(>&2 echo "Remediating rule 177/219: 'xccdf_org.ssgproject.content_rule_chronyd_run_as_chrony_user'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ] && { rpm --quiet -q chrony; }; then

if grep -q 'OPTIONS=.*' /etc/sysconfig/chronyd; then
	# trying to solve cases where the parameter after OPTIONS
	#may or may not be enclosed in quotes
	sed -i -E -e 's/\s*-u\s*\w+\s*/ /' -e 's/^([\s]*OPTIONS=["]?[^"]*)("?)/\1 -u chrony\2/' /etc/sysconfig/chronyd
else
	echo 'OPTIONS="-u chrony"' >> /etc/sysconfig/chronyd
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_chronyd_run_as_chrony_user'

###############################################################################
# BEGIN fix (178 / 219) for 'xccdf_org.ssgproject.content_rule_chronyd_specify_remote_server'
###############################################################################
(>&2 echo "Remediating rule 178/219: 'xccdf_org.ssgproject.content_rule_chronyd_specify_remote_server'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ] && { rpm --quiet -q chrony; }; then

var_multiple_time_servers='0.ntp.cloud.aliyuncs.com,1.ntp.aliyun.com,2.ntp1.aliyun.com,3.ntp1.cloud.aliyuncs.com'


config_file="/etc/chrony.conf"

if ! grep -q '^[[:space:]]*\(server\|pool\)[[:space:]]\+[[:graph:]]\+' "$config_file" ; then
  if ! grep -q '#[[:space:]]*server' "$config_file" ; then
    for server in $(echo "$var_multiple_time_servers" | tr ',' '\n') ; do
      printf '\nserver %s' "$server" >> "$config_file"
    done
  else
    sed -i 's/#[ \t]*server/server/g' "$config_file"
  fi
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_chronyd_specify_remote_server'

###############################################################################
# BEGIN fix (179 / 219) for 'xccdf_org.ssgproject.content_rule_ntpd_configure_restrictions'
###############################################################################
(>&2 echo "Remediating rule 179/219: 'xccdf_org.ssgproject.content_rule_ntpd_configure_restrictions'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_ntpd_configure_restrictions' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_ntpd_configure_restrictions'

###############################################################################
# BEGIN fix (180 / 219) for 'xccdf_org.ssgproject.content_rule_ntpd_run_as_ntp_user'
###############################################################################
(>&2 echo "Remediating rule 180/219: 'xccdf_org.ssgproject.content_rule_ntpd_run_as_ntp_user'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_ntpd_run_as_ntp_user' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_ntpd_run_as_ntp_user'

###############################################################################
# BEGIN fix (181 / 219) for 'xccdf_org.ssgproject.content_rule_ntpd_specify_remote_server'
###############################################################################
(>&2 echo "Remediating rule 181/219: 'xccdf_org.ssgproject.content_rule_ntpd_specify_remote_server'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_ntpd_specify_remote_server' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_ntpd_specify_remote_server'

###############################################################################
# BEGIN fix (182 / 219) for 'xccdf_org.ssgproject.content_rule_service_rsyncd_disabled'
###############################################################################
(>&2 echo "Remediating rule 182/219: 'xccdf_org.ssgproject.content_rule_service_rsyncd_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_rsyncd_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_rsyncd_disabled'

###############################################################################
# BEGIN fix (183 / 219) for 'xccdf_org.ssgproject.content_rule_package_tcp_wrappers_installed'
###############################################################################
(>&2 echo "Remediating rule 183/219: 'xccdf_org.ssgproject.content_rule_package_tcp_wrappers_installed'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_package_tcp_wrappers_installed' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_package_tcp_wrappers_installed'

###############################################################################
# BEGIN fix (184 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_hosts_allow'
###############################################################################
(>&2 echo "Remediating rule 184/219: 'xccdf_org.ssgproject.content_rule_file_permissions_etc_hosts_allow'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chmod u-xs,g-xws,o-xwt /etc/hosts.allow

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_hosts_allow'

###############################################################################
# BEGIN fix (185 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_hosts_deny'
###############################################################################
(>&2 echo "Remediating rule 185/219: 'xccdf_org.ssgproject.content_rule_file_permissions_etc_hosts_deny'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chmod u-xs,g-xws,o-xwt /etc/hosts.deny

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_etc_hosts_deny'

###############################################################################
# BEGIN fix (186 / 219) for 'xccdf_org.ssgproject.content_rule_package_ypbind_removed'
###############################################################################
(>&2 echo "Remediating rule 186/219: 'xccdf_org.ssgproject.content_rule_package_ypbind_removed'")

# CAUTION: This remediation script will remove ypbind
#	   from the system, and may remove any packages
#	   that depend on ypbind. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

if rpm -q --quiet "ypbind" ; then

    yum remove -y "ypbind"

fi

# END fix for 'xccdf_org.ssgproject.content_rule_package_ypbind_removed'

###############################################################################
# BEGIN fix (187 / 219) for 'xccdf_org.ssgproject.content_rule_service_ypserv_disabled'
###############################################################################
(>&2 echo "Remediating rule 187/219: 'xccdf_org.ssgproject.content_rule_service_ypserv_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_ypserv_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_ypserv_disabled'

###############################################################################
# BEGIN fix (188 / 219) for 'xccdf_org.ssgproject.content_rule_package_rsh_removed'
###############################################################################
(>&2 echo "Remediating rule 188/219: 'xccdf_org.ssgproject.content_rule_package_rsh_removed'")

# CAUTION: This remediation script will remove rsh
#	   from the system, and may remove any packages
#	   that depend on rsh. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

if rpm -q --quiet "rsh" ; then

    yum remove -y "rsh"

fi

# END fix for 'xccdf_org.ssgproject.content_rule_package_rsh_removed'

###############################################################################
# BEGIN fix (189 / 219) for 'xccdf_org.ssgproject.content_rule_service_rsh_disabled'
###############################################################################
(>&2 echo "Remediating rule 189/219: 'xccdf_org.ssgproject.content_rule_service_rsh_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_rsh_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_rsh_disabled'

###############################################################################
# BEGIN fix (190 / 219) for 'xccdf_org.ssgproject.content_rule_no_rsh_trust_files'
###############################################################################
(>&2 echo "Remediating rule 190/219: 'xccdf_org.ssgproject.content_rule_no_rsh_trust_files'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_no_rsh_trust_files' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_no_rsh_trust_files'

###############################################################################
# BEGIN fix (191 / 219) for 'xccdf_org.ssgproject.content_rule_package_talk-server_removed'
###############################################################################
(>&2 echo "Remediating rule 191/219: 'xccdf_org.ssgproject.content_rule_package_talk-server_removed'")

# CAUTION: This remediation script will remove talk-server
#	   from the system, and may remove any packages
#	   that depend on talk-server. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

if rpm -q --quiet "talk-server" ; then

    yum remove -y "talk-server"

fi

# END fix for 'xccdf_org.ssgproject.content_rule_package_talk-server_removed'

###############################################################################
# BEGIN fix (192 / 219) for 'xccdf_org.ssgproject.content_rule_package_talk_removed'
###############################################################################
(>&2 echo "Remediating rule 192/219: 'xccdf_org.ssgproject.content_rule_package_talk_removed'")

# CAUTION: This remediation script will remove talk
#	   from the system, and may remove any packages
#	   that depend on talk. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

if rpm -q --quiet "talk" ; then

    yum remove -y "talk"

fi

# END fix for 'xccdf_org.ssgproject.content_rule_package_talk_removed'

###############################################################################
# BEGIN fix (193 / 219) for 'xccdf_org.ssgproject.content_rule_package_telnet_removed'
###############################################################################
(>&2 echo "Remediating rule 193/219: 'xccdf_org.ssgproject.content_rule_package_telnet_removed'")

# CAUTION: This remediation script will remove telnet
#	   from the system, and may remove any packages
#	   that depend on telnet. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

if rpm -q --quiet "telnet" ; then

    yum remove -y "telnet"

fi

# END fix for 'xccdf_org.ssgproject.content_rule_package_telnet_removed'

###############################################################################
# BEGIN fix (194 / 219) for 'xccdf_org.ssgproject.content_rule_service_telnet_disabled'
###############################################################################
(>&2 echo "Remediating rule 194/219: 'xccdf_org.ssgproject.content_rule_service_telnet_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_telnet_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_telnet_disabled'

###############################################################################
# BEGIN fix (195 / 219) for 'xccdf_org.ssgproject.content_rule_service_tftp_disabled'
###############################################################################
(>&2 echo "Remediating rule 195/219: 'xccdf_org.ssgproject.content_rule_service_tftp_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_tftp_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_tftp_disabled'

###############################################################################
# BEGIN fix (196 / 219) for 'xccdf_org.ssgproject.content_rule_service_squid_disabled'
###############################################################################
(>&2 echo "Remediating rule 196/219: 'xccdf_org.ssgproject.content_rule_service_squid_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_squid_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_squid_disabled'

###############################################################################
# BEGIN fix (197 / 219) for 'xccdf_org.ssgproject.content_rule_service_smb_disabled'
###############################################################################
(>&2 echo "Remediating rule 197/219: 'xccdf_org.ssgproject.content_rule_service_smb_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_smb_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_smb_disabled'

###############################################################################
# BEGIN fix (198 / 219) for 'xccdf_org.ssgproject.content_rule_service_snmpd_disabled'
###############################################################################
(>&2 echo "Remediating rule 198/219: 'xccdf_org.ssgproject.content_rule_service_snmpd_disabled'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_service_snmpd_disabled' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_service_snmpd_disabled'

###############################################################################
# BEGIN fix (199 / 219) for 'xccdf_org.ssgproject.content_rule_file_groupowner_sshd_config'
###############################################################################
(>&2 echo "Remediating rule 199/219: 'xccdf_org.ssgproject.content_rule_file_groupowner_sshd_config'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chgrp 0 /etc/ssh/sshd_config

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_groupowner_sshd_config'

###############################################################################
# BEGIN fix (200 / 219) for 'xccdf_org.ssgproject.content_rule_file_owner_sshd_config'
###############################################################################
(>&2 echo "Remediating rule 200/219: 'xccdf_org.ssgproject.content_rule_file_owner_sshd_config'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chown 0 /etc/ssh/sshd_config

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_owner_sshd_config'

###############################################################################
# BEGIN fix (201 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_sshd_config'
###############################################################################
(>&2 echo "Remediating rule 201/219: 'xccdf_org.ssgproject.content_rule_file_permissions_sshd_config'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

chmod u-xs,g-xwrs,o-xwrt /etc/ssh/sshd_config

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_sshd_config'

###############################################################################
# BEGIN fix (202 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_sshd_private_key'
###############################################################################
(>&2 echo "Remediating rule 202/219: 'xccdf_org.ssgproject.content_rule_file_permissions_sshd_private_key'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

for keyfile in /etc/ssh/*_key; do
    test -f "$keyfile" || continue
    if test root:root = "$(stat -c "%U:%G" "$keyfile")"; then
    
	chmod u-xs,g-xwrs,o-xwrt "$keyfile"
    
    
    else
        echo "Key-like file '$keyfile' is owned by an unexpected user:group combination"
    fi
done

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_sshd_private_key'

###############################################################################
# BEGIN fix (203 / 219) for 'xccdf_org.ssgproject.content_rule_file_permissions_sshd_pub_key'
###############################################################################
(>&2 echo "Remediating rule 203/219: 'xccdf_org.ssgproject.content_rule_file_permissions_sshd_pub_key'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

find -H /etc/ssh/ -maxdepth 1 -perm /u+xs,g+xws,o+xwt  -type f -regex '^.*\.pub$' -exec chmod u-xs,g-xws,o-xwt {} \;

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_sshd_pub_key'

###############################################################################
# BEGIN fix (204 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_set_keepalive'
###############################################################################
(>&2 echo "Remediating rule 204/219: 'xccdf_org.ssgproject.content_rule_sshd_set_keepalive'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sshd_set_keepalive' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_set_keepalive'

###############################################################################
# BEGIN fix (205 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_set_idle_timeout'
###############################################################################
(>&2 echo "Remediating rule 205/219: 'xccdf_org.ssgproject.content_rule_sshd_set_idle_timeout'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

sshd_idle_timeout_value='900'


if [ -e "/etc/ssh/sshd_config" ] ; then
    
    LC_ALL=C sed -i "/^\s*ClientAliveInterval\s\+/Id" "/etc/ssh/sshd_config"
else
    touch "/etc/ssh/sshd_config"
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/ssh/sshd_config"

cp "/etc/ssh/sshd_config" "/etc/ssh/sshd_config.bak"
# Insert at the beginning of the file
printf '%s\n' "ClientAliveInterval $sshd_idle_timeout_value" > "/etc/ssh/sshd_config"
cat "/etc/ssh/sshd_config.bak" >> "/etc/ssh/sshd_config"
# Clean up after ourselves.
rm "/etc/ssh/sshd_config.bak"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_set_idle_timeout'

###############################################################################
# BEGIN fix (206 / 219) for 'xccdf_org.ssgproject.content_rule_disable_host_auth'
###############################################################################
(>&2 echo "Remediating rule 206/219: 'xccdf_org.ssgproject.content_rule_disable_host_auth'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

if [ -e "/etc/ssh/sshd_config" ] ; then
    
    LC_ALL=C sed -i "/^\s*HostbasedAuthentication\s\+/Id" "/etc/ssh/sshd_config"
else
    touch "/etc/ssh/sshd_config"
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/ssh/sshd_config"

cp "/etc/ssh/sshd_config" "/etc/ssh/sshd_config.bak"
# Insert at the beginning of the file
printf '%s\n' "HostbasedAuthentication no" > "/etc/ssh/sshd_config"
cat "/etc/ssh/sshd_config.bak" >> "/etc/ssh/sshd_config"
# Clean up after ourselves.
rm "/etc/ssh/sshd_config.bak"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_disable_host_auth'

###############################################################################
# BEGIN fix (207 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_allow_only_protocol2'
###############################################################################
(>&2 echo "Remediating rule 207/219: 'xccdf_org.ssgproject.content_rule_sshd_allow_only_protocol2'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sshd_allow_only_protocol2' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_allow_only_protocol2'

###############################################################################
# BEGIN fix (208 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_disable_empty_passwords'
###############################################################################
(>&2 echo "Remediating rule 208/219: 'xccdf_org.ssgproject.content_rule_sshd_disable_empty_passwords'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

if [ -e "/etc/ssh/sshd_config" ] ; then
    
    LC_ALL=C sed -i "/^\s*PermitEmptyPasswords\s\+/Id" "/etc/ssh/sshd_config"
else
    touch "/etc/ssh/sshd_config"
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/ssh/sshd_config"

cp "/etc/ssh/sshd_config" "/etc/ssh/sshd_config.bak"
# Insert at the beginning of the file
printf '%s\n' "PermitEmptyPasswords no" > "/etc/ssh/sshd_config"
cat "/etc/ssh/sshd_config.bak" >> "/etc/ssh/sshd_config"
# Clean up after ourselves.
rm "/etc/ssh/sshd_config.bak"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_disable_empty_passwords'

###############################################################################
# BEGIN fix (209 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_disable_rhosts'
###############################################################################
(>&2 echo "Remediating rule 209/219: 'xccdf_org.ssgproject.content_rule_sshd_disable_rhosts'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

if [ -e "/etc/ssh/sshd_config" ] ; then
    
    LC_ALL=C sed -i "/^\s*IgnoreRhosts\s\+/Id" "/etc/ssh/sshd_config"
else
    touch "/etc/ssh/sshd_config"
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/ssh/sshd_config"

cp "/etc/ssh/sshd_config" "/etc/ssh/sshd_config.bak"
# Insert at the beginning of the file
printf '%s\n' "IgnoreRhosts yes" > "/etc/ssh/sshd_config"
cat "/etc/ssh/sshd_config.bak" >> "/etc/ssh/sshd_config"
# Clean up after ourselves.
rm "/etc/ssh/sshd_config.bak"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_disable_rhosts'

###############################################################################
# BEGIN fix (210 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_disable_root_login'
###############################################################################
(>&2 echo "Remediating rule 210/219: 'xccdf_org.ssgproject.content_rule_sshd_disable_root_login'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

if [ -e "/etc/ssh/sshd_config" ] ; then
    
    LC_ALL=C sed -i "/^\s*PermitRootLogin\s\+/Id" "/etc/ssh/sshd_config"
else
    touch "/etc/ssh/sshd_config"
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/ssh/sshd_config"

cp "/etc/ssh/sshd_config" "/etc/ssh/sshd_config.bak"
# Insert at the beginning of the file
printf '%s\n' "PermitRootLogin no" > "/etc/ssh/sshd_config"
cat "/etc/ssh/sshd_config.bak" >> "/etc/ssh/sshd_config"
# Clean up after ourselves.
rm "/etc/ssh/sshd_config.bak"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_disable_root_login'

###############################################################################
# BEGIN fix (211 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_do_not_permit_user_env'
###############################################################################
(>&2 echo "Remediating rule 211/219: 'xccdf_org.ssgproject.content_rule_sshd_do_not_permit_user_env'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

if [ -e "/etc/ssh/sshd_config" ] ; then
    
    LC_ALL=C sed -i "/^\s*PermitUserEnvironment\s\+/Id" "/etc/ssh/sshd_config"
else
    touch "/etc/ssh/sshd_config"
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/ssh/sshd_config"

cp "/etc/ssh/sshd_config" "/etc/ssh/sshd_config.bak"
# Insert at the beginning of the file
printf '%s\n' "PermitUserEnvironment no" > "/etc/ssh/sshd_config"
cat "/etc/ssh/sshd_config.bak" >> "/etc/ssh/sshd_config"
# Clean up after ourselves.
rm "/etc/ssh/sshd_config.bak"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_do_not_permit_user_env'

###############################################################################
# BEGIN fix (212 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_enable_warning_banner'
###############################################################################
(>&2 echo "Remediating rule 212/219: 'xccdf_org.ssgproject.content_rule_sshd_enable_warning_banner'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

if [ -e "/etc/ssh/sshd_config" ] ; then
    
    LC_ALL=C sed -i "/^\s*Banner\s\+/Id" "/etc/ssh/sshd_config"
else
    touch "/etc/ssh/sshd_config"
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/ssh/sshd_config"

cp "/etc/ssh/sshd_config" "/etc/ssh/sshd_config.bak"
# Insert at the beginning of the file
printf '%s\n' "Banner /etc/issue" > "/etc/ssh/sshd_config"
cat "/etc/ssh/sshd_config.bak" >> "/etc/ssh/sshd_config"
# Clean up after ourselves.
rm "/etc/ssh/sshd_config.bak"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_enable_warning_banner'

###############################################################################
# BEGIN fix (213 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_limit_user_access'
###############################################################################
(>&2 echo "Remediating rule 213/219: 'xccdf_org.ssgproject.content_rule_sshd_limit_user_access'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sshd_limit_user_access' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_limit_user_access'

###############################################################################
# BEGIN fix (214 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_set_login_grace_time'
###############################################################################
(>&2 echo "Remediating rule 214/219: 'xccdf_org.ssgproject.content_rule_sshd_set_login_grace_time'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

var_sshd_set_login_grace_time='60'


if [ -e "/etc/ssh/sshd_config" ] ; then
    
    LC_ALL=C sed -i "/^\s*LoginGraceTime\s\+/Id" "/etc/ssh/sshd_config"
else
    touch "/etc/ssh/sshd_config"
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/ssh/sshd_config"

cp "/etc/ssh/sshd_config" "/etc/ssh/sshd_config.bak"
# Insert at the beginning of the file
printf '%s\n' "LoginGraceTime $var_sshd_set_login_grace_time" > "/etc/ssh/sshd_config"
cat "/etc/ssh/sshd_config.bak" >> "/etc/ssh/sshd_config"
# Clean up after ourselves.
rm "/etc/ssh/sshd_config.bak"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_set_login_grace_time'

###############################################################################
# BEGIN fix (215 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_set_loglevel_verbose'
###############################################################################
(>&2 echo "Remediating rule 215/219: 'xccdf_org.ssgproject.content_rule_sshd_set_loglevel_verbose'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

if [ -e "/etc/ssh/sshd_config" ] ; then
    
    LC_ALL=C sed -i "/^\s*LogLevel\s\+/Id" "/etc/ssh/sshd_config"
else
    touch "/etc/ssh/sshd_config"
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/ssh/sshd_config"

cp "/etc/ssh/sshd_config" "/etc/ssh/sshd_config.bak"
# Insert at the beginning of the file
printf '%s\n' "LogLevel VERBOSE" > "/etc/ssh/sshd_config"
cat "/etc/ssh/sshd_config.bak" >> "/etc/ssh/sshd_config"
# Clean up after ourselves.
rm "/etc/ssh/sshd_config.bak"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_set_loglevel_verbose'

###############################################################################
# BEGIN fix (216 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_set_max_auth_tries'
###############################################################################
(>&2 echo "Remediating rule 216/219: 'xccdf_org.ssgproject.content_rule_sshd_set_max_auth_tries'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sshd_set_max_auth_tries' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_set_max_auth_tries'

###############################################################################
# BEGIN fix (217 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_use_approved_ciphers'
###############################################################################
(>&2 echo "Remediating rule 217/219: 'xccdf_org.ssgproject.content_rule_sshd_use_approved_ciphers'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sshd_use_approved_ciphers' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_use_approved_ciphers'

###############################################################################
# BEGIN fix (218 / 219) for 'xccdf_org.ssgproject.content_rule_sshd_use_approved_macs'
###############################################################################
(>&2 echo "Remediating rule 218/219: 'xccdf_org.ssgproject.content_rule_sshd_use_approved_macs'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_sshd_use_approved_macs' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_sshd_use_approved_macs'

###############################################################################
# BEGIN fix (219 / 219) for 'xccdf_org.ssgproject.content_rule_xwindows_remove_packages'
###############################################################################
(>&2 echo "Remediating rule 219/219: 'xccdf_org.ssgproject.content_rule_xwindows_remove_packages'")


# remove packages
if rpm -q --quiet "xorg-x11-server-Xorg" ; then

    yum remove -y "xorg-x11-server-Xorg"

fi
if rpm -q --quiet "xorg-x11-server-utils" ; then

    yum remove -y "xorg-x11-server-utils"

fi
if rpm -q --quiet "xorg-x11-server-common" ; then

    yum remove -y "xorg-x11-server-common"

fi

if rpm -q --quiet "xorg-x11-server-Xwayland" ; then

    yum remove -y "xorg-x11-server-Xwayland"

fi

# END fix for 'xccdf_org.ssgproject.content_rule_xwindows_remove_packages'

