#!/usr/bin/env bash
###############################################################################
#
# Bash Remediation Script for ANSSI-BP-028 (minimal)
#
# Profile Description:
# This profile contains configurations that align to ANSSI-BP-028 v2.0 at the minimal hardening level.
# ANSSI is the French National Information Security Agency, and stands for Agence nationale de la sécurité des systèmes d'information.
# ANSSI-BP-028 is a configuration recommendation for GNU/Linux systems.
# A copy of the ANSSI-BP-028 can be found at the ANSSI website:
# https://www.ssi.gouv.fr/administration/guide/recommandations-de-securite-relatives-a-un-systeme-gnulinux/
# Only the components strictly necessary to the service provided by the system should be installed.
# Those whose presence can not be justified should be disabled, removed or deleted.
# Performing a minimal install is a good starting point, but doesn't provide any assurance
# over any package installed later.
# Manual review is required to assess if the installed services are minimal.
#
# Profile ID:  xccdf_org.ssgproject.content_profile_anssi_bp28_minimal
# Benchmark ID:  xccdf_org.ssgproject.content_benchmark_SLE-15
# Benchmark Version:  0.1.70
# XCCDF Version:  1.2
#
# This file can be generated by OpenSCAP using:
# $ oscap xccdf generate fix --profile xccdf_org.ssgproject.content_profile_anssi_bp28_minimal --fix-type bash ssg-sle15-ds.xml
#
# This Bash Remediation Script is generated from an XCCDF profile without preliminary evaluation.
# It attempts to fix every selected rule, even if the system is already compliant.
#
# How to apply this Bash Remediation Script:
# $ sudo ./remediation-script.sh
#
###############################################################################

###############################################################################
# BEGIN fix (1 / 43) for 'xccdf_org.ssgproject.content_rule_sudo_remove_no_authenticate'
###############################################################################
(>&2 echo "Remediating rule 1/43: 'xccdf_org.ssgproject.content_rule_sudo_remove_no_authenticate'")

for f in /etc/sudoers /etc/sudoers.d/* ; do
  if [ ! -e "$f" ] ; then
    continue
  fi
  matching_list=$(grep -P '^(?!#).*[\s]+\!authenticate.*$' $f | uniq )
  if ! test -z "$matching_list"; then
    while IFS= read -r entry; do
      # comment out "!authenticate" matches to preserve user data
      sed -i "s/^${entry}$/# &/g" $f
    done <<< "$matching_list"

    /usr/sbin/visudo -cf $f &> /dev/null || echo "Fail to validate $f with visudo"
  fi
done

# END fix for 'xccdf_org.ssgproject.content_rule_sudo_remove_no_authenticate'

###############################################################################
# BEGIN fix (2 / 43) for 'xccdf_org.ssgproject.content_rule_sudo_remove_nopasswd'
###############################################################################
(>&2 echo "Remediating rule 2/43: 'xccdf_org.ssgproject.content_rule_sudo_remove_nopasswd'")

for f in /etc/sudoers /etc/sudoers.d/* ; do
  if [ ! -e "$f" ] ; then
    continue
  fi
  matching_list=$(grep -P '^(?!#).*[\s]+NOPASSWD[\s]*\:.*$' $f | uniq )
  if ! test -z "$matching_list"; then
    while IFS= read -r entry; do
      # comment out "NOPASSWD" matches to preserve user data
      sed -i "s/^${entry}$/# &/g" $f
    done <<< "$matching_list"

    /usr/sbin/visudo -cf $f &> /dev/null || echo "Fail to validate $f with visudo"
  fi
done

# END fix for 'xccdf_org.ssgproject.content_rule_sudo_remove_nopasswd'

###############################################################################
# BEGIN fix (3 / 43) for 'xccdf_org.ssgproject.content_rule_package_dnf-automatic_installed'
###############################################################################
(>&2 echo "Remediating rule 3/43: 'xccdf_org.ssgproject.content_rule_package_dnf-automatic_installed'")

zypper install -y "dnf-automatic"

# END fix for 'xccdf_org.ssgproject.content_rule_package_dnf-automatic_installed'

###############################################################################
# BEGIN fix (4 / 43) for 'xccdf_org.ssgproject.content_rule_dnf-automatic_apply_updates'
###############################################################################
(>&2 echo "Remediating rule 4/43: 'xccdf_org.ssgproject.content_rule_dnf-automatic_apply_updates'")

found=false

# set value in all files if they contain section or key
for f in $(echo -n "/etc/dnf/automatic.conf"); do
    if [ ! -e "$f" ]; then
        continue
    fi

    # find key in section and change value
    if grep -qzosP "[[:space:]]*\[commands\]([^\n\[]*\n+)+?[[:space:]]*apply_updates" "$f"; then
            sed -i "s/apply_updates[^(\n)]*/apply_updates = yes/" "$f"
            found=true

    # find section and add key = value to it
    elif grep -qs "[[:space:]]*\[commands\]" "$f"; then
            sed -i "/[[:space:]]*\[commands\]/a apply_updates = yes" "$f"
            found=true
    fi
done

# if section not in any file, append section with key = value to FIRST file in files parameter
if ! $found ; then
    file=$(echo "/etc/dnf/automatic.conf" | cut -f1 -d ' ')
    mkdir -p "$(dirname "$file")"
    echo -e "[commands]\napply_updates = yes" >> "$file"
fi

# END fix for 'xccdf_org.ssgproject.content_rule_dnf-automatic_apply_updates'

###############################################################################
# BEGIN fix (5 / 43) for 'xccdf_org.ssgproject.content_rule_dnf-automatic_security_updates_only'
###############################################################################
(>&2 echo "Remediating rule 5/43: 'xccdf_org.ssgproject.content_rule_dnf-automatic_security_updates_only'")

found=false

# set value in all files if they contain section or key
for f in $(echo -n "/etc/dnf/automatic.conf"); do
    if [ ! -e "$f" ]; then
        continue
    fi

    # find key in section and change value
    if grep -qzosP "[[:space:]]*\[commands\]([^\n\[]*\n+)+?[[:space:]]*upgrade_type" "$f"; then
            sed -i "s/upgrade_type[^(\n)]*/upgrade_type = security/" "$f"
            found=true

    # find section and add key = value to it
    elif grep -qs "[[:space:]]*\[commands\]" "$f"; then
            sed -i "/[[:space:]]*\[commands\]/a upgrade_type = security" "$f"
            found=true
    fi
done

# if section not in any file, append section with key = value to FIRST file in files parameter
if ! $found ; then
    file=$(echo "/etc/dnf/automatic.conf" | cut -f1 -d ' ')
    mkdir -p "$(dirname "$file")"
    echo -e "[commands]\nupgrade_type = security" >> "$file"
fi

# END fix for 'xccdf_org.ssgproject.content_rule_dnf-automatic_security_updates_only'

###############################################################################
# BEGIN fix (6 / 43) for 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_globally_activated'
###############################################################################
(>&2 echo "Remediating rule 6/43: 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_globally_activated'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q zypper; then

# Strip any search characters in the key arg so that the key can be replaced without
# adding any search characters to the config file.
stripped_key=$(sed 's/[\^=\$,;+]*//g' <<< "^gpgcheck")

# shellcheck disable=SC2059
printf -v formatted_output "%s = %s" "$stripped_key" "1"

# If the key exists, change it. Otherwise, add it to the config_file.
# We search for the key string followed by a word boundary (matched by \>),
# so if we search for 'setting', 'setting2' won't match.
if LC_ALL=C grep -q -m 1 -i -e "^gpgcheck\\>" "/etc/zypp/zypp.conf"; then
    escaped_formatted_output=$(sed -e 's|/|\\/|g' <<< "$formatted_output")
    LC_ALL=C sed -i --follow-symlinks "s/^gpgcheck\\>.*/$escaped_formatted_output/gi" "/etc/zypp/zypp.conf"
else
    if [[ -s "/etc/zypp/zypp.conf" ]] && [[ -n "$(tail -c 1 -- "/etc/zypp/zypp.conf" || true)" ]]; then
        LC_ALL=C sed -i --follow-symlinks '$a'\\ "/etc/zypp/zypp.conf"
    fi
    cce="CCE-83290-7"
    printf '# Per %s: Set %s in %s\n' "${cce}" "${formatted_output}" "/etc/zypp/zypp.conf" >> "/etc/zypp/zypp.conf"
    printf '%s\n' "$formatted_output" >> "/etc/zypp/zypp.conf"
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_globally_activated'

###############################################################################
# BEGIN fix (7 / 43) for 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_local_packages'
###############################################################################
(>&2 echo "Remediating rule 7/43: 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_local_packages'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q yum; then

# Strip any search characters in the key arg so that the key can be replaced without
# adding any search characters to the config file.
stripped_key=$(sed 's/[\^=\$,;+]*//g' <<< "^localpkg_gpgcheck")

# shellcheck disable=SC2059
printf -v formatted_output "%s = %s" "$stripped_key" "1"

# If the key exists, change it. Otherwise, add it to the config_file.
# We search for the key string followed by a word boundary (matched by \>),
# so if we search for 'setting', 'setting2' won't match.
if LC_ALL=C grep -q -m 1 -i -e "^localpkg_gpgcheck\\>" "/etc/zypp/zypp.conf"; then
    escaped_formatted_output=$(sed -e 's|/|\\/|g' <<< "$formatted_output")
    LC_ALL=C sed -i --follow-symlinks "s/^localpkg_gpgcheck\\>.*/$escaped_formatted_output/gi" "/etc/zypp/zypp.conf"
else
    if [[ -s "/etc/zypp/zypp.conf" ]] && [[ -n "$(tail -c 1 -- "/etc/zypp/zypp.conf" || true)" ]]; then
        LC_ALL=C sed -i --follow-symlinks '$a'\\ "/etc/zypp/zypp.conf"
    fi
    cce="CCE-91167-7"
    printf '# Per %s: Set %s in %s\n' "${cce}" "${formatted_output}" "/etc/zypp/zypp.conf" >> "/etc/zypp/zypp.conf"
    printf '%s\n' "$formatted_output" >> "/etc/zypp/zypp.conf"
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_local_packages'

###############################################################################
# BEGIN fix (8 / 43) for 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_never_disabled'
###############################################################################
(>&2 echo "Remediating rule 8/43: 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_never_disabled'")

sed -i 's/gpgcheck\s*=.*/gpgcheck=1/g' /etc/zypp/repos.d/*

# END fix for 'xccdf_org.ssgproject.content_rule_ensure_gpgcheck_never_disabled'

###############################################################################
# BEGIN fix (9 / 43) for 'xccdf_org.ssgproject.content_rule_security_patches_up_to_date'
###############################################################################
(>&2 echo "Remediating rule 9/43: 'xccdf_org.ssgproject.content_rule_security_patches_up_to_date'")


zypper patch -g security -y

# END fix for 'xccdf_org.ssgproject.content_rule_security_patches_up_to_date'

###############################################################################
# BEGIN fix (10 / 43) for 'xccdf_org.ssgproject.content_rule_timer_dnf-automatic_enabled'
###############################################################################
(>&2 echo "Remediating rule 10/43: 'xccdf_org.ssgproject.content_rule_timer_dnf-automatic_enabled'")

SYSTEMCTL_EXEC='/usr/bin/systemctl'
"$SYSTEMCTL_EXEC" start 'dnf-automatic.timer'
"$SYSTEMCTL_EXEC" enable 'dnf-automatic.timer'

# END fix for 'xccdf_org.ssgproject.content_rule_timer_dnf-automatic_enabled'

###############################################################################
# BEGIN fix (11 / 43) for 'xccdf_org.ssgproject.content_rule_accounts_password_pam_unix_remember'
###############################################################################
(>&2 echo "Remediating rule 11/43: 'xccdf_org.ssgproject.content_rule_accounts_password_pam_unix_remember'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q pam; then

var_password_pam_unix_remember='2'


if [ -f /usr/bin/authselect ]; then
    if authselect list-features minimal | grep -q with-pwhistory; then
        if ! authselect check; then
        echo "
        authselect integrity check failed. Remediation aborted!
        This remediation could not be applied because an authselect profile was not selected or the selected profile is not intact.
        It is not recommended to manually edit the PAM files when authselect tool is available.
        In cases where the default authselect profile does not cover a specific demand, a custom authselect profile is recommended."
        exit 1
        fi
        authselect enable-feature with-pwhistory

        authselect apply-changes -b
    else
        
        if ! authselect check; then
        echo "
        authselect integrity check failed. Remediation aborted!
        This remediation could not be applied because an authselect profile was not selected or the selected profile is not intact.
        It is not recommended to manually edit the PAM files when authselect tool is available.
        In cases where the default authselect profile does not cover a specific demand, a custom authselect profile is recommended."
        exit 1
        fi

        CURRENT_PROFILE=$(authselect current -r | awk '{ print $1 }')
        # If not already in use, a custom profile is created preserving the enabled features.
        if [[ ! $CURRENT_PROFILE == custom/* ]]; then
            ENABLED_FEATURES=$(authselect current | tail -n+3 | awk '{ print $2 }')
            authselect create-profile hardening -b $CURRENT_PROFILE
            CURRENT_PROFILE="custom/hardening"
            
            authselect apply-changes -b --backup=before-hardening-custom-profile
            authselect select $CURRENT_PROFILE
            for feature in $ENABLED_FEATURES; do
                authselect enable-feature $feature;
            done
            
            authselect apply-changes -b --backup=after-hardening-custom-profile
        fi
        PAM_FILE_NAME=$(basename "/etc/pam.d/common-password")
        PAM_FILE_PATH="/etc/authselect/$CURRENT_PROFILE/$PAM_FILE_NAME"

        authselect apply-changes -b
        if ! grep -qP '^\s*password\s+'"requisite"'\s+pam_pwhistory.so\s*.*' "$PAM_FILE_PATH"; then
            # Line matching group + control + module was not found. Check group + module.
            if [ "$(grep -cP '^\s*password\s+.*\s+pam_pwhistory.so\s*' "$PAM_FILE_PATH")" -eq 1 ]; then
                # The control is updated only if one single line matches.
                sed -i -E --follow-symlinks 's/^(\s*password\s+).*(\bpam_pwhistory.so.*)/\1'"requisite"' \2/' "$PAM_FILE_PATH"
            else
                LAST_MATCH_LINE=$(grep -nP "^password.*requisite.*pam_pwquality\.so" "$PAM_FILE_PATH" | tail -n 1 | cut -d: -f 1)
                if [ ! -z $LAST_MATCH_LINE ]; then
                    sed -i --follow-symlinks $LAST_MATCH_LINE' a password     '"requisite"'    pam_pwhistory.so' "$PAM_FILE_PATH"
                else
                    echo 'password    '"requisite"'    pam_pwhistory.so' >> "$PAM_FILE_PATH"
                fi
            fi
        fi
    fi
else
    if ! grep -qP '^\s*password\s+'"requisite"'\s+pam_pwhistory.so\s*.*' "/etc/pam.d/common-password"; then
        # Line matching group + control + module was not found. Check group + module.
        if [ "$(grep -cP '^\s*password\s+.*\s+pam_pwhistory.so\s*' "/etc/pam.d/common-password")" -eq 1 ]; then
            # The control is updated only if one single line matches.
            sed -i -E --follow-symlinks 's/^(\s*password\s+).*(\bpam_pwhistory.so.*)/\1'"requisite"' \2/' "/etc/pam.d/common-password"
        else
            LAST_MATCH_LINE=$(grep -nP "^password.*requisite.*pam_pwquality\.so" "/etc/pam.d/common-password" | tail -n 1 | cut -d: -f 1)
            if [ ! -z $LAST_MATCH_LINE ]; then
                sed -i --follow-symlinks $LAST_MATCH_LINE' a password     '"requisite"'    pam_pwhistory.so' "/etc/pam.d/common-password"
            else
                echo 'password    '"requisite"'    pam_pwhistory.so' >> "/etc/pam.d/common-password"
            fi
        fi
    fi
fi

PWHISTORY_CONF="/etc/security/pwhistory.conf"
if [ -f $PWHISTORY_CONF ]; then
    regex="^\s*remember\s*="
    line="remember = $var_password_pam_unix_remember"
    if ! grep -q $regex $PWHISTORY_CONF; then
        echo $line >> $PWHISTORY_CONF
    else
        sed -i --follow-symlinks 's|^\s*\(remember\s*=\s*\)\(\S\+\)|\1'"$var_password_pam_unix_remember"'|g' $PWHISTORY_CONF
    fi
    if [ -e "/etc/pam.d/common-password" ] ; then
        PAM_FILE_PATH="/etc/pam.d/common-password"
        if [ -f /usr/bin/authselect ]; then
            
            if ! authselect check; then
            echo "
            authselect integrity check failed. Remediation aborted!
            This remediation could not be applied because an authselect profile was not selected or the selected profile is not intact.
            It is not recommended to manually edit the PAM files when authselect tool is available.
            In cases where the default authselect profile does not cover a specific demand, a custom authselect profile is recommended."
            exit 1
            fi

            CURRENT_PROFILE=$(authselect current -r | awk '{ print $1 }')
            # If not already in use, a custom profile is created preserving the enabled features.
            if [[ ! $CURRENT_PROFILE == custom/* ]]; then
                ENABLED_FEATURES=$(authselect current | tail -n+3 | awk '{ print $2 }')
                authselect create-profile hardening -b $CURRENT_PROFILE
                CURRENT_PROFILE="custom/hardening"
                
                authselect apply-changes -b --backup=before-hardening-custom-profile
                authselect select $CURRENT_PROFILE
                for feature in $ENABLED_FEATURES; do
                    authselect enable-feature $feature;
                done
                
                authselect apply-changes -b --backup=after-hardening-custom-profile
            fi
            PAM_FILE_NAME=$(basename "/etc/pam.d/common-password")
            PAM_FILE_PATH="/etc/authselect/$CURRENT_PROFILE/$PAM_FILE_NAME"

            authselect apply-changes -b
        fi
        
    if grep -qP '^\s*password\s.*\bpam_pwhistory.so\s.*\bremember\b' "$PAM_FILE_PATH"; then
        sed -i -E --follow-symlinks 's/(.*password.*pam_pwhistory.so.*)\bremember\b=?[[:alnum:]]*(.*)/\1\2/g' "$PAM_FILE_PATH"
    fi
        if [ -f /usr/bin/authselect ]; then
            
            authselect apply-changes -b
        fi
    else
        echo "/etc/pam.d/common-password was not found" >&2
    fi
else
    PAM_FILE_PATH="/etc/pam.d/common-password"
    if [ -f /usr/bin/authselect ]; then
        
        if ! authselect check; then
        echo "
        authselect integrity check failed. Remediation aborted!
        This remediation could not be applied because an authselect profile was not selected or the selected profile is not intact.
        It is not recommended to manually edit the PAM files when authselect tool is available.
        In cases where the default authselect profile does not cover a specific demand, a custom authselect profile is recommended."
        exit 1
        fi

        CURRENT_PROFILE=$(authselect current -r | awk '{ print $1 }')
        # If not already in use, a custom profile is created preserving the enabled features.
        if [[ ! $CURRENT_PROFILE == custom/* ]]; then
            ENABLED_FEATURES=$(authselect current | tail -n+3 | awk '{ print $2 }')
            authselect create-profile hardening -b $CURRENT_PROFILE
            CURRENT_PROFILE="custom/hardening"
            
            authselect apply-changes -b --backup=before-hardening-custom-profile
            authselect select $CURRENT_PROFILE
            for feature in $ENABLED_FEATURES; do
                authselect enable-feature $feature;
            done
            
            authselect apply-changes -b --backup=after-hardening-custom-profile
        fi
        PAM_FILE_NAME=$(basename "/etc/pam.d/common-password")
        PAM_FILE_PATH="/etc/authselect/$CURRENT_PROFILE/$PAM_FILE_NAME"

        authselect apply-changes -b
    fi
    if ! grep -qP '^\s*password\s+'"requisite"'\s+pam_pwhistory.so\s*.*' "$PAM_FILE_PATH"; then
        # Line matching group + control + module was not found. Check group + module.
        if [ "$(grep -cP '^\s*password\s+.*\s+pam_pwhistory.so\s*' "$PAM_FILE_PATH")" -eq 1 ]; then
            # The control is updated only if one single line matches.
            sed -i -E --follow-symlinks 's/^(\s*password\s+).*(\bpam_pwhistory.so.*)/\1'"requisite"' \2/' "$PAM_FILE_PATH"
        else
            echo 'password    '"requisite"'    pam_pwhistory.so' >> "$PAM_FILE_PATH"
        fi
    fi
    # Check the option
    if ! grep -qP '^\s*password\s+'"requisite"'\s+pam_pwhistory.so\s*.*\sremember\b' "$PAM_FILE_PATH"; then
        sed -i -E --follow-symlinks '/\s*password\s+'"requisite"'\s+pam_pwhistory.so.*/ s/$/ remember='"$var_password_pam_unix_remember"'/' "$PAM_FILE_PATH"
    else
        sed -i -E --follow-symlinks 's/(\s*password\s+'"requisite"'\s+pam_pwhistory.so\s+.*)('"remember"'=)[[:alnum:]]+\s*(.*)/\1\2'"$var_password_pam_unix_remember"' \3/' "$PAM_FILE_PATH"
    fi
    if [ -f /usr/bin/authselect ]; then
        
        authselect apply-changes -b
    fi
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_password_pam_unix_remember'

###############################################################################
# BEGIN fix (12 / 43) for 'xccdf_org.ssgproject.content_rule_accounts_passwords_pam_tally2'
###############################################################################
(>&2 echo "Remediating rule 12/43: 'xccdf_org.ssgproject.content_rule_accounts_passwords_pam_tally2'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q pam; then

var_password_pam_tally2='5'

# Use a non-number regexp to force update of the value of the deny option




if ! grep -qP '^\s*auth\s+'"required"'\s+pam_tally2.so\s*.*' "/etc/pam.d/login"; then
    # Line matching group + control + module was not found. Check group + module.
    if [ "$(grep -cP '^\s*auth\s+.*\s+pam_tally2.so\s*' "/etc/pam.d/login")" -eq 1 ]; then
        # The control is updated only if one single line matches.
        sed -i -E --follow-symlinks 's/^(\s*auth\s+).*(\bpam_tally2.so.*)/\1'"required"' \2/' "/etc/pam.d/login"
    else
        echo 'auth    '"required"'    pam_tally2.so' >> "/etc/pam.d/login"
    fi
fi
# Check the option
if ! grep -qP '^\s*auth\s+'"required"'\s+pam_tally2.so\s*.*\sdeny\b' "/etc/pam.d/login"; then
    sed -i -E --follow-symlinks '/\s*auth\s+'"required"'\s+pam_tally2.so.*/ s/$/ deny='"${var_password_pam_tally2}"'/' "/etc/pam.d/login"
else
    sed -i -E --follow-symlinks 's/(\s*auth\s+'"required"'\s+pam_tally2.so\s+.*)('"deny"'=)[[:alnum:]]+\s*(.*)/\1\2'"${var_password_pam_tally2}"' \3/' "/etc/pam.d/login"
fi
if ! grep -qP '^\s*auth\s+'"required"'\s+pam_tally2.so\s*.*' "/etc/pam.d/login"; then
    # Line matching group + control + module was not found. Check group + module.
    if [ "$(grep -cP '^\s*auth\s+.*\s+pam_tally2.so\s*' "/etc/pam.d/login")" -eq 1 ]; then
        # The control is updated only if one single line matches.
        sed -i -E --follow-symlinks 's/^(\s*auth\s+).*(\bpam_tally2.so.*)/\1'"required"' \2/' "/etc/pam.d/login"
    else
        LAST_MATCH_LINE=$(grep -nP "(fail)" "/etc/pam.d/login" | tail -n 1 | cut -d: -f 1)
        if [ ! -z $LAST_MATCH_LINE ]; then
            sed -i --follow-symlinks $LAST_MATCH_LINE' a auth     '"required"'    pam_tally2.so' "/etc/pam.d/login"
        else
            echo 'auth    '"required"'    pam_tally2.so' >> "/etc/pam.d/login"
        fi
    fi
fi
# Check the option
if ! grep -qP '^\s*auth\s+'"required"'\s+pam_tally2.so\s*.*\sonerr\b' "/etc/pam.d/login"; then
    sed -i -E --follow-symlinks '/\s*auth\s+'"required"'\s+pam_tally2.so.*/ s/$/ onerr='"fail"'/' "/etc/pam.d/login"
else
    sed -i -E --follow-symlinks 's/(\s*auth\s+'"required"'\s+pam_tally2.so\s+.*)('"onerr"'=)[[:alnum:]]+\s*(.*)/\1\2'"fail"' \3/' "/etc/pam.d/login"
fi
if ! grep -qP '^\s*account\s+'"required"'\s+pam_tally2.so\s*.*' "/etc/pam.d/common-account"; then
    # Line matching group + control + module was not found. Check group + module.
    if [ "$(grep -cP '^\s*account\s+.*\s+pam_tally2.so\s*' "/etc/pam.d/common-account")" -eq 1 ]; then
        # The control is updated only if one single line matches.
        sed -i -E --follow-symlinks 's/^(\s*account\s+).*(\bpam_tally2.so.*)/\1'"required"' \2/' "/etc/pam.d/common-account"
    else
        echo 'account    '"required"'    pam_tally2.so' >> "/etc/pam.d/common-account"
    fi
fi
# Check the option
if ! grep -qP '^\s*account\s+'"required"'\s+pam_tally2.so\s*.*\s\b' "/etc/pam.d/common-account"; then
    sed -i -E --follow-symlinks '/\s*account\s+'"required"'\s+pam_tally2.so.*/ s/$/ /' "/etc/pam.d/common-account"
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_passwords_pam_tally2'

###############################################################################
# BEGIN fix (13 / 43) for 'xccdf_org.ssgproject.content_rule_accounts_passwords_pam_tally2_deny_root'
###############################################################################
(>&2 echo "Remediating rule 13/43: 'xccdf_org.ssgproject.content_rule_accounts_passwords_pam_tally2_deny_root'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q pam; then

if grep -qP '^\s*auth\s+'"required"'\s+pam_tally2.so\s.*\bonerr=fail\b' "/etc/pam.d/login"; then
    sed -i -E --follow-symlinks 's/(.*auth.*'"required"'.*pam_tally2.so.*)\sonerr=fail=?[[:alnum:]]*(.*)/\1\2/g' "/etc/pam.d/login"
fi
if ! grep -qP '^\s*auth\s+'"required"'\s+pam_tally2.so\s*.*' "/etc/pam.d/login"; then
    # Line matching group + control + module was not found. Check group + module.
    if [ "$(grep -cP '^\s*auth\s+.*\s+pam_tally2.so\s*' "/etc/pam.d/login")" -eq 1 ]; then
        # The control is updated only if one single line matches.
        sed -i -E --follow-symlinks 's/^(\s*auth\s+).*(\bpam_tally2.so.*)/\1'"required"' \2/' "/etc/pam.d/login"
    else
        echo 'auth    '"required"'    pam_tally2.so' >> "/etc/pam.d/login"
    fi
fi
# Check the option
if ! grep -qP '^\s*auth\s+'"required"'\s+pam_tally2.so\s*.*\seven_deny_root\b' "/etc/pam.d/login"; then
    sed -i -E --follow-symlinks '/\s*auth\s+'"required"'\s+pam_tally2.so.*/ s/$/ even_deny_root/' "/etc/pam.d/login"
fi
if ! grep -qP '^\s*account\s+'"required"'\s+pam_tally2.so\s*.*' "/etc/pam.d/common-account"; then
    # Line matching group + control + module was not found. Check group + module.
    if [ "$(grep -cP '^\s*account\s+.*\s+pam_tally2.so\s*' "/etc/pam.d/common-account")" -eq 1 ]; then
        # The control is updated only if one single line matches.
        sed -i -E --follow-symlinks 's/^(\s*account\s+).*(\bpam_tally2.so.*)/\1'"required"' \2/' "/etc/pam.d/common-account"
    else
        echo 'account    '"required"'    pam_tally2.so' >> "/etc/pam.d/common-account"
    fi
fi
# Check the option
if ! grep -qP '^\s*account\s+'"required"'\s+pam_tally2.so\s*.*\s\b' "/etc/pam.d/common-account"; then
    sed -i -E --follow-symlinks '/\s*account\s+'"required"'\s+pam_tally2.so.*/ s/$/ /' "/etc/pam.d/common-account"
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_passwords_pam_tally2_deny_root'

###############################################################################
# BEGIN fix (14 / 43) for 'xccdf_org.ssgproject.content_rule_accounts_passwords_pam_tally2_unlock_time'
###############################################################################
(>&2 echo "Remediating rule 14/43: 'xccdf_org.ssgproject.content_rule_accounts_passwords_pam_tally2_unlock_time'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q pam; then

var_accounts_passwords_pam_tally2_unlock_time='1800'

# Use a non-number regexp to force update of the value of the deny option
if ! grep -qP '^\s*auth\s+'"required"'\s+pam_tally2.so\s*.*' "/etc/pam.d/login"; then
    # Line matching group + control + module was not found. Check group + module.
    if [ "$(grep -cP '^\s*auth\s+.*\s+pam_tally2.so\s*' "/etc/pam.d/login")" -eq 1 ]; then
        # The control is updated only if one single line matches.
        sed -i -E --follow-symlinks 's/^(\s*auth\s+).*(\bpam_tally2.so.*)/\1'"required"' \2/' "/etc/pam.d/login"
    else
        echo 'auth    '"required"'    pam_tally2.so' >> "/etc/pam.d/login"
    fi
fi
# Check the option
if ! grep -qP '^\s*auth\s+'"required"'\s+pam_tally2.so\s*.*\sunlock_time\b' "/etc/pam.d/login"; then
    sed -i -E --follow-symlinks '/\s*auth\s+'"required"'\s+pam_tally2.so.*/ s/$/ unlock_time='"${var_accounts_passwords_pam_tally2_unlock_time}"'/' "/etc/pam.d/login"
else
    sed -i -E --follow-symlinks 's/(\s*auth\s+'"required"'\s+pam_tally2.so\s+.*)('"unlock_time"'=)[[:alnum:]]+\s*(.*)/\1\2'"${var_accounts_passwords_pam_tally2_unlock_time}"' \3/' "/etc/pam.d/login"
fi
if ! grep -qP '^\s*account\s+'"required"'\s+pam_tally2.so\s*.*' "/etc/pam.d/common-account"; then
    # Line matching group + control + module was not found. Check group + module.
    if [ "$(grep -cP '^\s*account\s+.*\s+pam_tally2.so\s*' "/etc/pam.d/common-account")" -eq 1 ]; then
        # The control is updated only if one single line matches.
        sed -i -E --follow-symlinks 's/^(\s*account\s+).*(\bpam_tally2.so.*)/\1'"required"' \2/' "/etc/pam.d/common-account"
    else
        echo 'account    '"required"'    pam_tally2.so' >> "/etc/pam.d/common-account"
    fi
fi
# Check the option
if ! grep -qP '^\s*account\s+'"required"'\s+pam_tally2.so\s*.*\s\b' "/etc/pam.d/common-account"; then
    sed -i -E --follow-symlinks '/\s*account\s+'"required"'\s+pam_tally2.so.*/ s/$/ /' "/etc/pam.d/common-account"
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_passwords_pam_tally2_unlock_time'

###############################################################################
# BEGIN fix (15 / 43) for 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_dcredit'
###############################################################################
(>&2 echo "Remediating rule 15/43: 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_dcredit'")

declare -a VALUES=()
declare -a VALUE_NAMES=()
declare -a ARGS=()
declare -a NEW_ARGS=()

var_password_pam_dcredit='-1'

VALUES+=("$var_password_pam_dcredit")
VALUE_NAMES+=("dcredit")
ARGS+=("")
NEW_ARGS+=("")

for idx in "${!VALUES[@]}"
do
    if [ -e "/etc/pam.d/common-password" ] ; then
        valueRegex="${VALUES[$idx]}" defaultValue="${VALUES[$idx]}"
        # non-empty values need to be preceded by an equals sign
        [ -n "${valueRegex}" ] && valueRegex="=${valueRegex}"
        # add an equals sign to non-empty values
        [ -n "${defaultValue}" ] && defaultValue="=${defaultValue}"

        # fix the value for 'option' if one exists but does not match 'valueRegex'
        if grep -q -P "^\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s+${VALUE_NAMES[$idx]}(?"'!'"${valueRegex}(\\s|\$))" < "/etc/pam.d/common-password" ; then
            sed --follow-symlinks -i -E -e "s/^(\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s)${VALUE_NAMES[$idx]}=[^[:space:]]*/\\1${VALUE_NAMES[$idx]}${defaultValue}/" "/etc/pam.d/common-password"

        # add 'option=default' if option is not set
        elif grep -q -E "^\\s*password\\s+requisite\\s+pam_cracklib.so" < "/etc/pam.d/common-password" &&
                grep    -E "^\\s*password\\s+requisite\\s+pam_cracklib.so" < "/etc/pam.d/common-password" | grep -q -E -v "\\s${VALUE_NAMES[$idx]}(=|\\s|\$)" ; then

            sed --follow-symlinks -i -E -e "s/^(\\s*password\\s+requisite\\s+pam_cracklib.so[^\\n]*)/\\1 ${VALUE_NAMES[$idx]}${defaultValue}/" "/etc/pam.d/common-password"
        # add a new entry if none exists
        elif ! grep -q -P "^\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s+${VALUE_NAMES[$idx]}${valueRegex}(\\s|\$)" < "/etc/pam.d/common-password" ; then
            echo "password requisite pam_cracklib.so ${VALUE_NAMES[$idx]}${defaultValue}" >> "/etc/pam.d/common-password"
        fi
    else
        echo "/etc/pam.d/common-password doesn't exist" >&2
    fi
done

for idx in "${!ARGS[@]}"
do
    if ! grep -q -P "^\s*password\s+requisite\s+pam_cracklib.so.*\s+${ARGS[$idx]}\s*$" /etc/pam.d/common-password ; then
        sed --follow-symlinks -i -E -e "s/^\\s*password\\s+requisite\\s+pam_cracklib.so.*\$/& ${NEW_ARGS[$idx]}/" /etc/pam.d/common-password
    fi
done

# END fix for 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_dcredit'

###############################################################################
# BEGIN fix (16 / 43) for 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_lcredit'
###############################################################################
(>&2 echo "Remediating rule 16/43: 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_lcredit'")

declare -a VALUES=()
declare -a VALUE_NAMES=()
declare -a ARGS=()
declare -a NEW_ARGS=()

var_password_pam_lcredit='-1'

VALUES+=("$var_password_pam_lcredit")
VALUE_NAMES+=("lcredit")
ARGS+=("")
NEW_ARGS+=("")

for idx in "${!VALUES[@]}"
do
    if [ -e "/etc/pam.d/common-password" ] ; then
        valueRegex="${VALUES[$idx]}" defaultValue="${VALUES[$idx]}"
        # non-empty values need to be preceded by an equals sign
        [ -n "${valueRegex}" ] && valueRegex="=${valueRegex}"
        # add an equals sign to non-empty values
        [ -n "${defaultValue}" ] && defaultValue="=${defaultValue}"

        # fix the value for 'option' if one exists but does not match 'valueRegex'
        if grep -q -P "^\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s+${VALUE_NAMES[$idx]}(?"'!'"${valueRegex}(\\s|\$))" < "/etc/pam.d/common-password" ; then
            sed --follow-symlinks -i -E -e "s/^(\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s)${VALUE_NAMES[$idx]}=[^[:space:]]*/\\1${VALUE_NAMES[$idx]}${defaultValue}/" "/etc/pam.d/common-password"

        # add 'option=default' if option is not set
        elif grep -q -E "^\\s*password\\s+requisite\\s+pam_cracklib.so" < "/etc/pam.d/common-password" &&
                grep    -E "^\\s*password\\s+requisite\\s+pam_cracklib.so" < "/etc/pam.d/common-password" | grep -q -E -v "\\s${VALUE_NAMES[$idx]}(=|\\s|\$)" ; then

            sed --follow-symlinks -i -E -e "s/^(\\s*password\\s+requisite\\s+pam_cracklib.so[^\\n]*)/\\1 ${VALUE_NAMES[$idx]}${defaultValue}/" "/etc/pam.d/common-password"
        # add a new entry if none exists
        elif ! grep -q -P "^\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s+${VALUE_NAMES[$idx]}${valueRegex}(\\s|\$)" < "/etc/pam.d/common-password" ; then
            echo "password requisite pam_cracklib.so ${VALUE_NAMES[$idx]}${defaultValue}" >> "/etc/pam.d/common-password"
        fi
    else
        echo "/etc/pam.d/common-password doesn't exist" >&2
    fi
done

for idx in "${!ARGS[@]}"
do
    if ! grep -q -P "^\s*password\s+requisite\s+pam_cracklib.so.*\s+${ARGS[$idx]}\s*$" /etc/pam.d/common-password ; then
        sed --follow-symlinks -i -E -e "s/^\\s*password\\s+requisite\\s+pam_cracklib.so.*\$/& ${NEW_ARGS[$idx]}/" /etc/pam.d/common-password
    fi
done

# END fix for 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_lcredit'

###############################################################################
# BEGIN fix (17 / 43) for 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_minlen'
###############################################################################
(>&2 echo "Remediating rule 17/43: 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_minlen'")

declare -a VALUES=()
declare -a VALUE_NAMES=()
declare -a ARGS=()
declare -a NEW_ARGS=()

var_password_pam_minlen='18'

VALUES+=("$var_password_pam_minlen")
VALUE_NAMES+=("minlen")
ARGS+=("")
NEW_ARGS+=("")

for idx in "${!VALUES[@]}"
do
    if [ -e "/etc/pam.d/common-password" ] ; then
        valueRegex="${VALUES[$idx]}" defaultValue="${VALUES[$idx]}"
        # non-empty values need to be preceded by an equals sign
        [ -n "${valueRegex}" ] && valueRegex="=${valueRegex}"
        # add an equals sign to non-empty values
        [ -n "${defaultValue}" ] && defaultValue="=${defaultValue}"

        # fix the value for 'option' if one exists but does not match 'valueRegex'
        if grep -q -P "^\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s+${VALUE_NAMES[$idx]}(?"'!'"${valueRegex}(\\s|\$))" < "/etc/pam.d/common-password" ; then
            sed --follow-symlinks -i -E -e "s/^(\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s)${VALUE_NAMES[$idx]}=[^[:space:]]*/\\1${VALUE_NAMES[$idx]}${defaultValue}/" "/etc/pam.d/common-password"

        # add 'option=default' if option is not set
        elif grep -q -E "^\\s*password\\s+requisite\\s+pam_cracklib.so" < "/etc/pam.d/common-password" &&
                grep    -E "^\\s*password\\s+requisite\\s+pam_cracklib.so" < "/etc/pam.d/common-password" | grep -q -E -v "\\s${VALUE_NAMES[$idx]}(=|\\s|\$)" ; then

            sed --follow-symlinks -i -E -e "s/^(\\s*password\\s+requisite\\s+pam_cracklib.so[^\\n]*)/\\1 ${VALUE_NAMES[$idx]}${defaultValue}/" "/etc/pam.d/common-password"
        # add a new entry if none exists
        elif ! grep -q -P "^\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s+${VALUE_NAMES[$idx]}${valueRegex}(\\s|\$)" < "/etc/pam.d/common-password" ; then
            echo "password requisite pam_cracklib.so ${VALUE_NAMES[$idx]}${defaultValue}" >> "/etc/pam.d/common-password"
        fi
    else
        echo "/etc/pam.d/common-password doesn't exist" >&2
    fi
done

for idx in "${!ARGS[@]}"
do
    if ! grep -q -P "^\s*password\s+requisite\s+pam_cracklib.so.*\s+${ARGS[$idx]}\s*$" /etc/pam.d/common-password ; then
        sed --follow-symlinks -i -E -e "s/^\\s*password\\s+requisite\\s+pam_cracklib.so.*\$/& ${NEW_ARGS[$idx]}/" /etc/pam.d/common-password
    fi
done

# END fix for 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_minlen'

###############################################################################
# BEGIN fix (18 / 43) for 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_ocredit'
###############################################################################
(>&2 echo "Remediating rule 18/43: 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_ocredit'")

declare -a VALUES=()
declare -a VALUE_NAMES=()
declare -a ARGS=()
declare -a NEW_ARGS=()

var_password_pam_ocredit='-1'

VALUES+=("$var_password_pam_ocredit")
VALUE_NAMES+=("ocredit")
ARGS+=("")
NEW_ARGS+=("")

for idx in "${!VALUES[@]}"
do
    if [ -e "/etc/pam.d/common-password" ] ; then
        valueRegex="${VALUES[$idx]}" defaultValue="${VALUES[$idx]}"
        # non-empty values need to be preceded by an equals sign
        [ -n "${valueRegex}" ] && valueRegex="=${valueRegex}"
        # add an equals sign to non-empty values
        [ -n "${defaultValue}" ] && defaultValue="=${defaultValue}"

        # fix the value for 'option' if one exists but does not match 'valueRegex'
        if grep -q -P "^\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s+${VALUE_NAMES[$idx]}(?"'!'"${valueRegex}(\\s|\$))" < "/etc/pam.d/common-password" ; then
            sed --follow-symlinks -i -E -e "s/^(\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s)${VALUE_NAMES[$idx]}=[^[:space:]]*/\\1${VALUE_NAMES[$idx]}${defaultValue}/" "/etc/pam.d/common-password"

        # add 'option=default' if option is not set
        elif grep -q -E "^\\s*password\\s+requisite\\s+pam_cracklib.so" < "/etc/pam.d/common-password" &&
                grep    -E "^\\s*password\\s+requisite\\s+pam_cracklib.so" < "/etc/pam.d/common-password" | grep -q -E -v "\\s${VALUE_NAMES[$idx]}(=|\\s|\$)" ; then

            sed --follow-symlinks -i -E -e "s/^(\\s*password\\s+requisite\\s+pam_cracklib.so[^\\n]*)/\\1 ${VALUE_NAMES[$idx]}${defaultValue}/" "/etc/pam.d/common-password"
        # add a new entry if none exists
        elif ! grep -q -P "^\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s+${VALUE_NAMES[$idx]}${valueRegex}(\\s|\$)" < "/etc/pam.d/common-password" ; then
            echo "password requisite pam_cracklib.so ${VALUE_NAMES[$idx]}${defaultValue}" >> "/etc/pam.d/common-password"
        fi
    else
        echo "/etc/pam.d/common-password doesn't exist" >&2
    fi
done

for idx in "${!ARGS[@]}"
do
    if ! grep -q -P "^\s*password\s+requisite\s+pam_cracklib.so.*\s+${ARGS[$idx]}\s*$" /etc/pam.d/common-password ; then
        sed --follow-symlinks -i -E -e "s/^\\s*password\\s+requisite\\s+pam_cracklib.so.*\$/& ${NEW_ARGS[$idx]}/" /etc/pam.d/common-password
    fi
done

# END fix for 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_ocredit'

###############################################################################
# BEGIN fix (19 / 43) for 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_ucredit'
###############################################################################
(>&2 echo "Remediating rule 19/43: 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_ucredit'")

declare -a VALUES=()
declare -a VALUE_NAMES=()
declare -a ARGS=()
declare -a NEW_ARGS=()

var_password_pam_ucredit='-1'

VALUES+=("$var_password_pam_ucredit")
VALUE_NAMES+=("ucredit")
ARGS+=("")
NEW_ARGS+=("")

for idx in "${!VALUES[@]}"
do
    if [ -e "/etc/pam.d/common-password" ] ; then
        valueRegex="${VALUES[$idx]}" defaultValue="${VALUES[$idx]}"
        # non-empty values need to be preceded by an equals sign
        [ -n "${valueRegex}" ] && valueRegex="=${valueRegex}"
        # add an equals sign to non-empty values
        [ -n "${defaultValue}" ] && defaultValue="=${defaultValue}"

        # fix the value for 'option' if one exists but does not match 'valueRegex'
        if grep -q -P "^\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s+${VALUE_NAMES[$idx]}(?"'!'"${valueRegex}(\\s|\$))" < "/etc/pam.d/common-password" ; then
            sed --follow-symlinks -i -E -e "s/^(\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s)${VALUE_NAMES[$idx]}=[^[:space:]]*/\\1${VALUE_NAMES[$idx]}${defaultValue}/" "/etc/pam.d/common-password"

        # add 'option=default' if option is not set
        elif grep -q -E "^\\s*password\\s+requisite\\s+pam_cracklib.so" < "/etc/pam.d/common-password" &&
                grep    -E "^\\s*password\\s+requisite\\s+pam_cracklib.so" < "/etc/pam.d/common-password" | grep -q -E -v "\\s${VALUE_NAMES[$idx]}(=|\\s|\$)" ; then

            sed --follow-symlinks -i -E -e "s/^(\\s*password\\s+requisite\\s+pam_cracklib.so[^\\n]*)/\\1 ${VALUE_NAMES[$idx]}${defaultValue}/" "/etc/pam.d/common-password"
        # add a new entry if none exists
        elif ! grep -q -P "^\\s*password\\s+requisite\\s+pam_cracklib.so(\\s.+)?\\s+${VALUE_NAMES[$idx]}${valueRegex}(\\s|\$)" < "/etc/pam.d/common-password" ; then
            echo "password requisite pam_cracklib.so ${VALUE_NAMES[$idx]}${defaultValue}" >> "/etc/pam.d/common-password"
        fi
    else
        echo "/etc/pam.d/common-password doesn't exist" >&2
    fi
done

for idx in "${!ARGS[@]}"
do
    if ! grep -q -P "^\s*password\s+requisite\s+pam_cracklib.so.*\s+${ARGS[$idx]}\s*$" /etc/pam.d/common-password ; then
        sed --follow-symlinks -i -E -e "s/^\\s*password\\s+requisite\\s+pam_cracklib.so.*\$/& ${NEW_ARGS[$idx]}/" /etc/pam.d/common-password
    fi
done

# END fix for 'xccdf_org.ssgproject.content_rule_cracklib_accounts_password_pam_ucredit'

###############################################################################
# BEGIN fix (20 / 43) for 'xccdf_org.ssgproject.content_rule_set_password_hashing_algorithm_systemauth'
###############################################################################
(>&2 echo "Remediating rule 20/43: 'xccdf_org.ssgproject.content_rule_set_password_hashing_algorithm_systemauth'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q pam; then

if [ -e "/etc/pam.d/common-password" ] ; then
    PAM_FILE_PATH="/etc/pam.d/common-password"
    if [ -f /usr/bin/authselect ]; then
        
        if ! authselect check; then
        echo "
        authselect integrity check failed. Remediation aborted!
        This remediation could not be applied because an authselect profile was not selected or the selected profile is not intact.
        It is not recommended to manually edit the PAM files when authselect tool is available.
        In cases where the default authselect profile does not cover a specific demand, a custom authselect profile is recommended."
        exit 1
        fi

        CURRENT_PROFILE=$(authselect current -r | awk '{ print $1 }')
        # If not already in use, a custom profile is created preserving the enabled features.
        if [[ ! $CURRENT_PROFILE == custom/* ]]; then
            ENABLED_FEATURES=$(authselect current | tail -n+3 | awk '{ print $2 }')
            authselect create-profile hardening -b $CURRENT_PROFILE
            CURRENT_PROFILE="custom/hardening"
            
            authselect apply-changes -b --backup=before-hardening-custom-profile
            authselect select $CURRENT_PROFILE
            for feature in $ENABLED_FEATURES; do
                authselect enable-feature $feature;
            done
            
            authselect apply-changes -b --backup=after-hardening-custom-profile
        fi
        PAM_FILE_NAME=$(basename "/etc/pam.d/common-password")
        PAM_FILE_PATH="/etc/authselect/$CURRENT_PROFILE/$PAM_FILE_NAME"

        authselect apply-changes -b
    fi
    if ! grep -qP '^\s*password\s+'"required"'\s+pam_unix.so\s*.*' "$PAM_FILE_PATH"; then
            # Line matching group + control + module was not found. Check group + module.
            if [ "$(grep -cP '^\s*password\s+.*\s+pam_unix.so\s*' "$PAM_FILE_PATH")" -eq 1 ]; then
                # The control is updated only if one single line matches.
                sed -i -E --follow-symlinks 's/^(\s*password\s+).*(\bpam_unix.so.*)/\1'"required"' \2/' "$PAM_FILE_PATH"
            else
                echo 'password    '"required"'    pam_unix.so' >> "$PAM_FILE_PATH"
            fi
        fi
        # Check the option
        if ! grep -qP '^\s*password\s+'"required"'\s+pam_unix.so\s*.*\ssha512\b' "$PAM_FILE_PATH"; then
            sed -i -E --follow-symlinks '/\s*password\s+'"required"'\s+pam_unix.so.*/ s/$/ sha512/' "$PAM_FILE_PATH"
        fi
    if [ -f /usr/bin/authselect ]; then
        
        authselect apply-changes -b
    fi
else
    echo "/etc/pam.d/common-password was not found" >&2
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_set_password_hashing_algorithm_systemauth'

###############################################################################
# BEGIN fix (21 / 43) for 'xccdf_org.ssgproject.content_rule_set_password_hashing_min_rounds_logindefs'
###############################################################################
(>&2 echo "Remediating rule 21/43: 'xccdf_org.ssgproject.content_rule_set_password_hashing_min_rounds_logindefs'")

if [ -e "/etc/login.defs" ] ; then
    
    LC_ALL=C sed -i "/^\s*SHA_CRYPT_MIN_ROUNDS\s*/Id" "/etc/login.defs"
else
    printf '%s\n' "Path '/etc/login.defs' wasn't found on this system. Refusing to continue." >&2
    return 1
fi
# make sure file has newline at the end
sed -i -e '$a\' "/etc/login.defs"

cp "/etc/login.defs" "/etc/login.defs.bak"
# Insert at the end of the file
printf '%s\n' "SHA_CRYPT_MIN_ROUNDS 5000" >> "/etc/login.defs"
# Clean up after ourselves.
rm "/etc/login.defs.bak"

# END fix for 'xccdf_org.ssgproject.content_rule_set_password_hashing_min_rounds_logindefs'

###############################################################################
# BEGIN fix (22 / 43) for 'xccdf_org.ssgproject.content_rule_accounts_maximum_age_login_defs'
###############################################################################
(>&2 echo "Remediating rule 22/43: 'xccdf_org.ssgproject.content_rule_accounts_maximum_age_login_defs'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q shadow; then

var_accounts_maximum_age_login_defs='90'


grep -q ^PASS_MAX_DAYS /etc/login.defs && \
  sed -i "s/PASS_MAX_DAYS.*/PASS_MAX_DAYS     $var_accounts_maximum_age_login_defs/g" /etc/login.defs
if ! [ $? -eq 0 ]; then
    echo "PASS_MAX_DAYS      $var_accounts_maximum_age_login_defs" >> /etc/login.defs
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_maximum_age_login_defs'

###############################################################################
# BEGIN fix (23 / 43) for 'xccdf_org.ssgproject.content_rule_accounts_password_minlen_login_defs'
###############################################################################
(>&2 echo "Remediating rule 23/43: 'xccdf_org.ssgproject.content_rule_accounts_password_minlen_login_defs'")
# Remediation is applicable only in certain platforms
if rpm --quiet -q shadow; then

var_accounts_password_minlen_login_defs='18'


grep -q ^PASS_MIN_LEN /etc/login.defs && \
sed -i "s/PASS_MIN_LEN.*/PASS_MIN_LEN\t$var_accounts_password_minlen_login_defs/g" /etc/login.defs
if ! [ $? -eq 0 ]
then
  echo -e "PASS_MIN_LEN\t$var_accounts_password_minlen_login_defs" >> /etc/login.defs
fi

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_accounts_password_minlen_login_defs'

###############################################################################
# BEGIN fix (24 / 43) for 'xccdf_org.ssgproject.content_rule_dir_perms_world_writable_root_owned'
###############################################################################
(>&2 echo "Remediating rule 24/43: 'xccdf_org.ssgproject.content_rule_dir_perms_world_writable_root_owned'")

# At least under containerized env /proc can have files w/o possilibity to
# modify even as root. And touching /proc is not good idea anyways.
find / -path /proc -prune -o \
    -not -fstype afs -not -fstype ceph -not -fstype cifs -not -fstype smb3 -not -fstype smbfs \
    -not -fstype sshfs -not -fstype ncpfs -not -fstype ncp -not -fstype nfs -not -fstype nfs4 \
    -not -fstype gfs -not -fstype gfs2 -not -fstype glusterfs -not -fstype gpfs \
    -not -fstype pvfs2 -not -fstype ocfs2 -not -fstype lustre -not -fstype davfs \
    -not -fstype fuse.sshfs -type d -perm -0002 -uid +0 -exec chown root {} \;

# END fix for 'xccdf_org.ssgproject.content_rule_dir_perms_world_writable_root_owned'

###############################################################################
# BEGIN fix (25 / 43) for 'xccdf_org.ssgproject.content_rule_dir_perms_world_writable_sticky_bits'
###############################################################################
(>&2 echo "Remediating rule 25/43: 'xccdf_org.ssgproject.content_rule_dir_perms_world_writable_sticky_bits'")
df --local -P | awk '{if (NR!=1) print $6}' \
| xargs -I '$6' find '$6' -xdev -type d \
\( -perm -0002 -a ! -perm -1000 \) 2>/dev/null \
-exec chmod a+t {} +

# END fix for 'xccdf_org.ssgproject.content_rule_dir_perms_world_writable_sticky_bits'

###############################################################################
# BEGIN fix (26 / 43) for 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_sgid'
###############################################################################
(>&2 echo "Remediating rule 26/43: 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_sgid'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_sgid' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_sgid'

###############################################################################
# BEGIN fix (27 / 43) for 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_suid'
###############################################################################
(>&2 echo "Remediating rule 27/43: 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_suid'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_suid' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_suid'

###############################################################################
# BEGIN fix (28 / 43) for 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_world_writable'
###############################################################################
(>&2 echo "Remediating rule 28/43: 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_world_writable'")

find / -xdev -type f -perm -002 -exec chmod o-w {} \;

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_unauthorized_world_writable'

###############################################################################
# BEGIN fix (29 / 43) for 'xccdf_org.ssgproject.content_rule_file_permissions_ungroupowned'
###############################################################################
(>&2 echo "Remediating rule 29/43: 'xccdf_org.ssgproject.content_rule_file_permissions_ungroupowned'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_file_permissions_ungroupowned' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_file_permissions_ungroupowned'

###############################################################################
# BEGIN fix (30 / 43) for 'xccdf_org.ssgproject.content_rule_no_files_unowned_by_user'
###############################################################################
(>&2 echo "Remediating rule 30/43: 'xccdf_org.ssgproject.content_rule_no_files_unowned_by_user'")
(>&2 echo "FIX FOR THIS RULE 'xccdf_org.ssgproject.content_rule_no_files_unowned_by_user' IS MISSING!")

# END fix for 'xccdf_org.ssgproject.content_rule_no_files_unowned_by_user'

###############################################################################
# BEGIN fix (31 / 43) for 'xccdf_org.ssgproject.content_rule_package_dhcp_removed'
###############################################################################
(>&2 echo "Remediating rule 31/43: 'xccdf_org.ssgproject.content_rule_package_dhcp_removed'")

# CAUTION: This remediation script will remove dhcp-server
#	   from the system, and may remove any packages
#	   that depend on dhcp-server. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "dhcp-server"

# END fix for 'xccdf_org.ssgproject.content_rule_package_dhcp_removed'

###############################################################################
# BEGIN fix (32 / 43) for 'xccdf_org.ssgproject.content_rule_package_sendmail_removed'
###############################################################################
(>&2 echo "Remediating rule 32/43: 'xccdf_org.ssgproject.content_rule_package_sendmail_removed'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

# CAUTION: This remediation script will remove sendmail
#	   from the system, and may remove any packages
#	   that depend on sendmail. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "sendmail"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_package_sendmail_removed'

###############################################################################
# BEGIN fix (33 / 43) for 'xccdf_org.ssgproject.content_rule_package_xinetd_removed'
###############################################################################
(>&2 echo "Remediating rule 33/43: 'xccdf_org.ssgproject.content_rule_package_xinetd_removed'")
# Remediation is applicable only in certain platforms
if [ ! -f /.dockerenv ] && [ ! -f /run/.containerenv ]; then

# CAUTION: This remediation script will remove xinetd
#	   from the system, and may remove any packages
#	   that depend on xinetd. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "xinetd"

else
    >&2 echo 'Remediation is not applicable, nothing was done'
fi

# END fix for 'xccdf_org.ssgproject.content_rule_package_xinetd_removed'

###############################################################################
# BEGIN fix (34 / 43) for 'xccdf_org.ssgproject.content_rule_package_ypbind_removed'
###############################################################################
(>&2 echo "Remediating rule 34/43: 'xccdf_org.ssgproject.content_rule_package_ypbind_removed'")

# CAUTION: This remediation script will remove ypbind
#	   from the system, and may remove any packages
#	   that depend on ypbind. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "ypbind"

# END fix for 'xccdf_org.ssgproject.content_rule_package_ypbind_removed'

###############################################################################
# BEGIN fix (35 / 43) for 'xccdf_org.ssgproject.content_rule_package_ypserv_removed'
###############################################################################
(>&2 echo "Remediating rule 35/43: 'xccdf_org.ssgproject.content_rule_package_ypserv_removed'")

# CAUTION: This remediation script will remove ypserv
#	   from the system, and may remove any packages
#	   that depend on ypserv. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "ypserv"

# END fix for 'xccdf_org.ssgproject.content_rule_package_ypserv_removed'

###############################################################################
# BEGIN fix (36 / 43) for 'xccdf_org.ssgproject.content_rule_package_rsh-server_removed'
###############################################################################
(>&2 echo "Remediating rule 36/43: 'xccdf_org.ssgproject.content_rule_package_rsh-server_removed'")

# CAUTION: This remediation script will remove rsh-server
#	   from the system, and may remove any packages
#	   that depend on rsh-server. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "rsh-server"

# END fix for 'xccdf_org.ssgproject.content_rule_package_rsh-server_removed'

###############################################################################
# BEGIN fix (37 / 43) for 'xccdf_org.ssgproject.content_rule_package_rsh_removed'
###############################################################################
(>&2 echo "Remediating rule 37/43: 'xccdf_org.ssgproject.content_rule_package_rsh_removed'")

# CAUTION: This remediation script will remove rsh
#	   from the system, and may remove any packages
#	   that depend on rsh. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "rsh"

# END fix for 'xccdf_org.ssgproject.content_rule_package_rsh_removed'

###############################################################################
# BEGIN fix (38 / 43) for 'xccdf_org.ssgproject.content_rule_package_talk-server_removed'
###############################################################################
(>&2 echo "Remediating rule 38/43: 'xccdf_org.ssgproject.content_rule_package_talk-server_removed'")

# CAUTION: This remediation script will remove talk-server
#	   from the system, and may remove any packages
#	   that depend on talk-server. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "talk-server"

# END fix for 'xccdf_org.ssgproject.content_rule_package_talk-server_removed'

###############################################################################
# BEGIN fix (39 / 43) for 'xccdf_org.ssgproject.content_rule_package_talk_removed'
###############################################################################
(>&2 echo "Remediating rule 39/43: 'xccdf_org.ssgproject.content_rule_package_talk_removed'")

# CAUTION: This remediation script will remove talk
#	   from the system, and may remove any packages
#	   that depend on talk. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "talk"

# END fix for 'xccdf_org.ssgproject.content_rule_package_talk_removed'

###############################################################################
# BEGIN fix (40 / 43) for 'xccdf_org.ssgproject.content_rule_package_telnet-server_removed'
###############################################################################
(>&2 echo "Remediating rule 40/43: 'xccdf_org.ssgproject.content_rule_package_telnet-server_removed'")

# CAUTION: This remediation script will remove telnet-server
#	   from the system, and may remove any packages
#	   that depend on telnet-server. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "telnet-server"

# END fix for 'xccdf_org.ssgproject.content_rule_package_telnet-server_removed'

###############################################################################
# BEGIN fix (41 / 43) for 'xccdf_org.ssgproject.content_rule_package_telnet_removed'
###############################################################################
(>&2 echo "Remediating rule 41/43: 'xccdf_org.ssgproject.content_rule_package_telnet_removed'")

# CAUTION: This remediation script will remove telnet
#	   from the system, and may remove any packages
#	   that depend on telnet. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "telnet"

# END fix for 'xccdf_org.ssgproject.content_rule_package_telnet_removed'

###############################################################################
# BEGIN fix (42 / 43) for 'xccdf_org.ssgproject.content_rule_package_tftp-server_removed'
###############################################################################
(>&2 echo "Remediating rule 42/43: 'xccdf_org.ssgproject.content_rule_package_tftp-server_removed'")

# CAUTION: This remediation script will remove tftp-server
#	   from the system, and may remove any packages
#	   that depend on tftp-server. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "tftp-server"

# END fix for 'xccdf_org.ssgproject.content_rule_package_tftp-server_removed'

###############################################################################
# BEGIN fix (43 / 43) for 'xccdf_org.ssgproject.content_rule_package_tftp_removed'
###############################################################################
(>&2 echo "Remediating rule 43/43: 'xccdf_org.ssgproject.content_rule_package_tftp_removed'")

# CAUTION: This remediation script will remove tftp
#	   from the system, and may remove any packages
#	   that depend on tftp. Execute this
#	   remediation AFTER testing on a non-production
#	   system!

zypper remove -y "tftp"

# END fix for 'xccdf_org.ssgproject.content_rule_package_tftp_removed'

