#! /bin/bash

[[ -n "${TRACE:-}" ]] && set -x

set -eu

readonly g_distro_name_oracle="oracle"
readonly g_distro_name_centos="centos"
readonly g_distro_name_redhat="rhel"
readonly g_distro_name_amazon="amazon"
readonly g_distro_name_suse="suse"
readonly g_distro_name_opensuse="opensuse"
readonly g_distro_name_ubuntu="ubuntu"
readonly g_distro_name_debian="debian"
readonly g_distro_name_thinpro="thinpro"
readonly g_distro_name_gcos="gcos"
readonly g_distro_name_photon="photon"
readonly g_distro_name_suse_rancher="rancher"
readonly g_distro_name_fedora_core="fedora"
readonly g_distro_name_flatcar="flatcar"
readonly g_distro_name_almalinux="alma"
readonly g_distro_name_rockylinux="rocky"
readonly g_distro_name_rhcos="rhcos"
readonly g_distro_name_centos_stream="centos_stream"
readonly g_distro_name_bottlerocket="bottlerocket"
readonly g_distro_name_talos="talos"
readonly g_distro_name_mariner="mariner"
readonly g_distro_name_unknown="unknown"

readonly g_distro_version_unknown="0"

_rpm_query() {
    declare -r package_name="$1"
    declare -r paramater="$2"
    rpm -q --qf "%{$paramater}\n" "$package_name"
}

_rpm_exists() {
    declare -r package_name="$1"
    [[ "$(_rpm_query "$package_name" "name")" == "$package_name" ]]
}

_rpm_version() {
    declare -r package_name="$1"
    _rpm_query "$package_name" "version"
}

_resolve_distro_name_from_os_release() {

    declare -r host_root="${XDR_HOST_ROOT:-}"

    declare -r release_file="$host_root/etc/os-release"
    if [[ ! -f "$release_file" ]]; then
        return
    fi

    declare -r release_id_line="$(grep "^ID=" "$release_file")"
    declare -r release_pretty_name_line="$(grep "^PRETTY_NAME=" "$release_file")"

    if [[ ! "$release_id_line" =~ \
            ^ID=[\"\']?([0-9a-z\.\_\-]+)[\"\']?[[:space:]]*$ ]]; then
        return
    fi

    declare release_id="${BASH_REMATCH[1]}"
    if [ "$release_id" == "centos" ] && [[ "$release_pretty_name_line" == *"Stream"* ]]; then
        release_id="centos_stream"
    fi
    case "$release_id" in
        ol)
            echo "$g_distro_name_oracle";;
        centos)
            echo "$g_distro_name_centos";;
        rhel)
            echo "$g_distro_name_redhat";;
        rhcos)
            echo "$g_distro_name_rhcos";;
        almalinux)
            echo "$g_distro_name_almalinux";;
        rocky)
            echo "$g_distro_name_rockylinux";;
        amzn)
            echo "$g_distro_name_amazon";;
        sles)
            echo "$g_distro_name_suse";;
        opensuse-leap)
            echo "$g_distro_name_opensuse";;
        ubuntu)
            echo "$g_distro_name_ubuntu";;
        debian)
            echo "$g_distro_name_debian";;
        thinpro)
            echo "$g_distro_name_thinpro";;
        cos)
            echo "$g_distro_name_gcos";;
        photon)
            echo "$g_distro_name_photon";;
        rancheros)
            echo "$g_distro_name_suse_rancher";;
        fedora)
            echo "$g_distro_name_fedora_core";;
        flatcar)
            echo "$g_distro_name_flatcar";;
        centos_stream)
            echo "$g_distro_name_centos_stream";;
        bottlerocket)
            echo "$g_distro_name_bottlerocket";;
        talos)
            echo "$g_distro_name_talos";;
        mariner)
            echo "$g_distro_name_mariner";;
        *)
            echo "";;
    esac
}

_resolve_distro_name_legacy() {

    declare -r host_root="${XDR_HOST_ROOT:-}"

    if [[ -f "$host_root/etc/system-release" ]]; then
        if [[ -f "$host_root/etc/oracle-release" ]]; then
            echo "$g_distro_name_oracle"
        elif _rpm_exists "centos-release"; then
            echo "$g_distro_name_centos"
        elif _rpm_exists "redhat-release"; then
            echo "$g_distro_name_redhat"
        elif _rpm_exists "redhat-release-server"; then
            echo "$g_distro_name_redhat"
        elif _rpm_exists "system-release"; then
            echo "$g_distro_name_amazon"
        fi
    elif [[ -f "$host_root/etc/SuSE-release" ]]; then

        echo "$g_distro_name_suse"
    elif [[ -f "$host_root/etc/lsb-release" ]]; then

        declare -r lsb_distrib_id="$(grep "^DISTRIB_ID=" "$host_root/etc/lsb-release" | cut -d'=' -f2)"
        if [[ "$lsb_distrib_id" == "ThinPro" ]]; then
            echo "$g_distro_name_thinpro"
            return
        fi

        declare -r lsb_device_type="$(grep "^DEVICETYPE=" "$host_root/etc/lsb-release" | cut -d'=' -f2)"
        if [[ "${lsb_device_type^^}" == "OTHER" ]]; then
            echo "$g_distro_name_gcos"
            return
        fi

        echo "$g_distro_name_ubuntu"
    elif [[ -f "$host_root/etc/debian_version" ]]; then
        echo "$g_distro_name_debian"
    else
        echo ""
    fi
}

resolve_distro_name() {

    declare distro="$(_resolve_distro_name_from_os_release)"
    if [[ -z "$distro" ]]; then
        distro="$(_resolve_distro_name_legacy)"
    fi

    echo "${distro:-$g_distro_name_unknown}"
}

_resolve_distro_version_from_os_release() {

    declare -r distro="$1"
    case "$distro" in
        "$g_distro_name_debian" | \
        "$g_distro_name_gcos" | \
        "$g_distro_name_unknown")
            return
            ;;
    esac

    declare -r host_root="${XDR_HOST_ROOT:-}"

    declare -r release_file="$host_root/etc/os-release"
    if [[ ! -f "$release_file" ]]; then
        return
    fi

    declare -r release_version_line="$(grep "^VERSION_ID=" "$release_file")"

    if [[ ! "$release_version_line" =~ \
            ^VERSION_ID=[\"\']?([0-9a-z\.\_\-]+)[\"\']?[[:space:]]*$ ]]; then
        return
    fi

    declare -r release_version="${BASH_REMATCH[1]}"
    echo "$release_version"
}

_resolve_distro_version_legacy() {

    declare -r host_root="${XDR_HOST_ROOT:-}"

    declare -r distro="$1"
    case "$distro" in
        "$g_distro_name_redhat" | "$g_distro_name_oracle")
            if _rpm_exists "redhat-release-server"; then
                _rpm_version "redhat-release-server"
            elif _rpm_exists "redhat-release"; then
                _rpm_version "redhat-release"
            fi
            ;;

        "$g_distro_name_centos")
            _rpm_version "centos-release"
            ;;

        "$g_distro_name_amazon")
            _rpm_version "system-release"
            ;;

        "$g_distro_name_suse" | \
        "$g_distro_name_opensuse")
            _rpm_version "sles-release"
            ;;

        "$g_distro_name_ubuntu" | \
        "$g_distro_name_thinpro")
            grep -w "DISTRIB_RELEASE" "$host_root/etc/lsb-release" | cut -d'=' -f2
            ;;

        "$g_distro_name_debian")
            cat "$host_root/etc/debian_version"
            ;;

        "$g_distro_name_gcos")
            declare -r milestone="$(grep -w "^CHROMEOS_RELEASE_CHROME_MILESTONE=" "$host_root/etc/lsb-release" | cut -d'=' -f2)"
            declare -r build="$(grep -w "^CHROMEOS_RELEASE_BUILD_NUMBER=" "$host_root/etc/lsb-release" | cut -d'=' -f2)"
            echo "$milestone.$build"
            ;;

        *)
            echo ""
            ;;
    esac
}

resolve_distro_version() {

    declare -r distro="${distro:-$(resolve_distro_name)}"

    declare version="$(_resolve_distro_version_from_os_release "$distro")"
    if [[ -z "$version" ]]; then
        version="$(_resolve_distro_version_legacy "$distro")"
    fi

    echo "${version:-$g_distro_version_unknown}"
}

resolve_distro_major_version() {

    declare -r os_full_version="${os_full_version:-$(resolve_distro_version)}"

    if [[ "$os_full_version" =~ ^([0-9]+).*$ ]]; then
        declare -r os_version="${BASH_REMATCH[1]}"
    elif [[ -n "$os_full_version" ]]; then
        echo "[!] Detected non-numeric distro version ($os_full_version)"
        declare -r os_version=0
    fi

    echo "$os_version"
}

get_config() {
    declare -r filename="$1"
    declare -r element="$2"

    sed -n \
        -e "s/.*<$element>\(.*\)<\/$element>.*/\1/p" \
        "$filename"
}
