#! /bin/bash

[[ -n "${TRACE:-}" ]] && set -x

set -u

_check_kernel_version() {
    declare -r kernel_major_min=2
    declare -r kernel_minor_min=6
    declare -r kernel_patch_min=32

    declare -r kernel_version="$(uname -r)"
    if [[ ! "$kernel_version" =~ ([0-9]+)\.([0-9]+)\.([0-9]+).* ]]; then
        return 1
    fi

    declare -r major="${BASH_REMATCH[1]}"
    declare -r minor="${BASH_REMATCH[2]}"
    declare -r patch="${BASH_REMATCH[3]}"

    if [[ "$major" -lt "$kernel_major_min" ]]; then
        return 1
    elif [[ "$major" -eq "$kernel_major_min" ]];then
        if [[ "$minor" -lt "$kernel_minor_min" ]]; then
            return 1
        elif [[ "$minor" -eq "$kernel_minor_min" ]]; then
            if [[ "$patch" -lt "$kernel_patch_min" ]]; then
                return 1
            fi
        fi
    fi

    return 0
}

_rpm_get_desired_arch() {
    arch=$(rpm -q --qf "%{arch}\n" "$1" 2>/dev/null)
    if [[ "$?" -ne 0 ]]; then
        echo "UNKNOWN"
    elif [[ "$arch" == "noarch" ]]; then
        echo "noarch"
    elif [[ "$arch" == "aarch64" ]]; then
        echo "aarch64"
    else
        echo "x86_64"
    fi
}

_rpm_verify_pkg() {
    arch=$(_rpm_get_desired_arch "$1")
    result=$(rpm -q --qf "%{name}\n" "$1.$arch" 2>/dev/null)
    if [[ "$result" != "$1" ]]; then
        return 1
    fi
}

_rpm_get_pkg_version() {
    arch=$(_rpm_get_desired_arch "$1")
    result=$(rpm -q --qf "%{version}\n" "$1.$arch" 2>/dev/null)
    echo "$result"
}

_deb_get_desired_arch() {
    arch=$(dpkg-query -W -f='${Architecture}\n' "$1" 2>/dev/null)
    if [[ "$?" -ne 0 ]]; then
        echo "UNKNOWN"
    elif [[ "$arch" == "all" ]]; then
        echo "all"
    elif [[ "$arch" == "arm64" ]]; then
        echo "arm64"
    else
        echo "amd64"
    fi
}

_deb_verify_pkg() {
    arch=$(_deb_get_desired_arch "$1")
    result=$(dpkg-query -W -f='${Status}\n' "${1}:$arch" 2>/dev/null)
    if [[ "$result" != "install ok installed" && "$result" != "hold ok installed" ]]; then
        return 1
    fi
}

_deb_get_pkg_version() {
    arch=$(_deb_get_desired_arch "$1")
    result=$(dpkg-query -W -f='${Version}\n' "${1}:$arch" 2>/dev/null)
    echo "$result"
}

_tdnf_get_desired_arch() {
    arch=$(tdnf list installed | grep -E "^$1\." | awk -F'.' '{print $2}' | awk '{print $1}' 2>/dev/null)
    if [[ "$?" -ne 0 ]]; then
        echo "UNKNOWN"
    elif [[ -z "$arch" ]]; then
        echo "UNKNOWN"
    elif [[ "$arch" == "noarch" ]]; then
        echo "noarch"
    elif [[ "$arch" == "aarch64" ]]; then
        echo "aarch64"
    else
        echo "x86_64"
    fi
}

_tdnf_verify_pkg() {
    arch=$(_tdnf_get_desired_arch "$1")
    result=$(tdnf list installed | grep -E "^$1\.$arch" 2>/dev/null)
    if [[ -z "$result" ]]; then
        return 1
    fi
    return 0
}

_tdnf_get_pkg_version() {
    arch=$(_tdnf_get_desired_arch "$1")
    result=$(tdnf list installed | grep -E "^$1\.$arch" | awk '{print $2}' 2>/dev/null)
    echo "$result"
}

_resolve_distribution_operations() {
    case "$distro" in
        "$g_distro_name_redhat"     | \
        "$g_distro_name_centos"     | \
        "$g_distro_name_oracle"     | \
        "$g_distro_name_amazon"     | \
        "$g_distro_name_suse"       | \
        "$g_distro_name_opensuse"   | \
        "$g_distro_name_almalinux"  | \
        "$g_distro_name_rockylinux" | \
        "$g_distro_name_centos_stream" | \
        "$g_distro_name_mariner")
            verify_pkg() { _rpm_verify_pkg "$1"; }
            get_pkg_version() { _rpm_get_pkg_version "$1"; }
            ;;

        "$g_distro_name_ubuntu" | \
        "$g_distro_name_debian" | \
        "$g_distro_name_thinpro")
            verify_pkg() { _deb_verify_pkg "$1"; }
            get_pkg_version() { _deb_get_pkg_version "$1"; }
            ;;

        "$g_distro_name_photon")
            verify_pkg() { _tdnf_verify_pkg "$1"; }
            get_pkg_version() { _tdnf_get_pkg_version "$1"; }
            ;;
        *)

            verify_pkg() { return 1; }
            get_pkg_version() { echo ""; }
            ;;
    esac
}

_verify_required_libraries() {
    declare -a prereq_packages=(
        "openssl"
    )

    # Resolve distribution name, version and operations.
    declare -r distro="$(resolve_distro_name)"
    declare -r os_version="$(resolve_distro_major_version)"
    _resolve_distribution_operations

    if [[ "$distro" == "suse" && "$os_version" -eq 11 ]]; then
        prereq_packages+=("openssl-certs")
    else
        prereq_packages+=("ca-certificates")
    fi

    if $has_selinux; then
        case "$distro" in
            "$g_distro_name_redhat" | \
            "$g_distro_name_centos" | \
            "$g_distro_name_oracle" | \
            "$g_distro_name_almalinux" | \
            "$g_distro_name_rockylinux" | \
            "$g_distro_name_centos_stream")
                if [[ "$os_version" -le 7 ]]; then
                    prereq_packages+=("policycoreutils-python")
                else
                    prereq_packages+=("policycoreutils-python-utils")
                fi

                if [[ "$os_version" -ge 7 ]]; then
                    prereq_packages+=("selinux-policy-devel" "selinux-policy-targeted")
                fi
                ;;

            "$g_distro_name_amazon")
                if [[ "$os_version" -eq 1 ]]; then
                    prereq_packages+=("policycoreutils-python" "selinux-policy")
                elif [[ "$os_version" -eq 2 ]]; then
                    prereq_packages+=("policycoreutils-python" "selinux-policy-devel")
                else
                    prereq_packages+=("policycoreutils-python-utils" "selinux-policy-devel")
                fi
                ;;

            "$g_distro_name_suse" | \
            "$g_distro_name_opensuse")
                prereq_packages+=("policycoreutils-python" "selinux-policy-devel")
                ;;

            "$g_distro_name_mariner")
                prereq_packages+=("policycoreutils-python-utils" "selinux-policy-devel")
                ;;

            "$g_distro_name_photon")
                prereq_packages+=("policycoreutils" "selinux-policy-devel")
                ;;

            "$g_distro_name_ubuntu" | \
            "$g_distro_name_debian" | \
            "$g_distro_name_thinpro")
                prereq_packages+=("policycoreutils" "selinux-policy-dev")
                ;;

            *)
                ;;
        esac
    fi

    case "$distro" in
        "$g_distro_name_redhat" | \
        "$g_distro_name_centos" | \
        "$g_distro_name_oracle" | \
        "$g_distro_name_almalinux" | \
        "$g_distro_name_rockylinux" | \
        "$g_distro_name_centos_stream")
            echo "Verifying RHEL/CentOS/Oracle/AlmaLinux/RockyLinux/CentOS_Stream $os_version (rpm) packages:"
            ;;

        "$g_distro_name_amazon")
            echo "Verifying Amazon $os_version (rpm) packages:"
            ;;

        "$g_distro_name_suse" | \
        "$g_distro_name_opensuse")
            echo "Verifying OpenSuSE/SLES $os_version (rpm) packages:"
            ;;

        "$g_distro_name_mariner")
            echo "Verifying Mariner $os_version (rpm) packages:"
            ;;

        "$g_distro_name_ubuntu")
            echo "Verifying Ubuntu $os_version (dpkg) packages:"
            ;;

        "$g_distro_name_debian")
            echo "Verifying Debian $os_version (dpkg) packages:"
            ;;

        "$g_distro_name_thinpro")
            echo "Verifying ThinPro $os_version (dpkg) packages:"
            ;;

        "$g_distro_name_photon")
            echo "Verifying Photon $os_version (tdnf) packages:"
            ;;
        *)
            notice_bad "Cannot verify packages for distro ($distro)"
            return 1
            ;;
    esac

    for package in "${prereq_packages[@]}"; do
        echo -n "  * $package ... "
        verify_pkg "$package"
        if [[ $? -eq 0 ]]; then
            notice_good "OK"
        else
            notice_bad "MISSING"
            return 1
        fi
    done

    declare -r openssl_version="$(get_pkg_version "openssl")"

    if [[ "$distro" == "suse" && "$os_version" -eq 11 ]]; then
        declare -r openssl_min_ver="0.9.8"
    else
        declare -r openssl_min_ver="1.0.0"
    fi

    if ! _is_version_ge "$openssl_version" "$openssl_min_ver"; then
        error "Installed 'openssl' version $openssl_version is not supported (<$openssl_min_ver)"
        return 1
    fi

    case "$distro" in
        "$g_distro_name_redhat" | \
        "$g_distro_name_centos" | \
        "$g_distro_name_oracle" | \
        "$g_distro_name_amazon" | \
        "$g_distro_name_almalinux" | \
        "$g_distro_name_rockylinux" | \
        "$g_distro_name_centos_stream" | \
        "$g_distro_name_mariner")
            if [[ ! -f "/etc/pki/tls/certs/ca-bundle.trust.crt" ]]; then
                error "Trusted certificate store missing"
                return 1
            fi
            ;;

        "$g_distro_name_ubuntu" | \
        "$g_distro_name_debian" | \
        "$g_distro_name_thinpro")
            if [[ ! -f "/etc/ssl/certs/ca-certificates.crt" ]]; then
                error "Trusted certificate store missing"
                return 1
            fi
            ;;

        "$g_distro_name_suse")
            if [[ ! -d "/etc/ssl/certs/" ]]; then
                error "Trusted certificate directory missing"
                return 1
            fi
            ;;

        *)
            echo "Skipping certificate store verification"
            ;;
    esac

    return 0
}

pre_install() {
    declare -r should_verify_requirements="$1"

    if [[ "$EUID" -ne 0 ]]; then
        echo "Please run Cortex XDR installer as root"
        return 1
    fi

    if ! _check_kernel_version; then
        echo "System kernel version is not supported"
        return 1
    fi

    if $should_verify_requirements; then
        step_start "Checking prerequisites"

        _verify_required_libraries
        if [[ "$?" -ne 0 ]]; then
            echo "Prerequisites not met. Please install missing packages"
            return 1
        fi

        step_end
    else
        step_start "Skipping prerequisites check"

    fi

    if ! $upgrade_traps && ! is_valid_proxy_list "$proxy_list"; then
        error "Invalid proxy list: $proxy_list"
        return 1
    fi

    if [[ -z "$install_path" ]]; then
        install_path="$deploy_dir"
    fi

    if $upgrade_traps; then
        if [[ "$deploy_dir" != "$install_path" ]]; then
            echo "Custom installation path used only on initial install. Ignoring."
        fi
        return 0
    fi

    if $btrfs_subvol; then
        step_start "Creating btrfs subvolume at $deploy_dir"
        if ! _create_btrfs_subvolume "$deploy_dir"; then
            return 1
        fi
        step_end
    fi

    if [[ -e "$deploy_dir" ]]; then
        if [[ "$deploy_dir" != "$install_path" ]]; then

            echo "Found existing installation path at: $deploy_dir, ignoring custom path"
        fi
        return 0
    fi

    if ! _create_deploy_dir; then

        return 1
    fi

    return 0
}
