#! /bin/bash

[[ -n "${TRACE:-}" ]] && set -x

set -u

_trim_quotes() {
    declare proxy_list="$1"
    while [[ "$proxy_list" =~ \"(.*)\" ]] || [[ "$proxy_list" =~ \'(.*)\' ]]; do
        proxy_list="${BASH_REMATCH[1]}"
    done
    echo "$proxy_list"
}

ipv4_regex() {
    declare -r regex_octet="([0-9]|[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])"
    echo "(($regex_octet\.){3}$regex_octet)"
}

ipv6_regex() {
    declare -r regex_ipv4="$(ipv4_regex)"
    declare re="("
    re="${re}([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|"               # 1:2:3:4:5:6:7:8
    re="${re}([0-9a-fA-F]{1,4}:){1,7}:|"                              # 1::                              1:2:3:4:5:6:7::
    re="${re}([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|"              # 1::8             1:2:3:4:5:6::8  1:2:3:4:5:6::8
    re="${re}([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|"       # 1::7:8           1:2:3:4:5::7:8  1:2:3:4:5::8
    re="${re}([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|"       # 1::6:7:8         1:2:3:4::6:7:8  1:2:3:4::8
    re="${re}([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|"       # 1::5:6:7:8       1:2:3::5:6:7:8  1:2:3::8
    re="${re}([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|"       # 1::4:5:6:7:8     1:2::4:5:6:7:8  1:2::8
    re="${re}[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|"            # 1::3:4:5:6:7:8   1::3:4:5:6:7:8  1::8
    re="${re}:((:[0-9a-fA-F]{1,4}){1,7}|:)|"                          # ::2:3:4:5:6:7:8  ::2:3:4:5:6:7:8 ::8       ::
    re="${re}[fF][eE]80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|"    # fe80::7:8%eth0   fe80::7:8%1     (link-local IPv6 addresses with zone index)
    re="${re}::([fF]{4}(:0{1,4}){0,1}:){0,1}${regex_ipv4}|"           # ::255.255.255.255   ::ffff:255.255.255.255  ::ffff:0:255.255.255.255  (IPv4-mapped IPv6 addresses and IPv4-translated addresses)
    re="${re}([0-9a-fA-F]{1,4}:){1,4}:${regex_ipv4}"                  # 2001:db8:3:4::192.0.2.33  64:ff9b::192.0.2.33 (IPv4-Embedded IPv6 Address)
    echo "$re)"
}

is_valid_proxy_list() {
    declare -r regex_port="([1-9][0-9]{0,3}|[1-5][0-9]{4}|6[0-4][0-9]{3}|65[0-4][0-9]{2}|655[0-2][0-9]|6553[0-5])"
    declare -r regex_fqdn="([a-zA-Z0-9](([a-zA-Z0-9_-]){0,61}[a-zA-Z0-9])?(\.)?)+[a-zA-Z]{2,}"
    declare -r regex_ipv4="$(ipv4_regex)"
    declare -r regex_ipv6="$(ipv6_regex)"
    declare -r regex_ipv4_port="${regex_ipv4}:${regex_port}"
    declare -r regex_ipv6_port="\[${regex_ipv6}\]:${regex_port}"
    declare -r regex_fqdn_port="${regex_fqdn}:${regex_port}"

    declare -r regex_proxy_item="$regex_ipv4_port|$regex_ipv6_port|$regex_fqdn_port"
    declare -r regex_proxy_list="^($regex_proxy_item)(,($regex_proxy_item))*$"

    declare -r proxy_list="$(_trim_quotes "$1")"
    if [[ -z "$proxy_list" ]]; then
        return 0
    fi

    [[ "$proxy_list" =~ $regex_proxy_list ]]
}

configure_proxy_list() {
    declare -r upgrade_traps="$upgrade_traps"
    declare -r config_trapsd="$config_trapsd"
    declare -r proxy_list="$(_trim_quotes "$1")"

    # Make sure the element always exists.
    if ! config_has "$config_trapsd" "proxy_list"; then
        config_append "$config_trapsd" "proxy_list" ""
    fi

    if $upgrade_traps; then
        return 0
    fi

    if [[ -z "$proxy_list" ]]; then
        return 0
    fi

    step_start "Setting proxy list"
    config_set "$config_trapsd" "proxy_list" "$proxy_list"
    step_end

    return 0
}
