#! /bin/bash

[[ -n "${TRACE:-}" ]] && set -x

set -u

# Prints text in Green
notice_good() {
    echo -e "\e[1;32m$@\e[0m"
}

# Prints text in Yellow
notice_warn() {
    echo -e "\e[1;33m$@\e[0m"
}

# Prints text in Red
notice_bad() {
    echo -e "\e[1;31m$@\e[0m"
}

error() {
    # Like `notice_bad`, but indented.
    notice_bad "  $@"
}

prompt() {
    echo -ne "\e[1;34m$@\e[0m "
}

yesno() {
    prompt "$@"
    read -r -n 1 choice
    echo

    case "$choice" in
        y|Y)
            return 0
            ;;
    esac
    return 1
}

step_start() {
    printf "\e[1;34m[%2d] %s\e[0m\n" "${step:=1}" "$@"
    step=$[step+1]
}

step_end() {
    echo -e "\e[1;32mDone\e[0m"
}

print_file() {
    declare -r file_path="$1"
    echo "Contents of $file_path:"
    cat "$file_path" | sed 's/^/    /'
}

_get_canonical_path() {
    readlink -m "$1"
}

get_package_state() {
    if [[ ! -d "$deploy_dir" ]]; then
        echo ""
        return
    fi

    declare -r -a deploy_dir_contents=($(find "$deploy_dir/" \
        -mindepth 1 -maxdepth 1 \
        -type d))
    if [[ "${#deploy_dir_contents[@]}" -eq 0 ]]; then

        :
    elif [[ "${#deploy_dir_contents[@]}" -gt 1 ]]; then

        echo "installed"
        return
    elif [[ "${deploy_dir_contents[0]}" == "$deploy_dir/rpm-installer" ]]; then
        echo "rpm-intermediate"
        return
    elif [[ "${deploy_dir_contents[0]}" == "$deploy_dir/deb-installer" ]]; then
        echo "deb-intermediate"
        return
    fi

    echo ""
}

_is_version_ge() {
    declare -r first="$1"
    declare -r second="$2"

    if [[ ! "$first" =~ ^([0-9]+)\.([0-9]+)\.([0-9]+) ]]; then
        echo "Cannot parse version from '$first'"
        return 1
    fi

    declare -r first_major="${BASH_REMATCH[1]}"
    declare -r first_minor="${BASH_REMATCH[2]}"
    declare -r first_patch="${BASH_REMATCH[3]}"

    if [[ ! "$second" =~ ^([0-9]+)\.([0-9]+)\.([0-9]+) ]]; then
        echo "Cannot parse version from '$second'"
        return 1
    fi

    declare -r second_major="${BASH_REMATCH[1]}"
    declare -r second_minor="${BASH_REMATCH[2]}"
    declare -r second_patch="${BASH_REMATCH[3]}"

    if [[ "$first_major" -lt "$second_major" ]]; then
        return 1
    elif [[ "$first_major" -gt "$second_major" ]]; then
        return 0
    fi

    if [[ "$first_minor" -lt "$second_minor" ]]; then
        return 1
    elif [[ "$first_minor" -gt "$second_minor" ]]; then
        return 0
    fi

    if [[ "$first_patch" -lt "$second_patch" ]]; then
        return 1
    fi

    return 0
}

_create_deploy_dir() {

    mkdir -p -m 755 "$(dirname "$deploy_dir")"

    declare -r install_dirname="$(dirname "$install_path")"
    if [[ ! -d "$install_dirname" ]]; then
        error "Installation directory: $install_dirname wasn't found, Make sure it exists before installing."
        return 1
    fi

    dir_iter="$(_get_canonical_path "$install_dirname")"
    while [[ "$dir_iter" != "/" ]]; do
        perms="$(stat -c '%A' "$dir_iter")"
        if [[ ! "$perms" =~ ^......x..x$ ]]; then
            error "Installation directory should be accessible for user traversing. Make sure all directory components have at least: ------x--x"
            return 1
        fi
        dir_iter="$(dirname "$dir_iter")"
    done

    if ! mkdir -p "$install_path"; then
        error "Failed creating installation directory at: $install_path"
        return 1
    fi

    if [[ "$deploy_dir" != "$install_path" ]]; then
        echo "Linking custom installation directory: '$install_path'"
        if ! ln -s -f -n "$install_path" "$deploy_dir"; then
            error "Failed creating link"
            return 1
        fi
    fi

    return 0
}

_create_btrfs_subvolume() {
    if [[ -e "$deploy_dir" ]]; then
        echo "Failed to create btrfs subvolume. Path $deploy_dir already in-use"
        return 1
    fi

    declare -r deploy_dir_parent="$(dirname "$deploy_dir")"
    mkdir -p "$deploy_dir_parent"

    declare -r partition_type="$(stat -f -c "%T" "$deploy_dir_parent")"
    if [[ "$partition_type" == "btrfs" ]]; then
        if ! btrfs subvolume create "$deploy_dir"; then
            return 1
        fi
    fi

    return 0
}

strip_quotes() {
    declare input="${1%\"}"
    input="${input#\"}"
    echo "$input"
}
