#! /bin/bash

[[ -n "${TRACE:-}" ]] && set -x

set -u

declare start_services=true
declare upgrade_traps=false
declare verify_required_pkgs=true
declare has_apparmor=false
declare has_selinux=false
declare distribution_id=""
declare distribution_server=""
declare endpoint_tags=""
declare upgrade_esm2tms=false
declare unprivileged_user="cortexuser"
declare proxy_list=""
declare restrict_all=false
declare restrict_live_terminal=false
declare restrict_script_execution=false
declare restrict_file_retrieval=false
declare is_temporary_session=false
declare is_vm_template=false
declare btrfs_subvol=false
declare install_path=""
declare override_trapsd_config_path=""
declare km_enabled=true
declare is_rpm=false
declare is_deb=false
declare in_container=false
declare compat_libs=false

_usage() {
    echo "Usage: $0 -- [-huaAsSK] [-d DIST] [-x PROXIES] [--restrict=<RESTRICTION>]"
    echo ""
    echo "OPTIONS:"
    echo "  --upgrade, -u                       Upgrade a Cortex XDR installation."
    echo "  --[no-]apparmor, -a/-A              Enable/disable AppArmor policy installation."
    echo "  --[no-]selinux, -s/-S               Enable/disable SELinux policy installation."
    echo "  --no-km, -K                         Don't install kernel module."
    echo "  --dist-id DIST, -d DIST             Set distribution ID for an upgrade from ESM to TMS."
    echo "  --unprivileged-user NAME            Set the name of the unprivileged auxiliary user."
    echo "  --proxy-list PROXIES, -x PROXIES    Set proxy list (delimited by commas)."
    echo "  --restrict <RESTRICTION>            Restrict actions on the agent-side. RESTRICTION"
    echo "                                          can be any of 'live_terminal', 'script_execution',"
    echo "                                          'file_retrieval' or 'all'. Can be specified"
    echo "                                          multiple times."
    echo "  --endpoint-tags TAGS                Set tags (comma delimited list) for this endpoint."
    echo "  --temporary-session, --ts, -t       Configure as a temporary session."
    echo "  --vm-template, --vm, -v             Configure as a VM template."
    echo "  --btrfs-subvol                      If installing on btrfs filesystem, create install directory as a separate subvolume."
    echo "  --install-path                      Installation path (default: /opt/traps)"
    echo "  --override-trapsd-config-path       Absolute path to a config xml file whose entries override those in trapsd.xml"
    echo "  --library-compatibility, --lc       Use packaged compatibility libraries"
}

_set_restriction() {
    declare -r restriction="$1"
    case "$restriction" in
        all)
            restrict_all=true
            ;;

        live_terminal)
            restrict_live_terminal=true
            ;;

        script_execution)
            restrict_script_execution=true
            ;;

        file_retrieval)
            restrict_file_retrieval=true
            ;;

        *)
            return 1
            ;;
    esac
}

load_arguments_file() {
    declare -r arguments_path="/etc/panw/cortex.conf"
    if [[ ! -e "$arguments_path" ]]; then

        return 0
    fi

    owner_uid="$(stat -c "%u" "$arguments_path")"
    owner_gid="$(stat -c "%g" "$arguments_path")"
    mode="$(stat -c "%A" "$arguments_path")"
    if [[ ! -f "$arguments_path" ]]; then
        notice_bad "Ignoring $arguments_path since not a regular file $(stat -c "%F" "$arguments_path")"
        return 0
    elif [[ "$owner_uid" -ne 0 || "$owner_gid" -ne 0 ]]; then
        notice_bad "Ignoring $arguments_path since not owned by root (${owner_uid}:${owner_gid})"
        return 0
    elif [[ "$mode" =~ ........w. ]]; then
        notice_bad "Ignoring $arguments_path since world-writable ($mode)"
        return 0
    fi

    xargs --arg-file "$arguments_path" bash -c 'printf '\''%s\n'\'' "$@"' _
}

parse_installer_arguments() {
    # Not using `declare` since it hides the return value.
    options="$(getopt --name "$0" \
        -q \
        --options "hnuaAsSKd:x:Ptv" \
        --longoptions "help" \
        --longoptions "upgrade,no-km,dist-id:" \
        --longoptions "unprivileged-user:" \
        --longoptions "proxy-list:" \
        --longoptions "distribution-id:,distribution-server:" \
        --longoptions "endpoint-tags:" \
        --longoptions "apparmor,no-apparmor,selinux,no-selinux" \
        --longoptions "no-start,skip-prereq" \
        --longoptions "restrict:" \
        --longoptions "rpm,deb" \
        --longoptions "temporary-session,ts" \
        --longoptions "vm-template,vm" \
        --longoptions "btrfs-subvol" \
        --longoptions "install-path:" \
        --longoptions "override-trapsd-config-path:" \
        --longoptions "library-compatibility,lc" \
        --longoptions "in-container" \
        -- "$@")"
    if [[ $? -ne 0 ]]; then
        return 2
    fi

    # Re-arrange positional arguments.
    eval set -- "$options"

    if [[ -z "$distribution_id" ]] && [[ -f ".dist_id" ]]; then
        distribution_id="$(cat ".dist_id")"
    fi

    if [[ -z "$distribution_server" ]] && [[ -f ".dist_server" ]]; then
        distribution_server="$(cat ".dist_server")"
    fi

    if apparmor_status > /dev/null 2>&1; then
        has_apparmor=true
    fi

    if getenforce > /dev/null 2>&1; then
        if [[ "$(getenforce)" != "Disabled" ]]; then
            has_selinux=true
        fi
    elif [[ -e "/sys/fs/selinux/enforce" ]] ; then
        echo "NOTE: SELinux is enabled in kernel, but selinux-utils are missing"
    fi

    if [[ -e "/sys/kernel/security/lsm" ]] ; then
        declare -r lsm=$(cat "/sys/kernel/security/lsm")
        echo "Active kernel LSM: $lsm"
    fi

    while true; do
        case "$1" in
            -h | --help)
                _usage
                # We exit with an error code so that the cloud installer
                # will know to stop as well.
                return 2
                ;;

            -u | --upgrade)
                if [[ -d "$deploy_dir" || -L "$deploy_dir" ]] && ! $upgrade_esm2tms; then
                    upgrade_traps=true
                fi
                shift
                ;;

            -a | --apparmor)
                has_apparmor=true
                shift
                ;;

            -A | --no-apparmor)
                has_apparmor=false
                shift
                ;;

            -s | --selinux)
                has_selinux=true
                shift
                ;;

            -S | --no-selinux)
                has_selinux=false
                shift
                ;;

            -K | --no-km)
                km_enabled=false
                shift
                ;;

            -d | --dist-id)
                distribution_id="$2"
                upgrade_esm2tms=true
                upgrade_traps=false
                shift 2
                ;;

            --distribution-id)
                distribution_id="$2"
                shift 2
                ;;

            --distribution-server)
                distribution_server="$2"
                shift 2
                ;;

            --endpoint-tags)
                endpoint_tags="$2"
                shift 2
                ;;

            --unprivileged-user)
                if $upgrade_traps; then
                    echo "Can't set unprivileged username on upgrades, ignoring"
                elif [[ "$2" == "root" ]]; then

                    echo "Must provide an unprivileged user, not 'root'"
                    return 1
                fi
                unprivileged_user="$2"
                shift 2
                ;;

            -x | --proxy-list)
                proxy_list="$2"
                shift 2
                ;;

            --restrict)
                if ! _set_restriction "$2"; then
                    echo "Unknown restriction: $2"
                    return 1
                fi
                shift 2
                ;;

            --temporary-session | --ts | -t)
                is_temporary_session=true
                shift
                ;;

            --vm-template | --vm | -v)
                is_vm_template=true
                shift
                ;;

            --btrfs-subvol)
                btrfs_subvol=true
                shift
                ;;

            --install-path)
                install_path="${2%/}/traps"
                shift 2
                ;;

            --override-trapsd-config-path)
                override_trapsd_config_path="$2"
                shift 2
                ;;

            --library-compatibility | --lc)
                compat_libs=true
                shift
                ;;

            -n | --no-start)

                start_services=false
                shift
                ;;

            -P | --skip-prereq)

                verify_required_pkgs=false
                shift
                ;;

            --rpm)

                is_rpm=true
                shift
                ;;

            --deb)

                is_deb=true
                shift
                ;;

            --in-container)

                in_container=true
                shift
                ;;

            --)
                shift
                break
                ;;

            *)
                _usage
                # We exit with an error code so that the cloud installer
                # will know to stop as well.
                return 2
                ;;
        esac
    done

    if [[ -n "$@" ]]; then
        error "Unknown option/s: $@"
        return 2
    fi

    if $upgrade_traps && $btrfs_subvol; then
        error "Cannot create a new btrfs subvolume when upgrading. Ignoring."
        btrfs_subvol=false
    fi
}
