#! /bin/bash

[[ -n "${TRACE:-}" ]] && set -x

set -u

_is_glibc_version_ge_2_19() {

    declare -r maps_out="$(cat "/proc/self/maps")"
    declare -r libc_standard_regex="^.*/libc-([0-9]+)\.([0-9]+)(\.[^/]+)?.so\b"

    declare -r libc_link_regex="^[^/]+(.*/libc\..*)\b"

    if [[ "$maps_out" =~ $libc_standard_regex ]]; then
        declare -r glibc_major="${BASH_REMATCH[1]}"
        declare -r glibc_minor="${BASH_REMATCH[2]}"
        echo "Detected glibc v$glibc_major.$glibc_minor directly via maps file" >> "$install_log"

    elif [[ "$maps_out" =~ $libc_link_regex ]]; then
        declare -r libc_link="${BASH_REMATCH[1]}"
        declare -r linkedname="$(readlink --canonicalize "$libc_link")"

        if [[ "$linkedname" =~ $libc_standard_regex ]]; then
            declare -r glibc_major="${BASH_REMATCH[1]}"
            declare -r glibc_minor="${BASH_REMATCH[2]}"
            echo "Detected glibc v$glibc_major.$glibc_minor via link" >> "$install_log"
        else

            declare -r conf_out="$(getconf GNU_LIBC_VERSION 2>/dev/null)"
            declare -r conf_regex="^glibc[[:space:]]([0-9]+)\.([0-9]+)\b"

            if [[ "$conf_out" =~ $conf_regex ]]; then
                declare -r glibc_major="${BASH_REMATCH[1]}"
                declare -r glibc_minor="${BASH_REMATCH[2]}"
                echo "Detected glibc v$glibc_major.$glibc_minor via getconf"

            else
                echo "Unable to resolve glibc via link & getconf, assuming new glibc"
                return 2
            fi
        fi
    else
        echo "Failed to resolve glibc directly, assuming new glibc"
        return 2
    fi

    if [[ "$glibc_major" -gt 2 ]]; then
        return 0
    fi
    if [[ "$glibc_major" -lt 2 ]]; then
        return 1
    fi

    if [[ "$glibc_minor" -ge 19 ]]; then
        return 0
    fi

    return 1
}

_patch_packaged_elf() {
    declare -r elf="$1"
    declare -r nodeflib="$2"
    declare -r interpreter="$3"
    declare -r rpath="$4"

    if [[ "$nodeflib" = true ]]; then
        $patchelf_tool --no-default-lib "$elf" || return 1
    fi

    if [[ -n "$rpath" ]]; then
        $patchelf_tool --set-rpath "$rpath" "$elf" || return 1
    fi

    if [[ -n "$interpreter" ]]; then
        $patchelf_tool --set-interpreter "$interpreter" "$elf" || return 1
    fi

    return 0
}

_fix_packaged_elf() {
    declare -r elf="$1"
    declare -r nodeflib="$2"
    declare -r interpreter="$3"
    declare -r rpath="$4"

    declare patch_ret=0
    declare chmod_ret=0

    chmod u+w "$elf" || return 1

    # If we fail we still want to chmod back.
    _patch_packaged_elf "$elf" "$nodeflib" "$interpreter" "$rpath" || patch_ret=$?

    # The patchelf error code trumps chmod's.
    chmod u-w "$elf" || chmod_ret=$?

    if [[ $patch_ret -ne 0 || $chmod_ret -ne 0 ]]; then
        error "Failed fixing ELF [$elf]"
        return 1
    fi

    return 0
}

_fix_packaged_elf_dir() {
    # Usage: _fix_packaged_elf_dir <directory> <64bit-interpreter> <64bit-rpath> <32bit-interpreter> <32bit-rpath> [skip [skip...]]
    # Iterate over all the files in `directory`, and for every ELF file that isn't one
    # of the `skipped` (which can be either a full path or a regex), patch it to use
    # the correct (32bit or 64bit) interpreter and rpath
    declare -r directory="$1"
    declare -r interpreter64="$2"
    declare -r rpath64="$3"
    declare -r interpreter32="$4"
    declare -r rpath32="$5"
    shift 5
    declare -r -a skipped=("${@:-}")

    # When using this method, we assume all ELF files should be patched so that they
    # don't use the default libraries
    declare -r dont_use_default_libs=true

    for file in $(find "$directory" -type f); do
        declare file_type="$(file "$file")"
        if [[ ! "$file_type" =~ "ELF" ]]; then
            continue
        elif [[ "$file_type" =~ "statically linked" ]]; then
            continue
        fi

        local should_skip=false
        for to_skip in "${skipped[@]+"${skipped[@]}"}"; do
            # We don't put quotes around `to_skip` since it can include wildcards
            if [[ "$file" = $to_skip ]]; then
                should_skip=true
                break
            fi
        done

        if $should_skip; then
            continue
        fi

        declare interpreter=""
        declare rpath=""

        if [[ "$file_type" =~ "ELF 64-bit" ]]; then
            interpreter="$interpreter64"
            rpath="$rpath64"
        elif [[ "$file_type" =~ "ELF 32-bit" ]]; then
            interpreter="$interpreter32"
            rpath="$rpath32"
        fi

        _fix_packaged_elf "$file" "$dont_use_default_libs" "$interpreter" "$rpath" || return 1
    done

    return 0
}

_add_needed_library_to_elf() {
    declare -r lib="$1"
    declare -r elf="$2"

    declare patch_ret=0
    declare chmod_ret=0

    chmod u+w "$elf" || return 1

    # If we fail we still want to chmod back.
    $patchelf_tool --add-needed "$lib" "$elf" || patch_ret=$?

    # The patchelf error code trumps chmod's.
    chmod u-w "$elf" || chmod_ret=$?

    if [[ $patch_ret -ne 0 || $chmod_ret -ne 0 ]]; then
        error "Failed to add needed library [$lib] to ELF [$elf]"
        return 1
    fi

    return 0
}

_add_needed_library() {
    # Usage: _add_needed_library <lib> [elf [elf...]]
    # Patch all the specified elfs to load the library
    declare -r lib="$1"
    shift 1
    declare -r -a elfs=("${@:-}")

    for elf in "${elfs[@]+"${elfs[@]}"}"; do
        _add_needed_library_to_elf "$lib" "$elf" || return 1
    done

    return 0
}

fix_packaged_binaries() {
    declare -r deploy_dir="$deploy_dir"

    declare -r patchelf_tool="$deploy_dir/bin/patchelf"

    declare -r use_default_libs=false # 'false' because the arguments is 'dont use default libs'

    declare -r dont_fix_64bit_interpreter=""
    declare -r dont_fix_64bit_rpath=""

    declare -r dont_fix_32bit_interpreter=""
    declare -r dont_fix_32bit_rpath=""

    declare -r glibc_dir="$deploy_dir/glibc"
    declare -r glibc_lib="$glibc_dir/lib/x86_64-linux-gnu"
    declare -r glibc_ld="$glibc_lib/ld-linux-x86-64.so.2"
    declare -r glibc_lib32="$glibc_dir/lib32"
    declare -r glibc_ld32="$glibc_lib32/ld-linux.so.2"
    # NOTE: glibc_dir/lib64 contains a single symbolic link, so no need to even consider it
    declare -r glibc_usr_lib="$glibc_dir/usr/lib/x86_64-linux-gnu"
    declare -r glibc_usr_lib_audit="$glibc_usr_lib/audit"
    declare -r glibc_usr_lib_gconv="$glibc_usr_lib/gconv"
    declare -r glibc_usr_lib32="$glibc_dir/usr/lib32"
    declare -r glibc_usr_lib32_gconv="$glibc_usr_lib32/gconv"
    declare -r glibc_all_lib="$glibc_lib:$glibc_usr_lib_audit:$glibc_usr_lib_gconv"
    declare -r glibc_all_lib32="$glibc_lib32:$glibc_usr_lib32_gconv"

    declare -r traps_lib="$deploy_dir/lib"
    declare -r traps_lib32="$deploy_dir/lib32"
    declare -r traps_bin="$deploy_dir/bin"
    declare -r traps_analyzerd="$deploy_dir/analyzerd"

    _fix_packaged_elf_dir "$glibc_dir" "$dont_fix_64bit_interpreter" "$glibc_all_lib" "$dont_fix_32bit_interpreter" "$glibc_all_lib32" '*/ld-2.19.so'
    if [[ $? -ne 0 ]]; then
        return 1
    fi

    _fix_packaged_elf_dir "$traps_analyzerd" "$glibc_ld" "$glibc_all_lib:$traps_lib" "$dont_fix_32bit_interpreter" "$dont_fix_32bit_rpath" '*/clad'
    if [[ $? -ne 0 ]]; then
        return 1
    fi

    _fix_packaged_elf_dir "$traps_bin" "$glibc_ld" "$glibc_all_lib:$traps_lib" "$glibc_ld32" "$glibc_all_lib32:$traps_lib32" '*/add_sepolicy' '*/add_sepolicy_ng'
    if [[ $? -ne 0 ]]; then
        return 1
    fi

    # We patch it separately, because it needs to use the default libraries.
    _fix_packaged_elf "$traps_bin/add_sepolicy" "$use_default_libs" "$glibc_ld" "$glibc_all_lib:$traps_lib"
    if [[ $? -ne 0 ]]; then
        return 1
    fi

    # We patch it separately, because it needs to use the default libraries.
    _fix_packaged_elf "$traps_bin/add_sepolicy_ng" "$use_default_libs" "$glibc_ld" "$glibc_all_lib:$traps_lib"
    if [[ $? -ne 0 ]]; then
        return 1
    fi

    # We don't modify interpreter for libmodule. It should NOT depend on ld-linux.so.2 since that causes segfaults.
    _fix_packaged_elf_dir "$traps_lib" "$dont_fix_64bit_interpreter" "$glibc_all_lib:$traps_lib" "$dont_fix_32bit_interpreter" "$dont_fix_32bit_rpath" "$libmodule_64" "$libmoduleng_64"
    if [[ $? -ne 0 ]]; then
        return 1
    fi

    # We don't modify 32bit interpreter for libmodule. It should NOT depend on ld-linux.so.2 since that causes segfaults.
    _fix_packaged_elf_dir "$traps_lib32" "$dont_fix_64bit_interpreter" "$dont_fix_64bit_rpath" "$dont_fix_32bit_interpreter" "$glibc_all_lib32:$traps_lib32" "$libmodule_32" "$libmoduleng_32"
    if [[ $? -ne 0 ]]; then
        return 1
    fi

    # This is here for two reasons:
    #  * CPA-6371 (CPA-6274/CPA-6203): ESET compatiblity issue, since they "/etc/ld.so.preload" their
    #    library to the whole system, and it requires libdl.so.2. Because it doesn't have NODEFLIB
    #    enabled, it loads libdl.so.2 from the default paths, which causes havoc and crashes our
    #    applications. So to combat that, we require libdl explicitly.
    #  * CPA-5080: libdl.so.2 is also required by libmodule, which doesn't set NODEFLIB, so same as
    #    above. This should be fixed properly in CPA-5082 when we "split" libmodule into a static
    #    and shared libraries.
    _add_needed_library "libdl.so.2" \
        "$traps_bin/pmd" \
        "$traps_bin/cytool" \
        "$traps_bin/shellcode_offset64" \
        "$traps_bin/shellcode_offset32" \
        "$traps_bin/add_aapolicy" \
        "$traps_bin/add_sepolicy" \
        "$traps_bin/add_aapolicy_ng" \
        "$traps_bin/add_sepolicy_ng" \
        "$traps_analyzerd/spmd"
    if [[ $? -ne 0 ]]; then
        return 1
    fi

    return 0
}
