#! /bin/bash

[[ -n "${TRACE:-}" ]] && set -x

set -u

vet_path() {
    # Vet PATH to only allow non world-writable root directories.
    declare -r -a allowed_paths=(
        "/bin"
        "/sbin"
        "/usr/bin"
        "/usr/sbin"
    )

    # This is true on all supported distros.
    declare -r abs_stat="/usr/bin/stat"

    if [[ -z "${XDR_PATH_VET_SILENT:-}" ]]; then
        _warn() {
            echo "[!] $@" >&2
        }
    else
        _warn() {
            :
        }
    fi

    # Split by the ':' delimiter.
    declare -a path_final=()
    for curr_path in "${allowed_paths[@]}"; do
        # Drop directories outside of the machine's path.

        case ":${PATH}:" in
            *:${curr_path}:* | *:${curr_path}/:*)
                ;;
            *)
                _warn "Path '$curr_path' is not in PATH"
                continue
                ;;
        esac

        if [[ ! -d "$curr_path" ]]; then
            _warn "Path '$curr_path' doesn't exist"
            continue
        fi

        # Drop world-writable directories.
        owner_link_uid="$("$abs_stat" -c "%u" "$curr_path")"
        owner_link_gid="$("$abs_stat" -c "%g" "$curr_path")"
        owner_uid="$("$abs_stat" -Lc "%u" "$curr_path")"
        owner_gid="$("$abs_stat" -Lc "%g" "$curr_path")"
        mode="$("$abs_stat" -Lc "%A" "$curr_path")"
        if [[ "$owner_link_uid" -ne 0 || "$owner_link_gid" -ne 0 ]]; then
            _warn "Path '$curr_path' forbidden because not owned by root (${owner_link_uid}:${owner_link_gid})"
            continue
        elif [[ "$owner_uid" -ne 0 || "$owner_gid" -ne 0 ]]; then
            _warn "Path '$curr_path' forbidden because real path not owned by root (${owner_uid}:${owner_gid})"
            continue
        elif [[ "$mode" =~ ........w. ]]; then
            _warn "Path '$curr_path' forbidden because world-writable ($mode)"
            continue
        fi

        path_final+=("$curr_path")
    done

    if [[ "${#path_final[@]}" -eq 0 ]]; then
        _warn "No valid paths in \$PATH, aborting!"
        exit 1
    fi

    export PATH="$(IFS=":"; echo "${path_final[*]}")"
}

vet_path

unalias -a

source "./setup.d/log_redirect.sh"
source "./setup.d/utils.sh"
source "./setup.d/arguments.sh"
source "./setup.d/config.sh"
source "./setup.d/patches.sh"
source "./setup.d/apparmor.sh"
source "./setup.d/selinux.sh"
source "./setup.d/iptables.sh"
source "./setup.d/anti_malware_flow.sh"
source "./setup.d/daemons.sh"
source "./setup.d/proxy.sh"
source "./setup.d/fapolicyd.sh"
source "./setup.d/temporary_workloads.sh"
source "./scripts/distro_picker.sh"

declare g_config_backup_dir=""

_has_agent_installed() {
    if $upgrade_traps; then
        return 1
    elif [[ ! -d "$deploy_dir" ]]; then
        return 1
    fi

    declare -r state="$(get_package_state)"
    if [[ -z "$state" ]]; then
        return 1
    elif $is_rpm && [[ "$state" == "rpm-intermediate" ]]; then
        return 1
    elif $is_deb && [[ "$state" == "deb-intermediate" ]]; then
        return 1
    fi

    return 0
}

_resolve_lib_reqs() {
    local filename="$1"
    local req_directory="$config_dir/reqs"
    local req_filename="$req_directory/$(basename $filename).txt"
    declare -r list_reqs="$deploy_dir/scripts/list_requirements.awk"

    # NOTE: We declare the variable at a separate line because it otherwise overrides
    #       the exit code from the sub-shell
    declare required_libs=""
    required_libs="$(ldd "$filename" | awk -f "$list_reqs"; exit ${PIPESTATUS[0]})"
    if [[ $? -ne 0 ]]; then
        return 1
    fi

    declare -r -a required_libs_array=($required_libs)

    # Filter out libraries out of the "main" library paths. This makes sure we don't get
    # "ld-preloaded" somehow by 3rd party applications.
    declare -a filtered_libs=()
    for required_lib in "${required_libs_array[@]+"${required_libs_array[@]}"}"; do
        link_target=$(readlink -f "$required_lib")
        if [[ "$link_target" = /lib* ]]; then
            filtered_libs+=("$required_lib")
            continue
        elif [[ "$link_target" = /usr/lib* ]]; then
            filtered_libs+=("$required_lib")
            continue
        fi

        echo "  Ignoring non-canonical dependency $required_lib (->$link_target)"
    done

    mkdir -p "$req_directory"
    printf "%s\n" "${filtered_libs[@]+"${filtered_libs[@]}"}" > "$req_filename"
    chmod u=r,go-rwx "$req_filename" || return 1
    return 0
}

_persist_configs() {
    # For agents with "selinux_aware" variable
    # If no "selinux_aware" is present, it will put not disabled and will try fetching "selinux_disabled"
    if [[ "$(config_get "$g_config_backup_dir/config/common.xml" "selinux_aware")" == "0" ]]; then
        config_set "$config_common" "selinux_disabled" "1"
    else
        config_set "$config_common" "selinux_disabled" "0"
    fi

    declare -r -a persistent_configs_common=(
        "selinux_disabled"
    )
    for element in "${persistent_configs_common[@]}"; do
        config_pass "$g_config_backup_dir/config/common.xml" "$config_common" "$element"
    done

    declare -r -a persistent_configs_trapsd=(
        "distribution_id"
        "distribution_server"
        "endpoint_tags"
        "cloud_elb_address"
        "log_level"
        "proxy_list"
        "restrict_invasive_response_actions"
        "restrict_live_terminal"
        "restrict_script_execution"
        "restrict_file_retrieval"
        "ts_enabled"
        "vm_template"
    )
    for element in "${persistent_configs_trapsd[@]}"; do
        config_pass "$g_config_backup_dir/config/trapsd.xml" "$config_trapsd" "$element"
    done

    declare -r -a persistent_configs_pmd=(
        "km_enabled"
        "log_level"
        "nss_override"
        "edr_ignored_dirs"
    )
    for element in "${persistent_configs_pmd[@]}"; do
        config_pass "$g_config_backup_dir/config/pmd.xml" "$config_pmd" "$element"
    done

    if [[ -e "$g_config_backup_dir/config/authorized.xml" ]] ; then

        declare -r unprivileged_user_source_xml="$g_config_backup_dir/config/authorized.xml"
        rm -f "$config_authorized"
    else

        declare -r unprivileged_user_source_xml="$g_config_backup_dir/config/pmd.xml"
    fi

    declare -r -a persistent_configs_authorized=(
        "analyzerd_user"
    )
    for element in "${persistent_configs_authorized[@]}"; do
        config_pass "$unprivileged_user_source_xml" "$config_pmd" "$element"
    done
}

_remove_dir() {
    declare -r dir_name="$1"
    declare -r max_attempts=10
    declare attempt=0

    while ! rm -rf "$dir_name"; do
        if [[ "$attempt" -gt "$max_attempts" ]] ; then
            echo "Failed to remove $dir_name"
            _abort
        fi

        ((attempt=attempt+1))

        sleep 3
        echo "Failed to remove $dir_name, retrying..."
    done

    [[ "$attempt" -gt "0" ]] && echo "Successfully removed $dir_name"
    return 0
}

_do_upgrade() {

    step_start "Stopping daemons"
    "$deploy_dir/bin/cytool" runtime stop all || true
    step_end

    step_start "Removing old agent's files"

    declare -r -a directories_to_preserve=(
        "config"
        "persist"
        "scan"
        "download"
        "forensics"
        "selinux"
        "quarantine"
        "local_analysis"
        "edr"
        "ecl"
        "km"
        "rpm-installer"
        "deb-installer"
    )
    for dir_name in $(find "$deploy_dir" -mindepth 1 -maxdepth 1 -type d); do

        if [[ ! " ${directories_to_preserve[@]} " =~ " $(basename "$dir_name") " ]]; then
            _remove_dir "$dir_name"
        fi
    done

    if [[ ! -L "$deploy_dir/km" ]]; then
        rm -rf "$deploy_dir/km"
    fi

    if [[ -x "$km_manage" ]]; then
        "$km_manage" "uninstall"
    elif [[ -x "$deploy_dir/km_utils/module_install" ]]; then
        "$deploy_dir/km_utils/module_install" "uninstall"
    fi
    rm -f "$init_d_dir/traps_core" 2>/dev/null || true

    uninstall_retired_daemons
    step_end

    step_start "Backing up current agent's configuration"
    g_config_backup_dir="$(mktemp -d)"
    cp "$deploy_dir/version.txt" "$g_config_backup_dir/" || true
    cp -r "$config_dir/" "$g_config_backup_dir/" || true

    if [[ "$(config_get "$g_config_backup_dir/config/pmd.xml" "km_enabled")" == "0" ]]; then
        km_enabled=false
    fi
    step_end
}

_unpack_installation() {
    umask 0177

    tar --warning=no-timestamp --no-same-owner --no-same-permissions \
        -xzf "$agent_full_version.tar.gz" -C "$deploy_dir/"
    if [[ $? -ne 0 ]]; then
        rm -rf "$deploy_dir/"
        error "Couldn't unpack Cortex XDR."
        return 1
    fi

    find "$deploy_dir/" \
        -path "$deploy_dir/config/deployment" -prune \
        -o -exec chown --no-dereference root:root {} +

    find "$deploy_dir/" \
        -path "$deploy_dir/config/deployment" -prune\
        -o -exec chmod go-wrx {} +

    find "$deploy_dir/" \
        -path "$deploy_dir/config/deployment" -prune \
        -o -type d -exec chmod u+wrx {} +

    chmod    g+x,o+x  "$deploy_dir/"
    chmod    g+x,o+x  "$deploy_dir/policy/"
    chmod    g+x,o+x  "$deploy_dir/lib/"

    if [[ -e "$deploy_dir/lib32" ]]; then
        chmod    g+x,o+x  "$deploy_dir/lib32/"
    fi

    if [[ -e "$libmodule_32" ]]; then
        chmod    g+rx,o+rx  "$libmodule_32"
    fi

    if [[ -e "$libmoduleng_32" ]]; then
        chmod    g+r,o+r  "$libmoduleng_32"
    fi

    if [[ -e "$libmodule_64" ]]; then
        chmod    g+rx,o+rx  "$libmodule_64"
    fi

    if [[ -e "$libmoduleng_64" ]]; then
        chmod    g+r,o+r  "$libmoduleng_64"
    fi

    if [[ -e "$deploy_dir/glibc" ]]; then
        chmod -R     o+rx "$deploy_dir/glibc/"
    fi

    chmod -R 0700     "$deploy_dir/scripts/"
    chmod -R 0700     "$deploy_dir/bin/"
    chmod -R 0700     "$deploy_dir/km_utils/"

    return 0
}

_do_compat_libs() {
    # Fix packaged binaries for targets with libc version < 2.19
    if ! $compat_libs; then
        if _is_glibc_version_ge_2_19; then
            compat_libs=false
        else
            declare ret="$?"
            if [[ "$ret" -eq 1 ]]; then
                compat_libs=true
            elif [[ "$ret" -eq 2 ]]; then
                notice_bad "Failed parsing libc version"
                compat_libs=false
            fi
        fi
    fi

    if ! $compat_libs; then
        notice_good "Using system libraries"
        config_set "$config_pmd" "compat_libc" "0"
        rm -rf "$deploy_dir/glibc"
    else
        notice_good "Using packaged compatibility libraries"
        config_set "$config_pmd" "compat_libc" "1"
        if ! fix_packaged_binaries; then
            return 1
        fi
    fi

    return 0
}

_print_version() {
    echo "Version:"
    cat "$deploy_dir/version.txt" | sed -e 's/^/    /'
}

_uninstall() {
    if [[ -f "$uninstall_script" ]]; then
        "$uninstall_script" -n > /dev/null 2>&1
    fi
}

_abort() {
    error "Aborting."
    _uninstall
    exit 1
}

_cleanup() {
    if [[ -d "$g_config_backup_dir" ]]; then
        rm -rf "$g_config_backup_dir"
    fi
}

main() {

    if [[ "$EUID" -ne 0 ]]; then
        echo "Please run Cortex XDR installer as root"
        return 1
    fi

    setup_installer_log "Install"

    declare -r agent_full_version="$(cat ".version" | sed -e 's/-dbg$//')"
    if [[ "$agent_full_version" =~ ^(.*)-([0-9]+(\.[0-9]+){3})$ ]]; then

        declare -r agent_name="${BASH_REMATCH[1]}"
        declare -r agent_version="${BASH_REMATCH[2]}"
    else
        echo "Invalid agent version: $agent_full_version"
        return 1
    fi

    ce_suffix=$(cat ".ce_suffix")

    declare -r deploy_dir="/opt/traps"
    declare -r config_dir="$deploy_dir/config"

    declare -r uninstall_script="$deploy_dir/scripts/uninstall.sh"
    declare -r km_manage="$deploy_dir/km_utils/km_manage"

    declare -r config_common="$config_dir/common.xml"
    declare -r config_trapsd="$config_dir/trapsd.xml"
    declare -r config_pmd="$config_dir/pmd.xml"
    declare -r config_authorized="$config_dir/authorized.xml"

    declare -r libmodule_32="$deploy_dir/lib32/libmodule32.so"
    declare -r libmoduleng_32="$deploy_dir/lib32/libmoduleng32.so"
    declare -r libmodule_64="$deploy_dir/lib/libmodule64.so"
    declare -r libmoduleng_64="$deploy_dir/lib/libmoduleng64.so"

    declare -r init_d_dir="/etc/init.d"
    declare -r temp_dir="${TMPDIR:-/tmp}"

    trap _cleanup EXIT

    declare -r IFS_orig="$IFS"
    IFS=$'\n'
    read -d '' -r -a file_arguments <<< "$(load_arguments_file)" || true
    IFS="$IFS_orig"

    declare -r -a arguments=("${file_arguments[@]:-}" "$@")
    # Actual argument variables are defined in `setup.d/arguments.sh`.
    if ! parse_installer_arguments "${arguments[@]:-}"; then
        notice_bad "Failed to parse installer arguments"
        return $?
    fi

    # Detect "current" installation if an agent is preset.
    declare previous_installation=""
    if [[ -f "$config_trapsd" ]]; then
        previous_installation="$(config_get "$config_trapsd" "package_type")"
    fi

    if [[ -f "$config_common" ]]; then
        if [[ "$(config_get "$config_common" "containerized")" == "true" ]]; then
            in_container=true
        fi
    fi

    if [[ "$previous_installation" == "shell" ]]; then
        if $upgrade_esm2tms; then

            :
        elif $is_rpm; then
            echo "Detected a switch from a shell-installed agent to rpm"
            upgrade_traps=true
        elif $is_deb; then
            echo "Detected a switch from a shell-installed agent to deb"
            upgrade_traps=true
        fi
    fi

    if $in_container; then
        if [[ -n "${XDR_HOST_ROOT:-}" ]]; then
            echo "Resetting host fs path environment variable ($XDR_HOST_ROOT)"
            unset XDR_HOST_ROOT
        fi
    fi

    # Non-shell packages are validated during their own pre-installation phase.
    if ! $is_rpm && ! $is_deb; then
        # Don't allow upgrades from non-shell to shell packages.
        if [[ -n "$previous_installation" ]] && [[ "$previous_installation" != "shell" ]]; then
            error "Cannot upgrade from a $previous_installation package with a manual installer"
            error "Aborting."
            return 1
        fi

        # Run pre-installation for shell packages.
        source "./setup.d/pre-install.sh"
        if ! pre_install "$verify_required_pkgs"; then
            error "Aborting."
            return 1
        fi
    fi

    if $upgrade_esm2tms; then

        if $is_rpm; then
            mv "$deploy_dir/rpm-installer" "$temp_dir/"
        elif $is_deb; then
            mv "$deploy_dir/deb-installer" "$temp_dir/"
        fi

        if [[ -f "$uninstall_script" ]]; then
            step_start "Removing current agent installation"
            _uninstall
        else
            step_start "Removing Cortex XDR directories"
            rm -rf "$deploy_dir/"
            rm -rf "/var/log/traps/"
            rm -rf "/var/run/traps/"
        fi

        mkdir -p "$deploy_dir"
        if $is_rpm; then
            mv "$temp_dir/rpm-installer" "$deploy_dir/"
        elif $is_deb; then
            mv "$temp_dir/deb-installer" "$deploy_dir/"
        fi

        step_end
    elif _has_agent_installed; then

        notice_bad "Installation failed, previous installation present."
        notice_bad "Installation override is deprecated."
        echo "Either upgrade with --upgrade, or uninstall, then reinstall."
        return 1
    fi

    declare install_info="install"
    if $upgrade_traps; then
        if [[ -L "$deploy_dir" ]]; then
            step_start "Upgrading Cortex XDR [$agent_version$ce_suffix] at custom directory: $(readlink "$deploy_dir")"
        else
            step_start "Upgrading Cortex XDR [$agent_version$ce_suffix] at $deploy_dir"
        fi

        # Set install info to upgrade.
        install_info="upgrade"
        _do_upgrade
    else
        if [[ -L "$deploy_dir" ]]; then
            step_start "Installing Cortex XDR [$agent_version$ce_suffix] at custom directory: $(readlink "$deploy_dir")"
        else
            step_start "Installing Cortex XDR [$agent_version$ce_suffix] at $deploy_dir"
        fi
    fi

    # Unpack tarball.
    if ! _unpack_installation; then
        _abort
    fi

    # If we have compat glibc package avilable, check if we need to use it
    if [[ -d "$deploy_dir/glibc" ]]; then
        _do_compat_libs
    fi

    _print_version >> "$install_log"
    step_end

    # Create the logging directories.
    step_start "Creating runtime directory"
    if ! mkdir -p "/var/log/traps/"; then
        echo "Failed to create logs directory"
    fi
    if ! mkdir -p "/var/run/traps/"; then
        echo "Failed to create runtime directory"
    fi
    step_end

    if $upgrade_traps; then
        _persist_configs
        cp "$g_config_backup_dir/version.txt" "$deploy_dir/config/prev_version.txt" || true
    fi

    # Create the install.info file so trapsd knows if there was an install or upgrade
    # on its next start-up.
    echo "$install_info" > "$config_dir/install.info"

    # Resolve libmodules' requirements for chroot(2) injection.
    if [[ -e "$libmodule_64" ]]; then
        if ! _resolve_lib_reqs "$libmodule_64"; then
            error "Failed resolving module requirements."
            _abort
        fi
    fi

    # We don't require 32-bit version since not all systems support 32-bit binaries,
    # in which case injection wouldn't occur anyhow.
    if [[ -e "$libmodule_32" ]]; then
        _resolve_lib_reqs "$libmodule_32" || true
    fi

    # Setup LSMs.
    setup_apparmor || true

    if "$has_selinux"; then
        if ! setup_selinux; then
            _abort
        else
            config_set "$config_common" "selinux_disabled" "0"
        fi
    else
            config_set "$config_common" "selinux_disabled" "1"
    fi

    if has_fapolicyd_installed; then
        step_start "Installing fapolicyd rules"
        setup_fapolicyd || true
        step_end
    fi

    step_start "Verifying iptables prerequisite"
    _check_for_iptables_installation || true
    step_end

    # Setup service files.
    daemons_setup || true

    if ! $km_enabled; then
        echo "  Kernel module disabled by opt-out flag"
        config_set "$config_pmd" "km_enabled" "0"
    fi

    if ! setup_unprivileged_daemons; then
        _abort
    fi

    if $upgrade_esm2tms; then
        step_start "Upgrading from ESM - Configuring distribution ID"
        config_fix "$config_trapsd" "distribution_id" "$distribution_id"
        step_end
    else
        step_start "Configuring connection to server"
        if ! $upgrade_traps && [[ -n "$distribution_id" ]]; then
            config_fix "$config_trapsd" "distribution_id" "$distribution_id"
        fi
        if ! $upgrade_traps && [[ -n "$distribution_server" ]]; then
            config_fix "$config_trapsd" "cloud_elb_address" "$distribution_server"
        fi

        if ! $upgrade_traps; then
            configure_temporary_workload
        elif $is_temporary_session; then
            echo "Ignoring temporary session flag during upgrade"
        elif $is_vm_template; then
            echo "Ignoring VM template flag during upgrade"
        fi

        step_end
    fi

    # Configure endpoint tags.
    if ! $upgrade_traps && [[ -n "$endpoint_tags" ]]; then
        declare -r tags_stripped="$(strip_quotes "$endpoint_tags")"
        config_fix "$config_trapsd" "endpoint_tags" "$tags_stripped"
    fi

    # Configure proxy list.
    if ! configure_proxy_list "$proxy_list"; then
        _abort
    fi

    # Set restrictions.
    if $restrict_all; then
        config_fix "$config_trapsd" "restrict_invasive_response_actions" "1"
    fi
    if $restrict_live_terminal; then
        config_fix "$config_trapsd" "restrict_live_terminal" "1"
    fi
    if $restrict_script_execution; then
        config_fix "$config_trapsd" "restrict_script_execution" "1"
    fi
    if $restrict_file_retrieval; then
        config_fix "$config_trapsd" "restrict_file_retrieval" "1"
    fi

    # Set the package name.

    echo "$agent_name" > "$deploy_dir/package_name"

    # Set the package type (add to XML if needed).
    if $is_rpm; then
        config_fix "$config_trapsd" "package_type" "rpm"
    elif $is_deb; then
        config_fix "$config_trapsd" "package_type" "deb"
    else
        config_fix "$config_trapsd" "package_type" "shell"
    fi

    if $in_container; then

        config_fix_dynamic_feature "$config_trapsd" "containerized" "1"
        config_fix "$config_common" "containerized" "true"
    fi

    declare -r conf_path="/etc/panw/cortex.conf"
    if [[ -f "$conf_path" ]]; then
        print_file "$conf_path" >> "$install_log"
    fi

    if [[ -n "$override_trapsd_config_path" ]]; then

        if [[ ! -f "$override_trapsd_config_path" ]]; then
            echo "Override file: $override_trapsd_config_path doesn't exist"
            _abort
        fi

        print_file "$override_trapsd_config_path" >> "$install_log"
        if ! $(config_override "$override_trapsd_config_path" "$config_trapsd"); then
            echo "Override file: $override_trapsd_config_path is empty or doesn't contain valid elements"
	        _abort
        fi
    fi

    if $start_services; then
        step_start "Starting Cortex XDR security services"
        "$deploy_dir/bin/cytool" runtime start all
        step_end
    fi
}

[[ "$0" == "$BASH_SOURCE" ]] && main "$@"
